﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <vector>
#include <cstdlib>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/gfx/util/gfx_DebugFontTextWriter.h>
#include <nn/init/init_Malloc.h>
#include <nn/mem.h>
#include <nn/oe.h>
#include <nn/os.h>
#include <nn/vi.h>
#include <nn/util/util_ScopeExit.h>

#if defined(NN_BUILD_TARGET_PLATFORM_NX)
#include <nv/nv_MemoryManagement.h>
#endif

#if defined(USE_DEBUG_BTM_LOG)
#include <nn/btm/btm.h>
#include <nn/os/os_SystemEvent.h>
#endif

#include "ControllerConnectionAnalyzer_Color.h"
#include "ControllerConnectionAnalyzer_Definitions.h"
#include "ControllerConnectionAnalyzer_GraphicSystem.h"
#include "ControllerConnectionAnalyzer_NpadProperty.h"

namespace {

const size_t ApplicationHeapSize = 128 * 1024 * 1024;

#if defined(NN_BUILD_TARGET_PLATFORM_NX)
const size_t GraphicsMemorySize = 8 * 1024 * 1024;
#endif

const float Left = 25.0f;
const float Top =  15.0f;

const int FrameRate = 60;
}

//------------------------------------------------------------------------------
//  コントローラーの状態の出力
//------------------------------------------------------------------------------
void InitializeController()
{
    nn::hid::InitializeNpad();

    //使用する操作形態を設定
    nn::hid::SetSupportedNpadStyleSet(nn::hid::NpadStyleFullKey::Mask | nn::hid::NpadStyleJoyDual::Mask);

    // 使用するNpadを設定
    nn::hid::SetSupportedNpadIdType(NpadIds, NpadIdCountMax);
}

void Draw(nn::gfx::util::DebugFontTextWriter* pTextWriter, float regionOriginX, float regionOriginY)
{
    NN_ASSERT_NOT_NULL(pTextWriter);

    pTextWriter->SetTextColor(Colors::White);
    pTextWriter->SetScale(1.4f, 1.4f);

    pTextWriter->SetCursor(regionOriginX, regionOriginY);
    pTextWriter->Print("| Device style");

    pTextWriter->SetCursorX(regionOriginX + LedOriginLeft);
    pTextWriter->Print("| LED");

    pTextWriter->SetCursorX(regionOriginX + TimeSlotOriginLeft);
    pTextWriter->Print("| Num of Timeslot");

    pTextWriter->SetCursorX(regionOriginX + WlanCoexOriginLeft);
    pTextWriter->Print("| Coex");

    pTextWriter->SetCursorX(regionOriginX + PlrOriginLeft);
    pTextWriter->Print("| Packet loss");

    pTextWriter->SetCursor(regionOriginX, 675.0f);
    pTextWriter->Print("<R> Timeslot    <ZR> Coex    <A> Submit");
}

#if defined(USE_DEBUG_BTM_LOG)
void PrintDeviceCondition(nn::btm::DeviceConditionList &list)
{
    static const char SniffModeStr[4][16] = {"5ms", "10ms", "15ms", "ACTIVE"};
    static const char SlotSizeStr[6][16] = {"2", "4", "6", "ACTIVE"};
    static const char WlanModeStr[3][16] = {"Local4", "Local8", "NONE"};
    static const char BluetoothModeStr[2][16] = {"Dynamic2Slot", "StaticJoy"};

    NN_LOG("-- Bluetooth Device Condition --\n");
    NN_LOG("    WlanMode:%s\n", WlanModeStr[list.wlanMode]);
    NN_LOG("    BluetoothMode:%s\n", BluetoothModeStr[list.bluetoothMode]);
    NN_LOG("-- -------------------------- --\n");

    for (int i = 0; i < list.deviceCount; ++i)
    {
        NN_LOG("Dev:%d\n", i);
        NN_LOG("    Name:%s\n", list.device[i].bdName.name);

        if (list.device[i].profile == nn::btm::Profile_Hid)
        {
            NN_LOG("    SlotMode:%s\n", SlotSizeStr[list.device[i].hidDeviceCondition.slotMode]);
            NN_LOG("    SniffMode:%s, BurstMode:%s, Zero retx:%s\n",
                    SniffModeStr[list.device[i].hidDeviceCondition.sniffMode],
                    list.device[i].hidDeviceCondition.isBurstMode ? "ON" : "OFF",
                    list.device[i].hidDeviceCondition.zeroRetransmissionList.enabledReportIdCount > 0 ? "ON" : "OFF");
        }
    }
    NN_LOG("-- -------------------------- --\n");
}
#endif

extern "C" void nnMain()
{
#if defined(NN_BUILD_TARGET_PLATFORM_NX)
    nv::SetGraphicsAllocator(NvAllocate, NvFree, NvReallocate, NULL);
    nv::InitializeGraphics(std::malloc(GraphicsMemorySize), GraphicsMemorySize);
#endif

    ApplicationHeap applicationHeap;
    applicationHeap.Initialize(ApplicationHeapSize);

    GraphicsSystem* pGraphicsSystem = new ::GraphicsSystem();
    pGraphicsSystem->SetApplicationHeap(&applicationHeap);
    pGraphicsSystem->Initialize();

    std::vector<NpadProperty*> NpadProperties; //!< 管理対象の操作形態を格納するコンテナーです。
    NN_LOG("Controller Connection Analyzer Start.\n");

#if defined(USE_DEBUG_BTM_LOG)
    nn::btm::InitializeBtmInterface();
    nn::os::SystemEventType btmEvent;
    nn::btm::RegisterSystemEventForConnectedDeviceCondition(&btmEvent);
#endif

    InitializeController();

    for (int i = 0; i < NpadIdCountMax; ++i)
    {
        void* area = applicationHeap.Allocate(sizeof(NpadProperty), 16);
        NpadProperties.push_back(new (area) NpadProperty(NpadIds[i]));
    }

    for (std::vector<NpadProperty*>::iterator it = NpadProperties.begin(); it != NpadProperties.end(); ++it)
    {
        (*it)->Initialize();
    }

    // [TORIAEZU] 失敗したらアボート
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ldn::Initialize());

    // 終了通知を受け取る
    nn::oe::EnterExitRequestHandlingSection();

    nn::gfx::util::DebugFontTextWriter* pTextWriter = &pGraphicsSystem->GetDebugFont();

    bool isFocused = true;  // 起動時は Focus 状態であることが保証されている
    bool runs = true;
    while (runs)
    {
        // BG 中に描画するとループが止まるので、描画は FG 中のみ
        if (isFocused)
        {
#if defined(USE_DEBUG_BTM_LOG)
            // BT デバイスの接続、切断、DeviceCondition 変更に関する通知
            if (nn::os::TryWaitSystemEvent(&btmEvent))
            {
                nn::btm::DeviceConditionList list;
                nn::btm::GetConnectedDeviceCondition(&list);
                PrintDeviceCondition(list);
            }
#endif
            Draw(pTextWriter, Left, Top);

            float offset = Top;
            for (std::vector<NpadProperty*>::iterator it = NpadProperties.begin(); it != NpadProperties.end(); ++it)
            {
                if (!(*it)->IsConnected())
                {
                    continue;
                }

                (*it)->Update();
                offset = (*it)->Draw(pTextWriter, Left, offset);
            }

            pGraphicsSystem->BeginDraw();

            pTextWriter->Draw(&pGraphicsSystem->GetCommandBuffer());

            pGraphicsSystem->EndDraw();

            pGraphicsSystem->Synchronize(nn::TimeSpan::FromNanoSeconds(1000 * 1000 * 1000 / FrameRate));
        }
        else
        {
            nn::os::SleepThread(nn::TimeSpan::FromNanoSeconds(1000 * 1000 * 1000 / FrameRate));
        }

        // 通知メッセージの処理
        nn::oe::Message message;
        if (nn::oe::TryPopNotificationMessage(&message))
        {
            switch (message)
            {
            case nn::oe::MessageFocusStateChanged:
                {
                    auto state = nn::oe::GetCurrentFocusState();
                    isFocused = (state == nn::oe::FocusState_InFocus);
                }
                break;
            case nn::oe::MessageExitRequest:
                // メインループを抜ける
                runs = false;
                break;
            default:
                // 何もしない
                break;
            }
        }
    }

    nn::ldn::Finalize();

#if defined(USE_DEBUG_BTM_LOG)
    nn::btm::FinalizeBtmInterface();

    nn::os::DestroySystemEvent(&btmEvent);
#endif

    NN_LOG("Finished\n");

    pGraphicsSystem->Finalize();
    delete pGraphicsSystem;

    applicationHeap.Finalize();

    nn::oe::LeaveExitRequestHandlingSection();

}//NOLINT(readability/fn_size)
