﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "BcatTestApp_Common.h"
#include <glv_resources.h>
#include <glv/detail/FsUtilities.h>

#include <nn/util/util_Arithmetic.h>

#include <nv/nv_MemoryManagement.h>
#include <nvn/nvn.h>
#include <nvn/nvn_FuncPtrInline.h>
#include <nvnTool/nvnTool_GlslcInterface.h>
#include <nn/nifm.h>
#include <nn/time/time_Api.h>
#include <nn/account/account_ApiForApplications.h>
#include <nn/oe.h>

#include "Base/BcatTestApp_Main.h"
#include "Base/BcatTestApp_Context.h"

namespace {

    //!--------------------------------------------------------------------------------------
    //! プロセス確保総ヒープメモリ量
    //!--------------------------------------------------------------------------------------
    static const size_t AppletHeapSize = 256 * 1024 * 1024;                 //!< アプリケーション予約ヒープメモリサイズ
    static const size_t AppletAllocatableSize = 256 * 1024 * 1024;          //!< アプリケーション稼働ヒープ上限メモリサイズ
    static const size_t GraphicsSystemReservedMemorySize = 8 * 1024 * 1024; //!< NVNグラフィクス稼働予約メモリ領域サイズ

    //!--------------------------------------------------------------------------------------
    //! @brief 共通アロケータ
    //!--------------------------------------------------------------------------------------
    static void* Allocate( size_t size, size_t alignment, void* userPtr ) NN_NOEXCEPT
    {
        NN_UNUSED( userPtr );
        return ::aligned_alloc( alignment, size );
    }

    //!--------------------------------------------------------------------------------------
    //! @brief 共通デアロケータ
    //!--------------------------------------------------------------------------------------
    static void Deallocate( void* addr, void* userPtr ) NN_NOEXCEPT
    {
        NN_UNUSED( userPtr );
        ::free( addr );
    }

    //!--------------------------------------------------------------------------------------
    //! @brief 共通リアロケータ
    //!--------------------------------------------------------------------------------------
    static void* Reallocate( void* addr, size_t newSize, void* userPtr ) NN_NOEXCEPT
    {
        NN_UNUSED( userPtr );
        return ::realloc( addr, newSize );
    }

    //!--------------------------------------------------------------------------------------
    //! @brief GLV用ユーザ定義アロケータ.
    //!--------------------------------------------------------------------------------------
    void* glvMemoryAllocator( const size_t size, const size_t beginAlignment ) NN_NOEXCEPT
    {
        return ::aligned_alloc( beginAlignment, size );
    }

    //!--------------------------------------------------------------------------------------
    //! @brief GLV用ユーザ定義デアロケータ.
    //!--------------------------------------------------------------------------------------
    void glvMemoryDeallocator( void* address ) NN_NOEXCEPT
    {
        ::free( address );
    }

    //!--------------------------------------------------------------------------------------
    //! @brief ペリフェラルセットアップ
    //!--------------------------------------------------------------------------------------
    static void SetupPeripherals() NN_NOEXCEPT
    {
        // this memory allocation will be used from the nvn graphics systems at runtime.
        nv::SetGraphicsAllocator( Allocate, Deallocate, Reallocate, nullptr );
        nv::InitializeGraphics( ::malloc( GraphicsSystemReservedMemorySize ), GraphicsSystemReservedMemorySize );

        // this memory allocation interface will be used when compiling of shader code at runtime.
        glslcSetAllocator( Allocate, Deallocate, Reallocate, nullptr );
    }

    //!--------------------------------------------------------------------------------------
    //! @brief HID設定初期値
    //!--------------------------------------------------------------------------------------
    static const glv::HidInitialConfiguration LocalHidConfiguration = glv::HidInitialConfiguration( glv::HidInitialConfiguration::PadAssetAssignRule_BasicPadPrimary );
} //namespace

//!--------------------------------------------------------------------------------------
//! nninitStartup() is invoked before calling nnMain().
//! 256MB確保
//!--------------------------------------------------------------------------------------
NN_OS_EXTERN_C void nninitStartup() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::SetMemoryHeapSize( AppletHeapSize ) );
    uintptr_t address;
    const size_t MallocMemorySize = AppletAllocatableSize;
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::AllocateMemoryBlock( &address, MallocMemorySize ) );
    nn::init::InitializeAllocator( reinterpret_cast< void* >( address ), MallocMemorySize );

    // GLV内部で利用されるルートメモリアロケータを上書きします.
    // 上書きしない場合は C++11 の std::aligned_alloc 及び std::free が利用されます.
    glv::detail::ApplicationMemoryAllocator::AttachUserAllocator( glvMemoryAllocator, glvMemoryDeallocator );
}

//!--------------------------------------------------------------------------------------
//! @brief プロセスエントリポイント
//!--------------------------------------------------------------------------------------
NN_OS_EXTERN_C void nnMain() NN_NOEXCEPT
{
    // ペリフェラルを初期化
    SetupPeripherals();

    // nifm, time 初期化
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::nifm::Initialize() );
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::Initialize() );
    nn::oe::Initialize();
    nn::account::Initialize();
    nn::bcat::Initialize();

    // GLVランタイムエンジンの初期化及びアプリケーションフレームワークの初期化
    glv::ApplicationFrameworkInitialize( LocalHidConfiguration );

    // スクリーン大きさ
    const int width = glv::glutGet( GLUT_SCREEN_WIDTH );
    const int height = glv::glutGet( GLUT_SCREEN_HEIGHT );
    GLV_LOG( "screen( %zd x %zd )\n", width, height );

    // アプリメイン
    glv::Window* window = new glv::Window( width, height, "Example Window" );
    NN_ASSERT( window != nullptr );
    app::Main( *window );
    if ( nullptr != window )
    {
        delete window;
    }
}

