﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nn/bcat/bcat_DeliveryCacheFile.h>
#include <nn/crypto/crypto_Md5Generator.h>

#include "BcatTestApp_Common.h"
#include "BcatTestApp_StorageMenu.h"
#include "BcatTestApp_FileMenu.h"
#include "BcatTestApp_FileSystem.h"
#include "BcatTestApp_BinaryDump.h"

#include <vector>

namespace app
{
void ExecFileMenu_Initialize( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT;
void ExecFileMenu           ( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT;
void ExecFileMenu_Finalize  ( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT;
void DrawFileMenu( void* arg ) NN_NOEXCEPT;

ExecCallbackGroup ExecFileMenuGroup = {
    ExecFileMenu_Initialize,
    ExecFileMenu,
    ExecFileMenu_Finalize,
    nullptr,

    DrawFileMenu,
    nullptr,
    DrawPriority_FileMenu,
    0
};

void ExecFileActionMenu_Initialize( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT;
void ExecFileActionMenu           ( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT;
void ExecFileActionMenu_Finalize  ( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT;
void DrawFileActionMenu( void* arg ) NN_NOEXCEPT;

ExecCallbackGroup ExecFileActionMenuGroup = {
    ExecFileActionMenu_Initialize,
    ExecFileActionMenu,
    ExecFileActionMenu_Finalize,
    nullptr,

    DrawFileActionMenu,
    nullptr,
    DrawPriority_FileActionMenu,
    0
};

namespace
{
    //---------------- ファイルメニュー
    enum {
        Sequence_Init = 0,
        Sequence_FromErrorDialog
    };

    // コンソール情報
    const int FixConsoleWidth = 78;
    const int FixConsoleHeight = 33;
    const int FixConsoleBufferSize = APP_CONSOLE_BUFFER_SIZE( FixConsoleWidth, FixConsoleHeight );
    const int PropConsoleWidth = 100;
    const int PropConsoleHeight = 33;
    const int PropConsoleBufferSize = APP_CONSOLE_BUFFER_SIZE( PropConsoleWidth, PropConsoleHeight );
    char* g_FixConsoleBuffer;
    char* g_PropConsoleBuffer;
    bool g_IsConsoleCreated = false;
    app::FixedConsole<char16_t> g_FixConsole;
    app::FixedProportionalConsole<char16_t> g_PropConsole;

    const float FontSize = 24.0F;
    const float GridSizeX = 16.0F;
    const float GridSizeY = 28.0F;

    // エントリ情報
    nn::bcat::DeliveryCacheDirectoryEntry* g_pFileEntryArray;
    std::vector<nn::bcat::DeliveryCacheDirectoryEntry*> g_FileNameList( nn::bcat::DeliveryCacheFileCountMaxPerDirectory );
    struct FileEntryInfo
    {
    public:
        int entryIndex;
        bool isOpened;
        nn::bcat::DeliveryCacheFile *pFile;

        FileEntryInfo() NN_NOEXCEPT :
            entryIndex(-1),
            isOpened(false),
            pFile(nullptr)
        {
        }
    };
    std::vector<FileEntryInfo*> g_FileEntryInfoList;

    const int FileListViewMax = 12;
    int g_FileListCursor = 0;
    int g_FileListViewTop = 0;
    int g_FileListNum = 0;
    bool g_IsFileListUpdate = false;
    //int g_CurFile = 0;

    char g_FilePath[66];
    //int64_t g_FileSize;
    //char g_FileDigestStr[34];

    // 表示用
    const int ConsoleOffsetX = 10;
    const int ConsoleOffsetY = 10;
    nn::bcat::DirectoryName* g_pDirectoryName;

    //---------------- ファイルアクションメニュー
    app::Menu g_FileActionMenu;

    enum
    {
        MenuIndex_OpenFile = 0,
        MenuIndex_ReadFile,
        MenuIndex_FileSize,
        MenuIndex_CalculateDigest,
        MenuIndex_CloseFile,
        MenuIndex_SaveFile,
        MenuIndex_SaveToSd,
    };
    int g_InitialMenuIndex = MenuIndex_OpenFile;
    const int MenuMax = MenuIndex_SaveToSd + 1;

    //---------------- ファイルハッシュ
    union
    {
        char value[ nn::crypto::Md5Generator::HashSize ]; //16byte
        uint64_t value64[2];
    } g_Hash;

    //---------------- バイナリダンプ用
    app::BinaryDumpParam g_BinaryDumpParam;

} //namespace

namespace
{
//----------------------------------------------------------------
// 固定コンソールプリンタ
//
void DrawFixConsoleString( int x, int y, char16_t* string, char* attr, int stringLen, void* arg ) NN_NOEXCEPT
{
    DrawFixedConsoleGeneric16( Position_FileMenu.l + ConsoleOffsetX,
                               Position_FileMenu.t + ConsoleOffsetY,
                               FontSize, GridSizeX, GridSizeY,
                               x, y, string, attr, stringLen, arg );
}
//----------------------------------------------------------------
// プロポーショナルコンソールプリンタ
//
void DrawPropConsoleString( int x, int y, char16_t* string, char* attr, int stringLen, void* arg ) NN_NOEXCEPT
{
    DrawProportionalConsoleGeneric16( Position_FileMenu.l + ConsoleOffsetX,
                                      Position_FileMenu.t + ConsoleOffsetY,
                                      FontSize, GridSizeX, GridSizeY,
                                      x, y, string, attr, stringLen, arg );
}
}

//----------------------------------------------------------------
// ファイルメニュー用描画コールバック
//
void DrawFileMenu( void* arg ) NN_NOEXCEPT
{
    // フレーム
    app::DrawFrameRectangle( Position_FileMenu, DrawColorSet_FileMenuBack, DrawColorSet_FileMenuFrame, 3 );

    // カーソル
    if ( g_FileListNum > 0 )
    {
        int y = Position_FileMenu.t + (g_FileListCursor * GridSizeY) + (4 * GridSizeY) + ConsoleOffsetY;
        app::DrawRectangle( Position_FileMenu.l + GridSizeX * 3, y,
                            Position_FileMenu.r - GridSizeX * 2, y + GridSizeY,
                            app::DrawColorSet_ItemCursorBack );

        // スクロールバー
        app::DrawScrollBar( g_FileListNum, FileListViewMax, g_FileListViewTop, ScrollBarDefaultViewWidth, Position_FileMenuScrollBar );
    }

    // コンソール描画
    if ( g_IsConsoleCreated )
    {
        g_FixConsole.Display();
        g_PropConsole.Display();
    }
}

//----------------------------------------------------------------
void PrintFileCursor() NN_NOEXCEPT
{
    // ファイルがあるならカーソル表示
    if ( g_FileListNum > 0 )
    {
        g_FixConsole.PrintfEx( 1, g_FileListCursor + 4, app::ConsoleColor_Yellow, u">" );
    }
}

//----------------------------------------------------------------
void PrintFileList() NN_NOEXCEPT
{
    g_FixConsole.Clear();
    g_PropConsole.Clear();

    // ファイル数
    g_PropConsole.PrintfEx( 0, 0, app::ConvertToChar16_t("File List :  %d dir(s)", g_FileListNum) );

    // 現在のディレクトリ名
    g_PropConsole.PrintfEx( 16, 0, app::ConvertToChar16_t("Directory Name : %s", g_pDirectoryName->value ) );

    if ( g_FileListNum == 0 )
    {
        g_PropConsole.PrintfEx( 2, 3, app::ConsoleColor_Yellow, u"<ファイルがありません>" );
        return;
    }

    g_PropConsole.PrintfEx( 2, 2, u"No." );
    g_PropConsole.PrintfEx( 11, 2, u"File name" );
    g_PropConsole.PrintfEx( 41, 2, u"Size" );
    g_PropConsole.PrintfEx( 51, 2, u"Digest" );
    g_FixConsole.PrintfEx( 2, 3, u"---  ------------------------------- --------  -----------" );

    // ファイル名
    for( int i=0; i<FileListViewMax; i++ )
    {
        int num = g_FileListViewTop + i;
        if ( num >= g_FileListNum )
        {
            break;
        }

        int posY = i + 4;

        FileEntryInfo* pEntryInfo = g_FileEntryInfoList[ g_FileListViewTop + i ];
        nn::bcat::DeliveryCacheDirectoryEntry* pEntry = g_pFileEntryArray + pEntryInfo->entryIndex;

        // 名前・サイズ
        g_FixConsole.PrintfEx( 2, posY,
                               ( pEntryInfo->isOpened )? app::ConsoleColor_Yellow: app::ConsoleColor_White,
                                   app::ConvertToChar16_t("%3d: %s", num, pEntry->name.value) );
        g_FixConsole.PrintfEx( 39, posY, app::ConvertToChar16_t("%8lld", pEntry->size) );

        // ダイジェスト
        char diStr[17];
        nn::util::SNPrintf( diStr, sizeof(diStr), "%016llX", reinterpret_cast<uint64_t>(pEntry->digest.value[0]) );
        diStr[8] = '\0'; // 8文字しか使用しないので
        g_FixConsole.PrintfEx( 49, posY, app::ConvertToChar16_t("%s...", diStr) );

        // オープンか
        if ( pEntryInfo->isOpened )
        {
            g_FixConsole.PrintfEx( 6, posY, app::ConsoleColor_Yellow, u"*" );
        }

        //g_FixConsole.PrintfEx( 2, i + 4, app::ConvertToChar16_t( "%3d: %s", num, g_FileNameList[num]->name ) );
    }
}

//----------------------------------------------------------------
// ファイルメニューのキー説明
void PrintFileMenuHelp() NN_NOEXCEPT
{
    app::GetHelpConsole().Clear();
    app::GetHelpConsole().PrintfEx( 2, 0,
                                    u"@1操作説明:@7 @6[上][下]@7...カーソル  @4[A]@7...決定  @5[X]@7...ダンプ  @2[B]@7...戻る" );
}

//----------------------------------------------------------------
// ファイルメニュー(開始処理)
//
void ExecFileMenu_Initialize( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    // パラメータ受け取り
    app::FileMenuParameter* pParameter = reinterpret_cast<app::FileMenuParameter*>( arg );
    if ( pParameter )
    {
        g_pDirectoryName = pParameter->pDirectoryName;
        g_FileListNum = pParameter->fileEntryNum;
        g_pFileEntryArray = pParameter->pFileEntryArray;
    }

    g_FileEntryInfoList.clear();
    for( int i=0; i<g_FileListNum; i++ )
    {
        FileEntryInfo* p = new FileEntryInfo;
        p->entryIndex = i;
        g_FileEntryInfoList.push_back( p );
    }

    //xxxx ソートとかしておく
    // xxxsort();

    g_FileListCursor = 0;
    g_FileListViewTop = 0;

    // 固定コンソール
    g_FixConsoleBuffer = new char[ FixConsoleBufferSize ];
    g_FixConsole.SetBuffer( g_FixConsoleBuffer, FixConsoleBufferSize, FixConsoleWidth, FixConsoleHeight );
    g_FixConsole.SetPrinter( DrawFixConsoleString, nullptr );
    // プロポーショナルコンソール
    g_PropConsoleBuffer = new char[ PropConsoleBufferSize ];
    g_PropConsole.SetBuffer( g_PropConsoleBuffer, PropConsoleBufferSize, PropConsoleWidth, PropConsoleHeight );
    g_PropConsole.SetPrinter( DrawPropConsoleString, nullptr );

    g_IsConsoleCreated = true;

    PrintFileList();
    PrintFileCursor();

    // キー説明
    PrintFileMenuHelp();
}
//----------------------------------------------------------------
// ファイルメニュー(終了処理)
//
void ExecFileMenu_Finalize( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    if ( g_IsConsoleCreated )
    {
        delete g_FixConsoleBuffer;
        delete g_PropConsoleBuffer;
        g_FixConsoleBuffer = nullptr;
        g_PropConsoleBuffer = nullptr;
    }
    g_IsConsoleCreated = false;

    for( int i=0; i<g_FileListNum; i++ )
    {
        FileEntryInfo* pEntryInfo = g_FileEntryInfoList[i];
        if ( pEntryInfo->isOpened )
        {
            pEntryInfo->pFile->Close();
            app::GetSystemConsole().Printf("nn::bcat::DeliveryCacheFile::Close() : Success\n");

            delete pEntryInfo->pFile;
            pEntryInfo->pFile = nullptr;
        }
    }
    g_FileEntryInfoList.clear();
}

void FromBinaryDump( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    PrintFileMenuHelp();
}

void CallBinaryDump() NN_NOEXCEPT
{
    FileEntryInfo* pEntryInfo = g_FileEntryInfoList[ g_FileListViewTop + g_FileListCursor ];
    nn::bcat::DeliveryCacheDirectoryEntry* pEntry = g_pFileEntryArray + pEntryInfo->entryIndex;

    g_BinaryDumpParam.pDirectoryName = g_pDirectoryName->value;
    g_BinaryDumpParam.pFileName = pEntry->name.value;
    g_BinaryDumpParam.pFileEntry = pEntry;

    app::sequence::Call( ExecBinaryDumpGroup, reinterpret_cast<void*>(&g_BinaryDumpParam) );
    app::sequence::SetFromCall( FromBinaryDump, nullptr );
}

namespace
{
void FromCallFunc( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    app::g_IsFileListUpdate = true;
}
}

//----------------------------------------------------------------
// ファイルメニュー
//
void ExecFileMenu( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    if ( app::sequence::GetSequence() == Sequence_FromErrorDialog )
    {
        app::sequence::JumpTo( ExecStorageMenuGroup );
        return;
    }

    app::Pad pad( events );

    // A ボタンでファイルアクションメニューを開く
    if ( pad.IsButtonDownA() && g_FileListNum > 0 )
    {
        app::sequence::Call( ExecFileActionMenuGroup );
        app::sequence::SetFromCall( FromCallFunc, nullptr );
        return;
    }

    // B ボタンで戻る
    if ( pad.IsButtonDownB() )
    {
        app::sequence::Return();
    }

    // X ボタンでバイナリダンプ
    if ( pad.IsButtonDownX() && g_FileListNum > 0 )
    {
        CallBinaryDump();
        return;
    }

    // カーソル上
    if ( pad.IsButtonDownUp() || pad.IsButtonRepeatUp() )
    {
        RotateWithinViewRange( g_FileListCursor, -1, g_FileListViewTop, FileListViewMax, g_FileListNum );
        g_IsFileListUpdate = true;
    }
    // カーソル下
    if ( pad.IsButtonDownDown() || pad.IsButtonRepeatDown() )
    {
        RotateWithinViewRange( g_FileListCursor, 1, g_FileListViewTop, FileListViewMax, g_FileListNum );
        g_IsFileListUpdate = true;
    }
    // カーソル左
    if ( pad.IsButtonDownLeft() )
    {
        g_FileListCursor = 0;
        g_FileListViewTop = 0;
        g_IsFileListUpdate = true;
    }
    // カーソル右
    if ( pad.IsButtonDownRight() )
    {
        g_FileListCursor = (g_FileListNum < FileListViewMax)? (g_FileListNum - 1): (FileListViewMax - 1);
        g_FileListViewTop = (g_FileListNum < FileListViewMax)? 0: (g_FileListNum - FileListViewMax);
        g_IsFileListUpdate = true;
    }

    // リストの更新
    if ( g_IsFileListUpdate )
    {
        g_IsFileListUpdate = false;
        PrintFileList();
        PrintFileCursor();
    }
}

//================================================================
// ソート

//================================================================
// ファイルアクションメニュー
//----------------------------------------------------------------
// ファイルアクションメニュー用描画コールバック
void DrawFileActionMenu( void* arg ) NN_NOEXCEPT
{
}
//----------------------------------------------------------------
// ファイルアクションメニューのメニュー文字列描画
//
void DrawFileActionMenuItems() NN_NOEXCEPT
{
    FileEntryInfo* pEntryInfo = g_FileEntryInfoList[ g_FileListViewTop + g_FileListCursor ];
    nn::bcat::DeliveryCacheDirectoryEntry* pEntry = g_pFileEntryArray + pEntryInfo->entryIndex;

    app::FixedProportionalConsole<char16_t>* p = g_FileActionMenu.GetConsole16();
    if ( p )
    {
        app::SetGlvColor( app::ColorSet_White );
        p->PrintfEx( 1, 0,
                     u"-- ファイル操作メニュー --" );
        p->PrintfEx( 2, 2,
                     ( pEntryInfo->isOpened )? app::ConsoleColor_BrightRed: app::ConsoleColor_White,
                     u"ファイルオープン : nn::bcat::DeliveryCacheFile::Open()" );
        p->PrintfEx( 2, 3,
                     ( pEntryInfo->isOpened )? app::ConsoleColor_White: app::ConsoleColor_DarkWhite,
                     u"ファイルリード : nn::bcat::DeliveryCacheFile::Read()" );
        p->PrintfEx( 2, 4,
                     ( pEntryInfo->isOpened )? app::ConsoleColor_White: app::ConsoleColor_DarkWhite,
                     u"ファイルサイズ取得 : nn::bcat::DeliveryCacheFile::GetSize()" );
        p->PrintfEx( 2, 5,
                     ( pEntryInfo->isOpened )? app::ConsoleColor_White: app::ConsoleColor_DarkWhite,
                     u"ダイジェスト計算" );
        p->PrintfEx( 2, 6,
                     ( pEntryInfo->isOpened )? app::ConsoleColor_White: app::ConsoleColor_DarkWhite,
                     u"ファイルクローズ : nn::bcat::DeliveryCacheFile::Close()" );
        p->PrintfEx( 2, 7,
                     ( pEntryInfo->isOpened )? app::ConsoleColor_DarkWhite: app::ConsoleColor_White,
                     u"セーブデータへ保存" );
        p->PrintfEx( 2, 8,
                     app::ConsoleColor_DarkWhite,
                     u"SD カードへ保存(未実装)" );
    }

    // 選択したファイル名とオープン状態
    if ( g_FileListNum > 0 )
    {
        p->PrintfEx( 16, 0, app::ConvertToChar16_t("[ %s ]", pEntry->name.value ) );

        // オープン状態
        if ( pEntryInfo->isOpened )
        {
            p->PrintfEx( 32, 0, app::ConsoleColor_Yellow, u"オープン中" );
        }
        else
        {
            p->ClearPartial( 32, 0 );
        }
    }
}

//----------------------------------------------------------------
// ファイルアクションメニューのキー説明
void PrintFileActionMenuHelp() NN_NOEXCEPT
{
    app::GetHelpConsole().Clear();
    app::GetHelpConsole().PrintfEx( 2, 0,
                                    u"@1操作説明:@7 @6[上][下]@7...カーソル  @4[A]@7...決定  @2[B]@7...戻る" );
}

//----------------------------------------------------------------
// ファイルアクションメニュー(開始処理)
//
void ExecFileActionMenu_Initialize( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    // メニュー作成
    g_FileActionMenu.CreateConsole( app::ConsoleSize_Char16_t, app::Position_FileActionMenu,
                                         100, 6, 1, 2, 24, 4,
                                         app::DrawPriority_FileActionMenu + 1, 0x240 );
    g_FileActionMenu.SetBackColor( app::ColorSet_DarkGreen, app::ColorSet_Green, app::DrawFrameWidth );
    g_FileActionMenu.SetItemParameter( MenuMax, g_InitialMenuIndex );

    // メニュー項目のコンソールプリント
    DrawFileActionMenuItems();

    // キー説明
    PrintFileActionMenuHelp();
}
//----------------------------------------------------------------
// ファイルアクションメニュー(終了処理)
//
void ExecFileActionMenu_Finalize( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    g_FileActionMenu.DestroyConsole();
}

namespace
{
app::DialogParam* SetFileSizeDialog( int64_t size ) NN_NOEXCEPT
{
    app::DialogParam* p = app::AllocDialogParam( DialogParam::InfoType_Notice );

    p->GetConsole()->PrintfEx( 0, 0,
                               u"ファイルサイズを取得しました。" );
    p->GetConsole()->PrintfEx( 7, 1, app::ConvertToChar16_t( "file size : %lld (0x%llx) byte", size, size ) );

    FileEntryInfo* pEntryInfo = g_FileEntryInfoList[ g_FileListViewTop + g_FileListCursor ];
    nn::bcat::DeliveryCacheDirectoryEntry* pEntry = g_pFileEntryArray + pEntryInfo->entryIndex;
    if ( size == pEntry->size )
    {
        p->GetConsole()->PrintfEx( 0, 3, u"エントリ情報から取得できるサイズと同じです。" );
        app::GetSystemConsole().Printf( "Same with filesize in the directory entry.\n");
    }
    else
    {
        p->GetConsole()->PrintfEx( 0, 3, app::ConsoleColor_Red, u"エントリ情報から取得できるサイズと異なります。" );
        app::GetSystemConsole().Printf( "size = %lld (0x%llx) byte\n", size, size );
        app::GetSystemConsole().Printf( "Different from filesize in the directory entry.\n");
    }

    app::sequence::Call( app::ExecDialogGroup, p );
    return p;
}
}

//----------------------------------------------------------------
// 「ファイルオープン」項目
bool FileActionMenu_OpenFile() NN_NOEXCEPT
{
    FileEntryInfo* pEntryInfo = g_FileEntryInfoList[ g_FileListViewTop + g_FileListCursor ];
    nn::bcat::DeliveryCacheDirectoryEntry* pEntry = g_pFileEntryArray + pEntryInfo->entryIndex;
    nn::util::SNPrintf( g_FilePath, sizeof(g_FilePath), "%s/%s", g_pDirectoryName->value, pEntry->name.value );

    // ファイルクラスインスタンス
    if ( ! pEntryInfo->isOpened )
    {
        pEntryInfo->pFile = new nn::bcat::DeliveryCacheFile;
    }

    app::GetSystemConsole().Printf("nn::bcat::DeliveryFile::Open() :");
    nn::Result result = pEntryInfo->pFile->Open( g_FilePath );
    app::PrintErrorCode( result );
    if ( result.IsSuccess() )
    {
        pEntryInfo->isOpened = true;
    }
    else
    {
        app::SetErrorDialog( u"ファイルのオープンに失敗しました", result );
        return false;
    }

    return true;
}

// 「ファイルリード」項目
bool FileActionMenu_ReadFile() NN_NOEXCEPT
{
    FileEntryInfo* pEntryInfo = g_FileEntryInfoList[ g_FileListViewTop + g_FileListCursor ];

    if ( pEntryInfo->isOpened )
    {
        int64_t fileSize = pEntryInfo->pFile->GetSize();
        const int64_t HeadSize = 8L;

        static const int32_t BufferSizeForRead = 0x10000;
        char* bufferForRead = new char[ BufferSizeForRead ];
        char* bufferHead = new char[ HeadSize ]; // 表示する先頭
        NN_UTIL_SCOPE_EXIT
        {
            delete[] bufferForRead;
            delete[] bufferHead;
        };

        // 一通りリードする
        int64_t position = 0;
        nn::Result result;
        while( position < fileSize )
        {
            size_t outSize;
            result = pEntryInfo->pFile->Read( &outSize, position, bufferForRead, BufferSizeForRead );
            app::GetSystemConsole().Printf( "nn::bcat::DeliveryCacheFile::Read( offset=0x%llx, size=0x%llx ) outSize=0x%x :", position, BufferSizeForRead, outSize );
            app::PrintErrorCode( result );
            if ( result.IsFailure() )
            {
                break;
            }
            // 先頭のコピー
            if ( position == 0 )
            {
                std::memcpy( bufferHead, bufferForRead, std::min( HeadSize, fileSize ) );
            }
            position += outSize;
        }

        app::DialogParam* p;
        if( result.IsSuccess() )
        {
            p = app::AllocDialogParam( app::DialogParam::InfoType_Notice );
            app::FixedProportionalConsole<char16_t>* c = p->GetConsole();

            c->PrintfEx( 0, 0, u"ファイルリードに成功しました。" );

            for( int i=0; i<std::min(HeadSize, fileSize); i++ )
            {
                c->PrintfEx( i * 2, 2, app::ConvertToChar16_t( "%02X",  bufferHead[i] ) );
            }
            if ( fileSize > HeadSize )
            {
                c->PrintfEx( HeadSize * 2, 2, u"…" );
            }
        }
        else
        {
            p = app::AllocDialogParam( app::DialogParam::InfoType_Caution );
            app::FixedProportionalConsole<char16_t>* c = p->GetConsole();
            c->PrintfEx( 0, 0, u"ファイルリードに失敗しました。" );
            c->PrintfEx( 1, 0, app::ConvertToChar16_t( "offset=0x%llx size=0x%x", position, BufferSizeForRead ) );
            p->SetResult( result );
        }
        app::sequence::Call( app::ExecDialogGroup, p );
        return false;
    }

    return true;
}

// 「ファイルサイズ」項目
bool FileActionMenu_FileSize() NN_NOEXCEPT
{
    FileEntryInfo* pEntryInfo = g_FileEntryInfoList[ g_FileListViewTop + g_FileListCursor ];
    if ( pEntryInfo->isOpened )
    {
        int64_t size = pEntryInfo->pFile->GetSize();
        app::GetSystemConsole().Printf( "nn::bcat::DeliveryCacheFile::GetSize() : Success\n" );
        app::GetSystemConsole().Printf( "size = %lld (0x%llx) byte\n", size, size );
        app::DialogParam* p = SetFileSizeDialog( size );
        app::sequence::Call( app::ExecDialogGroup, p );
        return false;
    }

    return true;
}

//----------------------------------------------------------------
// MD5 の計算
//
nn::Result CalculateMd5( nn::bcat::DeliveryCacheFile *pFile, int64_t size ) NN_NOEXCEPT
{
    // ハッシュ計算用バッファ割り当て
    static const int32_t BufferSizeForHash = 0x10000;
    char* bufferForHash = new char[ BufferSizeForHash ];
    NN_UTIL_SCOPE_EXIT
    {
        // ハッシュ計算用バッファ破棄
        delete[] bufferForHash;
    };

    // ハッシュ計算
    nn::crypto::Md5Generator md5;
    md5.Initialize();

    int64_t position = 0;
    while( position < size )
    {
        size_t outSize;
        nn::Result result = pFile->Read( &outSize, position, bufferForHash, BufferSizeForHash );

        if ( result.IsFailure() )
        {
            app::GetSystemConsole().Printf( "nn::bcat::DeliveryCacheFile::Read( offset=0x%llx, size=0x%llx ) :", position, BufferSizeForHash );
            app::PrintErrorCode( result );
            return result;
        }
        else
        {
            app::GetSystemConsole().Printf( "nn::bcat::DeliveryCacheFile::Read( offset=0x%llx, size=0x%llx ) outSize=0x%x :", position, BufferSizeForHash, outSize );
            app::PrintErrorCode( result );
        }

        md5.Update( bufferForHash, outSize );
        position += outSize;
    }

    // ハッシュ取得
    md5.GetHash( g_Hash.value, nn::crypto::Md5Generator::HashSize );

#ifdef NN_BUILD_CONFIG_ENDIAN_LITTLE
    // エンディアンを考慮して反転
    nn::util::SwapEndian(&g_Hash.value64[0]);
    nn::util::SwapEndian(&g_Hash.value64[1]);
#endif

    NN_RESULT_SUCCESS;
}

// 「ダイジェスト計算」項目
bool FileActionMenu_CalculateDigest() NN_NOEXCEPT
{
    FileEntryInfo* pEntryInfo = g_FileEntryInfoList[ g_FileListViewTop + g_FileListCursor ];
    nn::bcat::DeliveryCacheDirectoryEntry* pEntry = g_pFileEntryArray + pEntryInfo->entryIndex;

    if ( pEntryInfo->isOpened )
    {
        app::GetSystemConsole().Printf( "Calculate the digest.\n" );
        nn::Result result = CalculateMd5( pEntryInfo->pFile, pEntryInfo->pFile->GetSize() );
        if ( result.IsSuccess() )
        {
            app::DialogParam* p = app::AllocDialogParam( app::DialogParam::InfoType_Notice );
            p->GetConsole()->PrintfEx( 0, 0, u"ダイジェスト値を計算しました。" );

            p->GetConsole()->PrintfEx( 0, 1, app::ConvertToChar16_t( "%016llX %016llX", g_Hash.value64[0], g_Hash.value64[1] ) );
            app::GetSystemConsole().Printf( "Digest: %016llX %016llX\n", g_Hash.value64[0], g_Hash.value64[1] );

            if ( g_Hash.value64[0] == pEntry->digest.value[0] && g_Hash.value64[1] == pEntry->digest.value[1] )
            {
                p->GetConsole()->PrintfEx( 0, 3, u"エントリ情報から取得できるダイジェスト値と同じです。" );
                app::GetSystemConsole().Printf( "Same with a digest in the directory entry.\n");
            }
            else
            {
                p->GetConsole()->PrintfEx( 0, 3, app::ConsoleColor_Red, u"エントリ情報から取得できるダイジェスト値と異なります。" );
                app::GetSystemConsole().Printf( "Different from a digest in the directory entry.\n");
            }
            app::sequence::Call( app::ExecDialogGroup, p );
        }
        else
        {
            app::GetSystemConsole().Printf( "Failed.\n" );
            app::SetErrorDialog( u"ダイジェスト値の計算に失敗しました",
                                 result );
        }
        return false;
    }

    return true;
}

// 「ファイルクローズ」項目
bool FileActionMenu_CloseFile() NN_NOEXCEPT
{
    FileEntryInfo* pEntryInfo = g_FileEntryInfoList[ g_FileListViewTop + g_FileListCursor ];
    if ( pEntryInfo->isOpened )
    {
        pEntryInfo->pFile->Close(); // 必ず成功
        app::GetSystemConsole().Printf("nn::bcat::DeliveryCacheFile::Close() : Success\n");

        delete pEntryInfo->pFile;
        pEntryInfo->pFile = nullptr;
        pEntryInfo->isOpened  = false;
    }

    return true;
}

// 「セーブファイル」項目
bool FileActionMenu_SaveFile() NN_NOEXCEPT
{
    FileEntryInfo* pEntryInfo = g_FileEntryInfoList[ g_FileListViewTop + g_FileListCursor ];
    if ( pEntryInfo->isOpened )
    {
        return true;
    }

    // ユーザ選択
    if ( ! IsUserValid() )
    {
        SelectUser();
        if ( ! IsUserValid() )
        {
            app::SetInvalidUserDialog( false );
            return false;
        }
    }

    bool isDirectoryCreated;
    app::SyncFileResult sfResult = app::SyncFileToSaveData( g_pDirectoryName, g_pFileEntryArray + pEntryInfo->entryIndex, false, &isDirectoryCreated );

    switch( sfResult )
    {
        case SyncFileResult_Copied:
            {
                app::DialogParam* p = app::AllocDialogParam( app::DialogParam::InfoType_Notice );
                p->GetConsole()->PrintfEx( 0, 0, u"ファイルをセーブデータにコピーしました" );
                app::sequence::Call( app::ExecDialogGroup, p );
            }
            break;
        case SyncFileResult_SizeSkipped:
            app::SetErrorDialog( u"ファイルサイズが大きいので保存できませんでした", nn::ResultSuccess() );
            break;
        case SyncFileResult_ErrorOccurred:
            {
                SyncFileError lastError = app::GetLastSyncFileError();
                nn::Result lastResult = app::GetLastSyncFileResult();
                if ( lastError == SyncFileError_SyncSaveDataDirectory )
                {
                    app::SetErrorDialog( u"ディレクトリの作成に失敗しました", lastResult );
                }
                else
                {
                    app::SetErrorDialog( u"ファイルの保存に失敗しました", lastResult );
                }
            }
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
    }

    return false;
}

// 「SDカードへ保存」項目
bool FileActionMenu_SaveToSd() NN_NOEXCEPT
{
    // 未実装
    return true;
}

//----------------------------------------------------------------
// ファイルアクションメニュー
//
void ExecFileActionMenu( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    g_FileActionMenu.SetHidEvent( &events );
    g_FileActionMenu.Update();

    // A ボタン
    bool isContinued = true;
    switch( g_FileActionMenu.CheckButtonOk() )
    {
        case MenuIndex_OpenFile:
            isContinued = FileActionMenu_OpenFile();
            break;
        case MenuIndex_ReadFile:
            isContinued = FileActionMenu_ReadFile();
            break;
        case MenuIndex_FileSize:
            isContinued = FileActionMenu_FileSize();
            break;
        case MenuIndex_CalculateDigest:
            isContinued = FileActionMenu_CalculateDigest();
            break;
        case MenuIndex_CloseFile:
            isContinued = FileActionMenu_CloseFile();
            break;
        case MenuIndex_SaveFile:
            isContinued = FileActionMenu_SaveFile();
            break;
        case MenuIndex_SaveToSd:
            isContinued = FileActionMenu_SaveToSd();
            break;
        default:
            break;
    }
    if ( ! isContinued )
    {
        return;
    }

    // B ボタンで戻る
    if ( g_FileActionMenu.CheckButtonCancel() )
    {
        app::sequence::Return();
    }

    if ( g_FileActionMenu.IsUpdated() )
    {
        DrawFileActionMenuItems();
        g_FileActionMenu.ClearUpdated();
    }
}


} // namespace app

