﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "BcatTestApp_Common.h"

#include <nn/bcat/bcat_DeliveryCacheProgress.h>
#include <mutex>

namespace app
{
namespace
{
    nn::os::Mutex g_Mutex( true );
    // 進捗
    nn::bcat::DeliveryCacheProgress* g_pDeliveryCacheProgress = nullptr;
    nn::Result g_ProgressLastResult;
    nn::bcat::DeliveryCacheProgressStatus g_ProgressPrevStatus;

    // 進捗の状態表示用文字列
    const char* ProgressStr[] = {
        "None", "Queued", "Connect", "List", "Download", "Commit", "Done", "<Unknown>"
    };
    enum ProgressStrIndex
    {
        ProgressStrIndex_None,
        ProgressStrIndex_Queued,
        ProgressStrIndex_Connect,
        ProgressStrIndex_List,
        ProgressStrIndex_Download,
        ProgressStrIndex_Commit,
        ProgressStrIndex_Done,
        ProgressStrIndex_Unknown,
    };

//----------------------------------------------------------------
// プログレスバー表示
//
void DrawProgressBar( int x, int y, int64_t dlSize, int64_t wholeSize ) NN_NOEXCEPT
{
    app::RectanglePosition framePosition( x, y, x + 40, y + 12 );
    app::DrawFrame( framePosition, DrawColorSet_ProgressBarFrame );

    float rate = dlSize * 100.F / wholeSize;
    rate = (rate < 0.F)? 0.F: (rate>100.F)? 100.F: rate;
    int rw = 37.F * rate / 100.F;
    app::RectanglePosition currentPosition( x + 2, y + 2, x + rw, y + 10 );
    app::DrawRectangle( currentPosition, DrawColorSet_ProgressBar );

    SetGlvColor( ColorSet_White );
    char s[8];
    sprintf( s, "%d%%", static_cast<int>(rate) );
    glv::draw::text( s, x + 50, y, 12.f );
}

//----------------------------------------------------------------
// ダウンロード量の表示
//
void DrawDownloaded( int x, int y, int64_t d ) NN_NOEXCEPT
{
    char s[16];
    nn::util::SNPrintf( s, sizeof(s), "%ld", d );

    glv::Font f;
    f.size( 16.f );
    f.lineSpacing( 1.5f );
    f.tabSpaces( 4 );

    float w;
    float h;
    f.getBounds( w, h, s );

    glv::draw::text( s, x - w, y, 16.f );
}

//----------------------------------------------------------------
// 結果表示
void DrawResult( int x, int y, nn::Result result ) NN_NOEXCEPT
{
    char s[32];
    nn::util::SNPrintf( s, sizeof(s),
                        "ERRCODE: %08x, %03d-%04d",
                        result.GetInnerValueForDebug(),
                        result.GetModule(), result.GetDescription());

    SetGlvColor( result.IsSuccess()? ColorSet_White: ColorSet_Red );
    glv::draw::text( s, x, y, 24.f );
}

//----------------------------------------------------------------
// 進捗ウィンドウの描画
//
void PrintBcatDeliveryCacheProgress() NN_NOEXCEPT
{
    std::lock_guard<decltype( g_Mutex )> lock( g_Mutex );

    // ウィンドウ枠
    app::DrawFrameRectangle( Position_Progress, DrawColorSet_TopMenuBack, DrawColorSet_TopMenuFrame, 3 );

    SetGlvColor( ColorSet_White );
    glv::draw::text( "データ配信状況：", 420, 532, 24.f );

    nn::bcat::DeliveryCacheProgressStatus st =
        g_pDeliveryCacheProgress ? g_pDeliveryCacheProgress->GetStatus() : nn::bcat::DeliveryCacheProgressStatus_None;

    const char* s =
        ( st == nn::bcat::DeliveryCacheProgressStatus_None )? ProgressStr[ ProgressStrIndex_None ] :
        ( st == nn::bcat::DeliveryCacheProgressStatus_Queued )? ProgressStr[ ProgressStrIndex_Queued ] :
        ( st == nn::bcat::DeliveryCacheProgressStatus_Connect )? ProgressStr[ ProgressStrIndex_Connect ] :
        ( st == nn::bcat::DeliveryCacheProgressStatus_ProcessList )? ProgressStr[ ProgressStrIndex_List ] :
        ( st == nn::bcat::DeliveryCacheProgressStatus_Download )? ProgressStr[ ProgressStrIndex_Download ] :
        ( st == nn::bcat::DeliveryCacheProgressStatus_Commit )? ProgressStr[ ProgressStrIndex_Commit ] :
        ( st == nn::bcat::DeliveryCacheProgressStatus_Done )? ProgressStr[ ProgressStrIndex_Done ] :
        ProgressStr[ ProgressStrIndex_Unknown ];

    glv::draw::text( s, 600, 532, 24.f );

    // ディレクトリ名表示・全体のプログレスバー表示
    if ( st == nn::bcat::DeliveryCacheProgressStatus_Download ||
         st == nn::bcat::DeliveryCacheProgressStatus_Commit )
    {
        SetGlvColor( ColorSet_White );
        glv::draw::text( "Directory:", 430, 562, 16.f );
        glv::draw::text( g_pDeliveryCacheProgress->GetCurrentDirectoryName().value, 520, 562, 16.f );

        int64_t w0 = g_pDeliveryCacheProgress->GetWholeDownloaded();
        int64_t w1 = g_pDeliveryCacheProgress->GetWholeTotal();
        DrawProgressBar( 1030, 540, w0, w1 );
    }

    // ファイル名表示・ファイルのプログレスバー表示
    if ( st == nn::bcat::DeliveryCacheProgressStatus_Download )
    {
        SetGlvColor( ColorSet_White );
        glv::draw::text( "File:", 430, 580, 16.f );
        glv::draw::text( g_pDeliveryCacheProgress->GetCurrentFileName().value, 520, 580, 16.f );

        int64_t w0 = g_pDeliveryCacheProgress->GetCurrentDownloaded();
        int64_t w1 = g_pDeliveryCacheProgress->GetCurrentTotal();
        DrawProgressBar( 930, 540, w0, w1 );
        DrawDownloaded( 1030, 580, w1 );
    }

    // 結果の表示
    if ( st == nn::bcat::DeliveryCacheProgressStatus_Done )
    {
        DrawResult( 700, 532, g_ProgressLastResult );
    }
}

//----------------------------------------------------------------
// 進捗情報の更新
//
void UpdateProgress() NN_NOEXCEPT
{
    std::lock_guard<decltype( g_Mutex )> lock( g_Mutex );

    if( !g_pDeliveryCacheProgress )
    {
        return;
    }

    g_pDeliveryCacheProgress->Update();
    nn::bcat::DeliveryCacheProgressStatus st = g_pDeliveryCacheProgress->GetStatus();

    // 進捗の状態が Done に変化した瞬間、結果を取得
    if( st != g_ProgressPrevStatus &&
        st == nn::bcat::DeliveryCacheProgressStatus_Done )
    {
        g_ProgressLastResult = g_pDeliveryCacheProgress->GetResult();

        // システムコンソールに報告
        GetSystemConsole().Printf( "nn::bcat::DeliveryCacheProgress.GetResult() :" );
        app::PrintErrorCode( g_ProgressLastResult );
    }

    g_ProgressPrevStatus = st;
}
} //namespace

//----------------------------------------------------------------
// 進捗情報のクリア
//
void InitializeProgressUpdate() NN_NOEXCEPT
{
    // 進捗
    g_ProgressPrevStatus = nn::bcat::DeliveryCacheProgressStatus_None;
    g_ProgressLastResult = nn::ResultSuccess();
}

//----------------------------------------------------------------
// 進捗ウィンドウ
//
// 描画
void DrawProgress( void* arg ) NN_NOEXCEPT
{
    PrintBcatDeliveryCacheProgress();
}
// 描画タスクを追加
void AddProgressDraw() NN_NOEXCEPT
{
    if ( ! app::sequence::IsExistDrawCallback( DrawProgress ) )
    {
        app::sequence::AddDrawCallback( DrawProgress, nullptr, 300, 0 );
    }
}
// 描画タスクを削除
void RemoveProgressDraw() NN_NOEXCEPT
{
    app::sequence::RemoveDrawCallback( DrawProgress );
}

//----------------------------------------------------------------
// 常に進捗を取得し続けるタスク
//
// 進捗の更新
void ExecProgressUpdate( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    UpdateProgress();
}
// 進捗オブジェクトを追加
void RegisterProgress( nn::bcat::DeliveryCacheProgress* pProgress ) NN_NOEXCEPT
{
    std::lock_guard<decltype( g_Mutex )> lock( g_Mutex );

    if (g_pDeliveryCacheProgress)
    {
        delete g_pDeliveryCacheProgress;
    }
    g_pDeliveryCacheProgress = pProgress;
}
// タスクを追加
void AddProgressExec() NN_NOEXCEPT
{
    if ( ! app::sequence::IsExistPreTaskCallback( ExecProgressUpdate ) )
    {
        app::InitializeProgressUpdate();
        app::sequence::AddPreTaskCallback( ExecProgressUpdate, nullptr, 100 );
    }
}
// タスクを削除
void RemoveProgressExec() NN_NOEXCEPT
{
    app::sequence::RemovePreTaskCallback( ExecProgressUpdate );
}

} //namespace app
