﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "BcatSystemDebugTool_Common.h"
#include "BcatSystemDebugTool_TopMenu.h"

#include <nn/bcat/bcat_ApiDebug.h>
#include <nn/time/time_Api.h>
#include <nn/time/time_TimeZoneApi.h>
#include <nn/time/time_StandardNetworkSystemClock.h>

//================================================================
// デバッグ用途のフラグ
// これを定義すると、表示実験のためにログをたくさん読み込んだことにする
//#define APP_PRETEND_TO_READ_MANY_LOGS_FOR_DEBUG
//================================================================

namespace app
{
void ExecPushNotificationLogMenu_Initialize( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT;
void ExecPushNotificationLogMenu           ( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT;
void ExecPushNotificationLogMenu_Finalize  ( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT;
void DrawPushNotificationLog( void* arg ) NN_NOEXCEPT;

ExecCallbackGroup ExecPushNotificationLogMenuGroup = {
    ExecPushNotificationLogMenu_Initialize,
    ExecPushNotificationLogMenu,
    ExecPushNotificationLogMenu_Finalize,
    nullptr,

    DrawPushNotificationLog,
    nullptr,
    DrawPriority_SubMenu,
    0
};

void ExecPushNotificationLogSubMenu_Initialize( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT;
void ExecPushNotificationLogSubMenu           ( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT;
void ExecPushNotificationLogSubMenu_Finalize  ( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT;

ExecCallbackGroup ExecPushNotificationLogSubMenuGroup = {
    ExecPushNotificationLogSubMenu_Initialize,
    ExecPushNotificationLogSubMenu,
    ExecPushNotificationLogSubMenu_Finalize,
    nullptr,

    nullptr,
    nullptr,
    0,
    0
};

namespace
{
    // コンソール
    const int ConsoleWidth = 80;
    const int ConsoleHeight = 35;
    const int32_t ConsoleBufferSize = APP_CONSOLE_BUFFER_SIZE( ConsoleWidth, ConsoleHeight );
    char* g_ConsoleBuffer;
    app::FixedConsole<char> g_Console;
    bool g_IsConsoleCreated = false;

    bool g_IsUpdate = false;

    // プッシュ通知ログ
    nn::bcat::PushNotificationLog* g_PushLogPtr;

    const int ListViewMax = 16;
    int g_ListViewTop = 0;
    int g_ListNum = 0;
    int g_ListCursor = 0;

    struct PushLogInfo
    {
        int pushLogIndex;
    };
    PushLogInfo g_PushLogInfo[ nn::bcat::PushNotificationLogCountMax ];

    // ソート
    enum SortType
    {
        SortType_ByApplicationId,
        SortType_ByLogNo,
    };
    SortType g_CurrentSortType = SortType_ByLogNo;

    // 定期リスト更新用
    int g_UpdateCount;
    const int UpdatePeriod = 300;
} //namespace

// 前方参照用
void SortPushNotificationLogByApplicationId() NN_NOEXCEPT;
void SortPushNotificationLogByLogNo() NN_NOEXCEPT;

//----------------------------------------------------------------
// プッシュ通知ログコンソールのプリンタ
//
void  DrawPushNotificationLogConsoleString( int x, int y, char* string, char* attr, int stringLen, void* arg ) NN_NOEXCEPT
{
    DrawFixedConsoleGeneric( 80, 110, 0, 24, 16, 24, x, y, string, attr, stringLen, arg );
}

//----------------------------------------------------------------
// プッシュ通知ログの描画コールバック
//
void DrawPushNotificationLog( void* arg ) NN_NOEXCEPT
{
    // フレーム
    app::DrawFrameRectangle( Position_PushNotificationLog, app::DrawColorSet_PushNotificationLogBack, app::DrawColorSet_PushNotificationLogFrame );

    if ( g_ListNum > 0 )
    {
        app::DrawRectangle( Position_PushNotificationLog.l + 42,
                            Position_PushNotificationLog.t + 108 + (g_ListCursor * 24),
                            Position_PushNotificationLog.r - 20,
                            Position_PushNotificationLog.t + 108 + (g_ListCursor * 24) + 24,
                            DrawColorSet_ItemCursorBack );
    }

    // スクロールバー
    app::DrawScrollBar( g_ListNum, ListViewMax, g_ListViewTop, ScrollBarDefaultViewWidth, Position_PushNotificationLogScrollBar );

    // コンソール
    if ( g_IsConsoleCreated )
    {
        g_Console.Display();
    }
}

//----------------------------------------------------------------
// プッシュ通知ログをコンソールへプリント
//
void PrintPushNotificationLog() NN_NOEXCEPT
{
    g_Console.Clear();

    g_Console.PrintfEx( 0, 0, " Push Notification Log : %d item(s)", g_ListNum );

    // ログがなかった場合
    if ( g_ListNum == 0 )
    {
        g_Console.PrintfEx( 0, 2, app::ConsoleColor_Yellow, "  <no push notification log>" );
        return;
    }

    g_Console.PrintfEx( 0, 2, "  No. LogNo.  Application ID       Received Time" );
    g_Console.PrintfEx( 0, 3, "  ---  ---  ------------------  -------------------------" );

    //現在時刻
    nn::time::PosixTime currentTime;
    nn::Result result = nn::time::StandardNetworkSystemClock::GetCurrentTime(&currentTime);

    // 各行の表示
    for( int i=0; i<ListViewMax; i++ )
    {
        int num = g_ListViewTop + i;
        if ( num >= g_ListNum )
        {
            break;
        }

        int displayLine = i + 4;
        int index = g_PushLogInfo[ num ].pushLogIndex;
        g_Console.PrintfEx( 2, displayLine, "%3d  %3d  0x%016llx",
                            num,
                            (0<=index && index<=nn::bcat::PushNotificationLogCountMax)? index: -1,
                            (0<=index && index<=nn::bcat::PushNotificationLogCountMax)? g_PushLogPtr[index].appId.value: 0xffffffff );

        if (0<=index && index<=nn::bcat::PushNotificationLogCountMax)
        {
            // ReceivedTime を CalenderTime に変換
            nn::time::CalendarTime calendarTime = {};
            nn::time::CalendarAdditionalInfo additionalInfo = {};
            NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::ToCalendarTime( &calendarTime, &additionalInfo, g_PushLogPtr[index].receivedTime ) );

            if ( g_PushLogPtr[index].receivedTime.value == 0 )
            {
                g_Console.PrintfEx( 32, displayLine, app::ConsoleColor_Red, "invalid" );
                continue;
            }

            char buf[64];
            nn::util::SNPrintf( buf, sizeof(buf), "%04d/%02d/%02d %02d:%02d:%02d %s",
                                calendarTime.year, calendarTime.month, calendarTime.day,
                                calendarTime.hour, calendarTime.minute, calendarTime.second,
                                additionalInfo.timeZone.standardTimeName );

            g_Console.PrintfEx( 32, displayLine, app::ConsoleColor_White, buf );

            // 現在時刻の取得に失敗している場合
            if ( result.IsFailure() )
            {
                continue;
            }

            // 現在時刻との差
            const int PosX = 58;
            nn::TimeSpan span = currentTime - g_PushLogPtr[index].receivedTime;
            if ( span < nn::TimeSpan::FromSeconds(0) )
            {
                g_Console.PrintfEx( PosX, displayLine, "-" );
            }
            else if ( span <= nn::TimeSpan::FromMinutes(1) ) // 1分以内
            {
                g_Console.PrintfEx( PosX, displayLine, "%ds", span.GetSeconds() );
            }
            else if ( span <= nn::TimeSpan::FromHours(1) ) // 1時間以内
            {
                g_Console.PrintfEx( PosX, displayLine, "%dm%ds", span.GetMinutes(), span.GetSeconds() % 60 );
            }
            else if ( span <= nn::TimeSpan::FromDays(1) ) // 1日以内
            {
                g_Console.PrintfEx( PosX, displayLine, "%dh%dm%ds", span.GetHours(), span.GetMinutes() % 60, span.GetSeconds() % 60 );
            }
            else if ( span <= nn::TimeSpan::FromDays(100) ) // 100日以内
            {
                g_Console.PrintfEx( PosX, displayLine, "%dd%dh%dm%ds", span.GetDays(), span.GetHours() % 24, span.GetMinutes() % 60, span.GetSeconds() % 60 );
            }
            else // それ以上
            {
                g_Console.PrintfEx( PosX, displayLine, "%dd", span.GetDays() );
            }
        }
    }
}

//----------------------------------------------------------------
// プッシュ通知ログを取得する
//
nn::Result GetPushNotificationLogFromSystem() NN_NOEXCEPT
{
    app::GetSystemConsole().Printf("nn::bcat::GetPushNotificationLog() :" );
    nn::Result result = nn::bcat::GetPushNotificationLog( &g_ListNum, g_PushLogPtr, nn::bcat::PushNotificationLogCountMax );
    app::PrintErrorCode( result );

    if ( result.IsFailure() )
    {
        g_ListNum = 0;
        return result;
    }
    app::GetSystemConsole().Printf("Push notification log number: %d\n", g_ListNum );

#ifdef APP_PRETEND_TO_READ_MANY_LOGS_FOR_DEBUG
    // (DEBUG) たくさんログを読めたことにする
    g_ListNum = 20; // 20個とする
    for( int i=0; i<g_ListNum; i++ )
    {
        g_PushLogPtr[i].appId.value = 0x120000 * i;

        nn::time::PosixTime t;
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::StandardNetworkSystemClock::GetCurrentTime(&t) );
        t = t - nn::TimeSpan::FromSeconds( i * 10 );
        g_PushLogPtr[i].receivedTime = t;
    }
#endif

    for( int i=0; i<g_ListNum; i++ )
    {
        g_PushLogInfo[i].pushLogIndex = i;
    }

    // 必要ならソート
    // sort
    return nn::ResultSuccess();
}


//----------------------------------------------------------------
// プッシュ通知ログメニューのキー説明
void PrintPushNotificationLogMenuHelp() NN_NOEXCEPT
{
    app::GetHelpConsole().Clear();
    app::GetHelpConsole().PrintfEx( 2, 0, GetText( TextResource_HelpPushNotificationLogMenu ) );
}

//----------------------------------------------------------------
// Call からの戻り関数
namespace
{
void AfterSort( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    g_IsUpdate = true;
    PrintPushNotificationLogMenuHelp();
}

void JumpToTopMenu( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    app::sequence::JumpTo( ExecTopMenuGroup, nullptr );
}
} //namespace

//----------------------------------------------------------------
// プッシュ通知ログの実行コールバック(初回処理)
//
void ExecPushNotificationLogMenu_Initialize( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    // コンソール作成など
    g_ConsoleBuffer = new char[ ConsoleBufferSize ];
    g_Console.SetBuffer( g_ConsoleBuffer, ConsoleBufferSize, ConsoleWidth, ConsoleHeight );
    g_Console.SetPrinter( DrawPushNotificationLogConsoleString );
    g_IsConsoleCreated = true;

    g_PushLogPtr = new nn::bcat::PushNotificationLog[ nn::bcat::PushNotificationLogCountMax ];
    g_IsUpdate = true;

    // 操作説明
    PrintPushNotificationLogMenuHelp();

    // ログの読み込み
    g_ListViewTop = 0;
    g_ListCursor = 0;
    nn::Result result = GetPushNotificationLogFromSystem();

    if ( result.IsFailure() )
    {
        app::SetErrorDialog( u"Cannot read the push notification log.", result );
        app::sequence::SetFromCall( JumpToTopMenu, nullptr );
        return;
    }

    // ソートしておく
    if ( g_CurrentSortType == SortType_ByApplicationId )
    {
        SortPushNotificationLogByApplicationId();
    }
    else if ( g_CurrentSortType == SortType_ByLogNo )
    {
        SortPushNotificationLogByLogNo();
    }

    // ログ表示
    PrintPushNotificationLog();

    // 定期リスト更新カウント
    g_UpdateCount = UpdatePeriod;
}

//----------------------------------------------------------------
// プッシュ通知ログの実行コールバック(終了処理)
//
void ExecPushNotificationLogMenu_Finalize( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    delete[] g_ConsoleBuffer;
    delete[] g_PushLogPtr;
}

//----------------------------------------------------------------
// プッシュ通知ログの実行コールバック
//
void ExecPushNotificationLogMenu( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    app::Pad pad( events );

    // X ボタンでサブメニュー
    if ( pad.IsButtonDownX() )
    {
        app::sequence::Call( ExecPushNotificationLogSubMenuGroup );
        app::sequence::SetFromCall( AfterSort, nullptr );
        return;
    }

    // B ボタンで戻る
    if ( pad.IsButtonDownB() )
    {
        app::sequence::JumpTo( ExecTopMenuGroup, nullptr );
        return;
    }

    if ( g_ListNum > 0 )
    {
        // カーソル上
        if ( pad.IsButtonDownUp() || pad.IsButtonRepeatUp() )
        {
            RotateWithinViewRange( g_ListCursor, -1, g_ListViewTop, ListViewMax, g_ListNum );
            g_IsUpdate = true;
        }
        // カーソル下
        if ( pad.IsButtonDownDown() || pad.IsButtonRepeatDown() )
        {
            RotateWithinViewRange( g_ListCursor, 1, g_ListViewTop, ListViewMax, g_ListNum );
            g_IsUpdate = true;
        }
        // カーソル左
        if ( pad.IsButtonDownLeft() )
        {
            g_ListCursor = 0;
            g_ListViewTop = 0;
            g_IsUpdate = true;
        }
        // カーソル右
        if ( pad.IsButtonDownRight() )
        {
            g_ListCursor = (g_ListNum < ListViewMax)? (g_ListNum - 1): (ListViewMax - 1);
            g_ListViewTop = (g_ListNum < ListViewMax)? 0: (g_ListNum - ListViewMax);
            g_IsUpdate = true;
        }
    }

    // 定期リスト更新
    if ( -- g_UpdateCount < 0 )
    {
        g_UpdateCount = UpdatePeriod;
        g_IsUpdate = true;
    }

    if ( g_IsUpdate )
    {
        g_IsUpdate = false;
        PrintPushNotificationLog();
    }
} //NOLINT(impl/function_size)

//================================================================
//
namespace
{
    app::Menu g_Menu;

    const int SubMenuPositionX = 310;
    const int SubMenuPositionY = 250;
    const int SubMenuSizeX = 600;
    const int SubMenuSizeY = 200;
    const app::RectanglePosition SubMenuPosition( SubMenuPositionX, SubMenuPositionY, SubMenuPositionX + SubMenuSizeX, SubMenuPositionY + SubMenuSizeY );

    enum SubMenuItem
    {
        MenuItem_SortByApplicationId = 0,
        MenuItem_SortByLogNo,
    };

} //namespace

//----------------------------------------------------------------
// サブメニューの描画コールバック
//
void PushNotificationLogSubMenuConsoleCallback( app::FixedProportionalConsole<char16_t>* console, void* ) NN_NOEXCEPT
{
    console->PrintfEx( 1, 0, app::ConsoleColor_Yellow, GetText( TextResource_SortLogTitle ) );
    console->SetAttribute( app::ConsoleColor_White );
    console->PrintfEx( 2, 2 + MenuItem_SortByApplicationId,
                       (g_CurrentSortType == SortType_ByApplicationId)? GetText( TextResource_SortByAppCurrent ): GetText( TextResource_SortByApp ) );
    console->PrintfEx( 2, 2 + MenuItem_SortByLogNo,
                       (g_CurrentSortType == SortType_ByLogNo)? GetText( TextResource_SortByNoCurrent ): GetText( TextResource_SortByNo ) );
}

//----------------------------------------------------------------
// サブメニューのキー説明
void PrintPushNotificationLogSubMenuHelp() NN_NOEXCEPT
{
    app::GetHelpConsole().Clear();
    app::GetHelpConsole().PrintfEx( 2, 0, GetText( TextResource_HelpPushNotificationLogMenuSubMenu ) );
}

//----------------------------------------------------------------
// ソート比較関数
//
bool CompFunction_ForSortPushLogByApplicationId( PushLogInfo lhs, PushLogInfo rhs ) NN_NOEXCEPT
{
    int lhsIndex = lhs.pushLogIndex;
    int rhsIndex = rhs.pushLogIndex;

    uint64_t lhsId = g_PushLogPtr[lhsIndex].appId.value;
    uint64_t rhsId = g_PushLogPtr[rhsIndex].appId.value;

    if ( lhsId != rhsId )
    {
        return lhsId > rhsId;
    }
    return lhsIndex < rhsIndex;
}
bool CompFunction1_ForSortPushLogByLogNo( PushLogInfo lhs, PushLogInfo rhs ) NN_NOEXCEPT
{
    return ( lhs.pushLogIndex < rhs.pushLogIndex );
}

//----------------------------------------------------------------
// ソート： アプリIDでプッシュ通知ログをソート
//
void SortPushNotificationLogByApplicationId() NN_NOEXCEPT
{
    // g_PushLogInfo[] をソート。個数は g_ListNum

    // g_PushLogInfo の複製
    PushLogInfo* s0 = new PushLogInfo[ nn::bcat::PushNotificationLogCountMax ];
    memcpy( s0, g_PushLogInfo, sizeof( PushLogInfo[ nn::bcat::PushNotificationLogCountMax ] ) );

    std::vector<PushLogInfo> vec( g_ListNum );
    for( int i=0; i<g_ListNum; i++ )
    {
        vec[i] = s0[i];
    }
    sort( vec.begin(), vec.end(), CompFunction_ForSortPushLogByApplicationId );

    for( int i=0; i<g_ListNum; i++ )
    {
        memcpy( &g_PushLogInfo[i], &vec.at(i), sizeof( PushLogInfo ) );
    }

    delete[] s0;
}

//----------------------------------------------------------------
// ソート： ログ番号でプッシュ通知ログをソート
//
void SortPushNotificationLogByLogNo() NN_NOEXCEPT
{
    // g_PushLogInfo[] をソート。個数は g_ListNum

    // g_PushLogInfo の複製
    PushLogInfo* s0 = new PushLogInfo[ nn::bcat::PushNotificationLogCountMax ];
    memcpy( s0, g_PushLogInfo, sizeof( PushLogInfo[ nn::bcat::PushNotificationLogCountMax ] ) );

    std::vector<PushLogInfo> vec( g_ListNum );
    for( int i=0; i<g_ListNum; i++ )
    {
        vec[i] = s0[i];
    }
    sort( vec.begin(), vec.end(), CompFunction1_ForSortPushLogByLogNo );

    for( int i=0; i<g_ListNum; i++ )
    {
        memcpy( &g_PushLogInfo[i], &vec.at(i), sizeof( PushLogInfo ) );
    }

    delete[] s0;
}

//----------------------------------------------------------------
void ExecPushNotificationLogSubMenu_Initialize( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    const int PushNotificationLogSubConsoleTag = 0x200;
    g_Menu.CreateConsole( app::ConsoleSize_Char16_t, SubMenuPosition, 80, 5, 1, 2, 24, 6, app::DrawPriority_SubMenu + 1, PushNotificationLogSubConsoleTag );
    g_Menu.SetItemParameter( 2, 0 );
    g_Menu.SetCallback16( PushNotificationLogSubMenuConsoleCallback, nullptr );

    // 説明
    PrintPushNotificationLogSubMenuHelp();
}
//----------------------------------------------------------------
void ExecPushNotificationLogSubMenu_Finalize( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    g_Menu.DestroyConsole();
}
//----------------------------------------------------------------
void ExecPushNotificationLogSubMenu( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    // キー確認
    g_Menu.SetHidEvent( &events );
    g_Menu.Update();

    switch( g_Menu.CheckButtonOk() )
    {
        case MenuItem_SortByApplicationId:
            {
                g_CurrentSortType = SortType_ByApplicationId;
                SortPushNotificationLogByApplicationId();
                app::sequence::Return();
                return;
            }
            break;
        case MenuItem_SortByLogNo:
            {
                g_CurrentSortType = SortType_ByLogNo;
                SortPushNotificationLogByLogNo();
                app::sequence::Return();
                return;
            }
            break;
        default:
            break;
    }

    //---- キャンセル
    if ( g_Menu.CheckButtonCancel() )
    {
        app::sequence::Return();
        return;
    }

    if ( g_Menu.IsUpdated() )
    {
        g_Menu.ClearUpdated();
    }
}

} //namespace app
