﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <vector>
// #include <nn/fs.h>
#include <nn/fs/fs_SaveDataTypes.h>
#include <nn/account/account_Api.h>
#include "BackupSaveData.h"

namespace {
    class AccountVector {
    private:
        std::vector<nn::fs::UserId> m_AccountVector;

    public:
        AccountVector() {}
        ~AccountVector() {}

        void SetAccount(nn::fs::UserId userId)
        {
            auto itr = std::find(m_AccountVector.begin(), m_AccountVector.end(), userId);
            if( itr == m_AccountVector.end() )
            {
                m_AccountVector.push_back(userId);
            }
        }

        void SetAccountVector(std::vector<nn::fs::UserId> accountvector)
        {
            m_AccountVector = accountvector;
        }

        nn::fs::UserId GetAccount(int index)
        {
            if(index < 0)
            {
                return nn::fs::InvalidUserId;
            }
            return m_AccountVector[index];
        }

        int GetIndex(nn::fs::UserId userId)
        {
            auto itr = std::find(m_AccountVector.begin(), m_AccountVector.end(), userId);
            int index = static_cast<int>(std::distance( m_AccountVector.begin(), itr ));
            if(index == Count())
            {
                return -1;
            }
            return index;
        }

        bool Contains(nn::fs::UserId userId)
        {
            return GetIndex(userId) >= 0;
        }

        int Count()
        {
            return static_cast<int>(m_AccountVector.size());
        }
    };

    bool accountVectorInitialized = false;
    AccountVector backuppedAccountVector;
    AccountVector devkitAccountVector;

    void AccountInitialize()
    {
        if(!accountVectorInitialized)
        {
            accountVectorInitialized = true;
        }
    }
}

bool SetDevKitAccount(nn::fs::UserId userId)
{
    AccountInitialize();

    devkitAccountVector.SetAccount(userId);
    return true;
}

bool SetBackuppedAccount(nn::fs::UserId userId)
{
    AccountInitialize();

    backuppedAccountVector.SetAccount(userId);
    return true;
}

bool SetBackuppedAccountVector(std::vector<nn::fs::UserId> accountvector)
{
    AccountInitialize();

    backuppedAccountVector.SetAccountVector(accountvector);
    return true;
}

nn::fs::UserId ConvertAccount(nn::fs::UserId userId)
{
    if(!backuppedAccountVector.Contains(userId))
    {
        return nn::fs::InvalidUserId;
    }

    int index = backuppedAccountVector.GetIndex(userId);
    return devkitAccountVector.GetAccount(index);
}

::nn::Result GetAccountVectorFromDevkit()
{
    nn::Result result;
    int userIdNum;
    nn::account::Uid userIds[nn::account::UserCountMax];

    result = nn::account::ListAllUsers(&userIdNum, userIds, nn::account::UserCountMax);
    if(!result.IsSuccess())
    {
        NN_LOG("error: ListAllUsers() failed. (%08x)\n", result.GetInnerValueForDebug());
        return result;
    }

    for(int i=0;i<userIdNum;i++)
    {
        nn::fs::UserId userId = {{userIds[i]._data[0], userIds[i]._data[1]}};
        devkitAccountVector.SetAccount(userId);
    }

    return nn::ResultSuccess();
}

int GetDevkitAccountNum()
{
    return devkitAccountVector.Count();
}

nn::fs::UserId GetDevKitAccount(int index)
{
    return devkitAccountVector.GetAccount(index);
}

bool CheckDevkitAccountExist(nn::fs::UserId userId)
{
    return devkitAccountVector.Contains(userId);
}

bool AccountNumCheck()
{
    int devkitAccountNum = devkitAccountVector.Count();
    int backuppedAccountNum = backuppedAccountVector.Count();
    if(backuppedAccountNum > nn::account::UserCountMax)
    {
        NN_LOG("error: The number of backupped account is more than nn::account::UserCountMax. (%d)\n", backuppedAccountNum);
        return false;
    }
    if(devkitAccountNum < backuppedAccountNum)
    {
        NN_LOG("error: The number of devkit account is less than backupped one. (devkit:%d, backup:%d)\n", devkitAccountNum, backuppedAccountNum);
        return false;
    }

    return true;
}

void DisplayConvertAccountInfo()
{
    // MEMO: AccountNumCheck() が実行済みであることが前提
    if(!accountVectorInitialized)
    {
        NN_LOG("warn: backuppedAccountVector is not initialized.\n");
        return;
    }

#if !defined(NN_DISABLE_LOG)
    NN_LOG("Account replace info:\n");
    for(int i=0; i< backuppedAccountVector.Count(); i++)
    {
        nn::fs::UserId devKitId = devkitAccountVector.GetAccount(i);
        nn::fs::UserId backuppedId = backuppedAccountVector.GetAccount(i);

        NN_LOG("    %016llx %016llx -> %016llx %016llx\n",
                backuppedId._data[0], backuppedId._data[1],
                devKitId._data[0], devKitId._data[1]);
    }
#endif
}
