﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/grcsrv/grcsrv_SetAllocatorForMalloc.h>

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/mem/mem_StandardAllocator.h>
#include <cstring>
#include <new>

namespace {

nn::mem::StandardAllocator* g_pMallocAllocator;

}

namespace nn { namespace grcsrv {

void SetStandardAllocatorForMalloc(mem::StandardAllocator* p) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(p ? g_pMallocAllocator == nullptr : true);
    g_pMallocAllocator = p;
}

}}

extern "C" void* malloc(size_t size)
{
    return g_pMallocAllocator->Allocate(size);
}

extern "C" void free(void* p)
{
    if (p)
    {
        g_pMallocAllocator->Free(p);
    }
}

extern "C" void* calloc(size_t num, size_t size)
{
    size *= num;
    auto p = g_pMallocAllocator->Allocate(size);
    if (!p)
    {
        return nullptr;
    }
    std::memset(p, 0, size);
    return p;
}

extern "C" void* realloc(void* p, size_t newSize)
{
    return g_pMallocAllocator->Reallocate(p, newSize);
}

extern "C" void* aligned_alloc(size_t alignment, size_t size)
{
    return g_pMallocAllocator->Allocate(size, alignment);
}

extern "C" size_t malloc_usable_size(void* p)
{
    if (!p)
    {
        return 0;
    }
    return g_pMallocAllocator->GetSizeOf(p);
}

void* operator new(std::size_t size)
{
    return malloc(size);
}
void* operator new(std::size_t size, const std::nothrow_t&) NN_NOEXCEPT
{
    return malloc(size);
}

void  operator delete(void* p) NN_NOEXCEPT
{
    free(p);
}

void* operator new[](std::size_t size)
{
    return malloc(size);
}

void* operator new[](std::size_t size, const std::nothrow_t&) NN_NOEXCEPT
{
    return malloc(size);
}

void  operator delete[](void* p) NN_NOEXCEPT
{
    free(p);
}
