﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Assert.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/os.h>
#include <nn/mem.h>
#include <nn/time.h>
#include <nn/fs/fs_ApiPrivate.h>
#include <nn/time/time_Api.h>
#include <nn/capsrv/capsrv_ServiceConfig.h>

#include "../../Libraries/capsrv/capsrv_DecoderControlApi.h"
#include "../../Libraries/capsrv/server/capsrvServer_Config.h"
#include "../../Libraries/capsrv/server/capsrvServer_InitializeForAlbumServer.h"

#include "capsrv_InitializeFileSystem.h"

#include <nn/spl/spl_Api.h>
#include "../../Libraries/capsrv/server/detail/capsrvServer_CalculateJpegMac.h"
#include "../../Libraries/capsrv/server/album/capsrvServer_AlbumMovieUtility.h"

//---------------
// ヒープ
//---------------
static const size_t HeapSize = 32 * 1024; // nn::sf で使う
NN_ALIGNAS(4096) char g_HeapMemory[HeapSize];

nn::mem::StandardAllocator g_Allocator;

extern "C" void* malloc(size_t size) NN_NOEXCEPT
{
    auto p = g_Allocator.Allocate(size);
    //ReportMemoryUsage();
    return p;
}

extern "C" void free(void* p) NN_NOEXCEPT
{
    if(p)
    {
        g_Allocator.Free(p);
    }
    //ReportMemoryUsage();
}

extern "C" void* calloc(size_t num, size_t size)
{
    size_t sum = num * size;
    void*  p   = malloc(sum);
    if (p)
    {
        std::memset(p, 0, sum);
    }
    return p;
}

extern "C" void* realloc(void* p, size_t newSize)
{
    void* q = g_Allocator.Reallocate(p, newSize);
    //ReportMemoryUsage();
    return q;
}

extern "C" void* aligned_alloc(size_t alignment, size_t size)
{
    void* p = g_Allocator.Allocate(size, alignment);
    //ReportMemoryUsage();
    return p;
}


//---------------
// スレッド設定
//---------------
static const size_t AlbumControlIpcServerThreadStackSize = 32 * 1024;
static const size_t AlbumIpcServerThreadStackSize        = 32 * 1024;

//----------------------------
// Threading
//----------------------------
namespace {
    nn::os::ThreadType g_AlbumControlIpcServerThread;
    nn::os::ThreadType g_AlbumIpcServerThread;

    NN_ALIGNAS(4096) char g_AlbumControlIpcServerThreadStack[AlbumControlIpcServerThreadStackSize];
    NN_ALIGNAS(4096) char g_AlbumIpcServerThreadStack       [AlbumIpcServerThreadStackSize];


    void CreateStartThreads() NN_NOEXCEPT
    {
        NN_CAPSRV_LOG_INFO("Starting AlbumControlIpcServerThread ...\n");
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::os::CreateThread(
                &g_AlbumControlIpcServerThread,
                nn::capsrv::server::AlbumControlServerThreadFunction,
                nullptr,
                &g_AlbumControlIpcServerThreadStack,
                AlbumControlIpcServerThreadStackSize,
                NN_SYSTEM_THREAD_PRIORITY(capsrv, SystemIpcServer)
            )
        );
        nn::os::SetThreadNamePointer(&g_AlbumControlIpcServerThread, NN_SYSTEM_THREAD_NAME(capsrv, SystemIpcServer));
        nn::os::StartThread(&g_AlbumControlIpcServerThread);

        NN_CAPSRV_LOG_INFO("Starting AlbumIpcServerThread ...\n");
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::os::CreateThread(
                &g_AlbumIpcServerThread,
                nn::capsrv::server::AlbumServerThreadFunction,
                nullptr,
                &g_AlbumIpcServerThreadStack,
                AlbumIpcServerThreadStackSize,
                NN_SYSTEM_THREAD_PRIORITY(capsrv, ApplicationIpcServer)
            )
        );
        nn::os::SetThreadNamePointer(&g_AlbumIpcServerThread, NN_SYSTEM_THREAD_NAME(capsrv, ApplicationIpcServer));
        nn::os::StartThread(&g_AlbumIpcServerThread);
    }

    void WaitDestroyThreads() NN_NOEXCEPT
    {
        nn::os::WaitThread(&g_AlbumControlIpcServerThread);
        nn::os::WaitThread(&g_AlbumIpcServerThread);

        nn::os::DestroyThread(&g_AlbumControlIpcServerThread);
        nn::os::DestroyThread(&g_AlbumIpcServerThread);
    }

    void SetScreenShotFileSignFunction() NN_NOEXCEPT
    {
        static const char KekSource[16] = { 0x9B, 0xBF, 0xDE, 0x01, 0xFA, 0x18, 0xBB, 0xEE, 0x87, 0xA4, 0xAC, 0xC5, 0x67, 0xAC, 0xD4, 0x21, };
        static const char KeySource[16] = { 0x15, 0x0D, 0x26, 0xD1, 0x17, 0xEE, 0x44, 0xFF, 0x79, 0x67, 0x43, 0x8A, 0x20, 0x14, 0x63, 0x30, };

        static nn::spl::AccessKey g_AccessKey;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::spl::GenerateAesKek(&g_AccessKey, KekSource, sizeof(KekSource), 0, 0));

        nn::capsrv::server::detail::SetScreenShotFileSignFunction(
            nn::capsrv::server::detail::ScreenShotFileSignatureVersion_Version1,
            [](nn::capsrv::server::detail::Signature* pOut, const void* data, size_t dataSize) -> nn::Result
        {
            int slot;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::spl::AllocateAesKeySlot(&slot));
            NN_UTIL_SCOPE_EXIT
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::spl::DeallocateAesKeySlot(slot));
            };
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::spl::LoadAesKey(slot, g_AccessKey, KeySource, sizeof(KeySource)));
            *pOut = {}; // nn::spl::ComputeCmac の対象サイズが sizeof(Signature) より小さいので先に全領域をゼロ初期化
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::spl::ComputeCmac(pOut, sizeof(*pOut), slot, data, dataSize));
            NN_RESULT_SUCCESS;
        });
    }

    void SetMovieFileSignFunction() NN_NOEXCEPT
    {
        static const char KekSource[16] = { 0x55, 0xB4, 0x30, 0xAF, 0x16, 0x8D, 0x9A, 0x66, 0x6A, 0x8F, 0xED, 0xE3, 0xA5, 0x52, 0x32, 0xEA, };
        static const char KeySource[16] = { 0x71, 0x70, 0x7D, 0xED, 0x3E, 0x2F, 0x48, 0x03, 0x37, 0x2F, 0x42, 0x35, 0x12, 0x39, 0x2C, 0xFE, };

        static nn::spl::AccessKey g_AccessKey;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::spl::GenerateAesKek(&g_AccessKey, KekSource, sizeof(KekSource), 0, 0));

        nn::capsrv::server::album::AlbumMovieUtility::SetCalculateFileSign([](nn::capsrv::server::album::AlbumMovieUtility::FileSign* pOut, const void* data, size_t dataSize)
        {
            int slot;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::spl::AllocateAesKeySlot(&slot));
            NN_UTIL_SCOPE_EXIT
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::spl::DeallocateAesKeySlot(slot));
            };
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::spl::LoadAesKey(slot, g_AccessKey, KeySource, sizeof(KeySource)));
            *pOut = {}; // nn::spl::ComputeCmac の対象サイズが sizeof(FileSign) より小さいので先に全領域をゼロ初期化
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::spl::ComputeCmac(pOut, sizeof(*pOut), slot, data, dataSize));
        });
    }

}

extern "C" void nninitStartup() NN_NOEXCEPT
{
    // 空実装にする
}

extern "C" void nndiagStartup() NN_NOEXCEPT
{
    // 空実装にする
}

extern "C" void nnMain() NN_NOEXCEPT
{
    g_Allocator.Initialize(g_HeapMemory, HeapSize);

    nn::capsrv::server::SetConfigsByFirmwareDebugSettings();

    NN_CAPSRV_LOG_INFO("Starting capsrv server\n");
    nn::os::SetThreadNamePointer(nn::os::GetCurrentThread(), NN_SYSTEM_THREAD_NAME(capsrv, Main));
    nn::os::ChangeThreadPriority(nn::os::GetCurrentThread(), NN_SYSTEM_THREAD_PRIORITY(capsrv, Main));

    nn::fs::InitializeWithMultiSessionForSystem();

    nn::capsrv::InitializeFileSystem();
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::Initialize());
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeDecoderControl());

    nn::spl::InitializeForCrypto();
    SetScreenShotFileSignFunction();
    SetMovieFileSignFunction();

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::server::InitializeForAlbumServer());
    CreateStartThreads();

    // ひたすらスレッドが終わるまで待つ。

    WaitDestroyThreads();
    nn::capsrv::server::FinalizeForAlbumServer();

    nn::capsrv::FinalizeDecoderControl();
    nn::spl::Finalize();

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::Finalize());
    nn::capsrv::FinalizeFileSystem();
    NN_CAPSRV_LOG_INFO("Exiting capsrv server\n");
}
