﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>

#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>

#include <nn/account/account_ApiPrivate.h>
#include <nn/account/account_ApiForAdministrators.h>
#include <nn/fs.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/nifm.h>
#include <nn/nifm/nifm_ApiForMenu.h>
#include <nn/nifm/nifm_ApiNetworkProfile.h>
#include <nn/os.h>
#include <nn/settings/system/settings_SystemApplication.h>
#include <nn/settings/system/settings_Eula.h>
#include <nn/settings/system/settings_Region.h>
#include <nn/time.h>

#include "DummyStarter_Processor.h"
#include "../../../../Eris/Sources/TargetTools/DevMenuCommand/DevMenuCommand_Option.h"
#include "../../../../Eris/Sources/TargetTools/DevMenuCommand/DevMenuCommand_ServiceDiscoveryCommand.h"

namespace
{
    // FS 用アロケータの定義
    const size_t g_HeapSize = 64 * 1024;
    NN_ALIGNAS(4096) char g_pHeap[g_HeapSize];
    nn::lmem::HeapHandle g_HeapHandle = NULL;

    void* Allocate(size_t size)
    {
        if(g_HeapHandle == NULL)
        {
            return NULL;
        }
        return nn::lmem::AllocateFromExpHeap(g_HeapHandle, size);
    }

    void Deallocate(void* p, size_t size)
    {
        NN_UNUSED(size);
        nn::lmem::FreeToExpHeap(g_HeapHandle, p);
    }

    // ネットワーク処理用ユーティリティ
    const int NETWORK_SETTING_TIMEOUT = 30; // seconds

    bool WaitRequestCompletion(nn::nifm::NetworkConnection* pNetworkConnection) NN_NOEXCEPT
    {
        int milliseconds = NETWORK_SETTING_TIMEOUT * 1000;
        bool isCompleted = pNetworkConnection->GetRequestState() != nn::nifm::RequestState_OnHold;

        while (milliseconds > 0 && !isCompleted)
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
            milliseconds -= 16;
            isCompleted = pNetworkConnection->GetRequestState() != nn::nifm::RequestState_OnHold;
        }

        if (!isCompleted)
        {
            pNetworkConnection->CancelRequest();
        }

        return isCompleted;
    }

}

bool Processor::ProcessEula()
{
    nn::settings::system::EulaVersion eula;

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::Initialize());

    eula.clockType = nn::settings::system::EulaVersionClockType_SteadyClock;
    eula.regionCode = nn::settings::system::RegionCode_Usa;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::StandardSteadyClock::GetCurrentTimePoint(&eula.steadyClock));
    eula.version = std::numeric_limits<uint32_t>::max();  // 最大値であればどんな EURA でも常に同意可能
    nn::settings::system::SetEulaVersions(&eula, 1);

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::Finalize());

    return true;
}

bool Processor::ProcessWifi()
{
    // imatake-aging
    const nn::nifm::WirelessSettingData wirelessSetting = {
        {  // ssidConfig
            {  // ssid
                13,  // length
                {0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x61,0x67,0x69,0x6e,0x67}  // hex
            },
            false  // nonBroadcast
        },
        {  // security
            {  //authEncryption
                nn::nifm::Authentication_Wpa2Psk,  // authentication
                nn::nifm::Encryption_Aes  // encryption
            },
            {  // sharedKey
                11,  // length
                "Shi2iTaiZen"  // keyMaterial
            }
        }
    };
    const nn::nifm::IpSettingData ipSetting = {
        {  // ip
            true,  // isAuto
            {},  // ipAddress
            {},  // subnetMask
            {}  // defaultGateway
        },
        {  // dns
            true,  // isAuto
            {},  // preferredDns
            {}  // alternateDns
        },
        {  // proxy
            false,  // isEnabled
            0,  // port
            "",  // proxy
            {  // authentication
                false,  // isEnabled
                "",  // username
                ""  // password
            }
        },
        1400  //mtu
    };
    const nn::nifm::NetworkProfileData networkProfileData = {
        nn::util::InvalidUuid,  // id
        "テンポラリ設定",  // name
        nn::nifm::NetworkProfileType_Temporary, // networkProfileType
        nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211,  // networkInterfaceType
        true,  // isAutoConnect
        true, // isLargeCapacity
        {
            wirelessSetting
        },
        ipSetting
    };
    nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(networkProfileData);
    nn::util::Uuid tempId = temporaryNetworkProfile.GetId();

    NN_ASSERT(nn::util::InvalidUuid != tempId);

    nn::nifm::NetworkConnection networkConnection;
    if (nn::nifm::SetRequestNetworkProfileId(networkConnection.GetRequestHandle(), tempId).IsFailure())
    {
        return false;
    }

    networkConnection.SubmitRequest();
    if (!WaitRequestCompletion(&networkConnection))
    {
        return false;
    }

    if (nn::nifm::PersistTemporaryNetworkProfile(temporaryNetworkProfile.GetHandle()).IsFailure())
    {
        return false;
    }

    return true;
}

bool Processor::ProcessServiceDiscovery()
{
    const int argc = 4;
    char* argv[argc];
    argv[0] = const_cast<char*>("");
    argv[1] = const_cast<char*>("servicediscovery");
    argv[2] = const_cast<char*>("import-all");
    argv[3] = const_cast<char*>("jd1env");

    Option option(argc, argv);

    bool success;
    if (ServiceDiscoveryCommand(&success, option).IsFailure() || !success)
    {
        return false;
    }

    return true;
}

bool Processor::ProcessAccount()
{
    // ROM のマウント
    size_t cacheSize = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::fs::QueryMountRomCacheSize( &cacheSize ) );
    void* pBuffer = Allocate(cacheSize);
    NN_ABORT_UNLESS(pBuffer != NULL);
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::fs::MountRom( "icon", pBuffer, cacheSize ) );

    nn::account::Uid user;
    nn::account::ProfileEditor profile;

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::BeginUserRegistration(&user));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetProfileEditor(&profile, user));

    // ニックネームの設定
    nn::account::Nickname nickname;
    char name[8] = "KAKIO";
    strncpy(nickname.name, name, 8);
    profile.SetNickname(nickname);

    // ユーザデータの設定
    const uint8_t DefaultUseData[nn::account::UserDataBytesMax] = {
#if defined(NN_BUILD_CONFIG_ENDIAN_LITTLE)
        0x01, 0x00, 0x00, 0x00, // version, mainImageType, padding(2),
        0x01, 0x00, 0x00, 0x00, // charId[LSB], charId>>8, charId>>16, charId[MSB]
        0x01, 0x00, 0x00, 0x01, // bgId[LSB], bgId>>8, bgId>>16, bgId[MSB]
#elif defined(NN_BUILD_CONFIG_ENDIAN_BIG)
        0x01, 0x00, 0x00, 0x00, // version, mainImageType, padding(2),
        0x00, 0x00, 0x00, 0x01, // charId[MSB], charId>>16, charId>>8, charId[LSB]
        0x01, 0x00, 0x00, 0x01, // bgId[MSB], bgId>>16, bgId>>8, bgId[LSB]
#else
    #error "Specified endian type is not supported."
#endif
    };
    profile.SetUserData(reinterpret_cast<const char*>(DefaultUseData), sizeof(DefaultUseData));

    // アイコン画像の設定
    nn::fs::FileHandle file;
    const char DefaultUserIconFilePath[] = "icon:/defaultIcon.jpg";
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&file, DefaultUserIconFilePath, nn::fs::OpenMode_Read));

    int64_t size;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::GetFileSize(&size, file));
    void* pBufferForIcon = Allocate(static_cast<size_t>(size));
    NN_SDK_ASSERT(pBufferForIcon != NULL);

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadFile(file, 0, pBufferForIcon, size));
    NN_ABORT_UNLESS_RESULT_SUCCESS(profile.FlushWithImage(pBufferForIcon, size));

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::CompleteUserRegistration(user));

    nn::fs::CloseFile(file);
    nn::fs::Unmount("icon");
    Deallocate(pBuffer, cacheSize);

    return true;
}

void Processor::Process() NN_NOEXCEPT
{
    nn::account::InitializeForAdministrator();

    g_HeapHandle = nn::lmem::CreateExpHeap(g_pHeap, g_HeapSize, nn::lmem::CreationOption_NoOption);
    NN_ABORT_UNLESS(g_HeapHandle != NULL);

    nn::fs::SetAllocator(Allocate, Deallocate);

    // 初回起動シーケンスを抜けているなら、何もしない
    {
        nn::settings::system::InitialLaunchSettings forGet = {};
        nn::settings::system::GetInitialLaunchSettings(&forGet);
        if (forGet.flags.Test<nn::settings::system::InitialLaunchFlag::IsCompleted>())
        {
            NN_LOG("[DummyStarter] 初回起動シーケンスは既に突破されています。\n");
            return;
        }
    }

    // EULA の同意
    SetStatus(Stage_ProcessEula, Status_Processing);
    if (!ProcessEula())
    {
        SetStatus(Stage_ProcessEula, Status_Failure);
        return;
    }
    SetStatus(Stage_ProcessEula, Status_Success);

    // Wifi の設定
#if defined(NN_DETAIL_DUMMY_STARTER_ENABLE_NETWORK_CONFIGURATION)
    SetStatus(Stage_ProcessWifi, Status_Processing);
    if (!ProcessWifi())
    {
        SetStatus(Stage_ProcessWifi, Status_Failure);
        return;
    }
    SetStatus(Stage_ProcessWifi, Status_Success);
#else
    SetStatus(Stage_ProcessWifi, Status_Skipped);
#endif

    // サービスディスカバリの設定
#if defined(NN_DETAIL_DUMMY_STARTER_ENABLE_NETWORK_CONFIGURATION)
    SetStatus(Stage_ProcessServiceDiscovery, Status_Processing);
    if (!ProcessServiceDiscovery())
    {
        SetStatus(Stage_ProcessServiceDiscovery, Status_Failure);
        return;
    }
    SetStatus(Stage_ProcessServiceDiscovery, Status_Success);
#else
    SetStatus(Stage_ProcessServiceDiscovery, Status_Skipped);
#endif

    // アカウントの設定
    SetStatus(Stage_ProcessAccount, Status_Processing);
    if (!ProcessAccount())
    {
        SetStatus(Stage_ProcessAccount, Status_Failure);
        return;
    }
    SetStatus(Stage_ProcessAccount, Status_Success);

    // 初回起動シーケンスを抜けたことにする
    {
        nn::settings::system::InitialLaunchSettings forSet = {};
        forSet.flags.Set<nn::settings::system::InitialLaunchFlag::IsCompleted>(true);
        nn::settings::system::SetInitialLaunchSettings(forSet);
    }

    nn::lmem::DestroyExpHeap(g_HeapHandle);
    nn::account::Finalize();
}
