﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <vector>
#include <nn/account.h>
#include <nn/nifm.h>
#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/nifm/nifm_ApiForTest.h>
#include <nn/npns/npns_ApiSystem.h>
#include <nn/oe/oe_IdlePolicyApi.h>
#include <nn/settings/system/settings_ErrorReport.h>
#include <nn/settings/system/settings_Sleep.h>
#include <nn/settings/system/settings_SystemApplication.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/rid/rid_Result.h>
#include "detail/rid_Settings.h"

namespace nn { namespace rid {

    namespace {
        Result DeleteAllUserAccountSaveData() NN_NOEXCEPT
        {
            std::vector<fs::SaveDataId> deleteList;
            std::unique_ptr<fs::SaveDataIterator> iter;
            NN_RESULT_DO(fs::OpenSaveDataIterator(&iter, fs::SaveDataSpaceId::User));

            while (NN_STATIC_CONDITION(true))
            {
                fs::SaveDataInfo info;
                int64_t count;
                NN_RESULT_DO(iter->ReadSaveDataInfo(&count, &info, 1));
                if (count == 0)
                {
                    break;
                }

                // アプリケーション(試遊台メニュー以外)のセーブデータを削除する
                if (info.applicationId != GetMenuApplicationId())
                {
                    deleteList.push_back(info.saveDataId);
                }
            }

            for (int i = 0; i < deleteList.size(); i++)
            {
                NN_RESULT_DO(fs::DeleteSaveData(deleteList[i]));
            }

            NN_RESULT_SUCCESS;
        }
    }

    Result CheckRetailInteractiveDisplayDevice() NN_NOEXCEPT
    {
        bool isQuest = settings::system::GetQuestFlag();
        NN_RESULT_THROW_UNLESS(isQuest, ResultNotRetailInteractiveDisplayDevice());

        NN_RESULT_SUCCESS;
    }

    Result MinimumStartUp() NN_NOEXCEPT
    {
        // 通知の無効化
        NN_RESULT_DO(npns::Suspend());

        // ネットワーク独占利用要求の取り下げ
        NN_RESULT_DO(nifm::SetExclusiveClient({ nifm::InvalidClientIdValue }));

        NN_RESULT_SUCCESS;
    }

    Result StartUp() NN_NOEXCEPT
    {
        // 画面焼け軽減無効設定
        oe::SetUserInactivityDetectionTimeExtendedUnsafe(true);

        // 自動スリープ無効設定
        settings::system::SleepFlagSet flag = {};
        settings::system::SleepSettings settings = { flag, nn::settings::system::HandheldSleepPlan_Never, nn::settings::system::ConsoleSleepPlan_Never };
        settings::system::SetSleepSettings(settings);

        // エラーレポート同意設定
        settings::system::SetErrorReportSharePermission(settings::system::ErrorReportSharePermission_Granted);

        // 全タイトルのセーブデータ削除
        NN_RESULT_DO(DeleteAllUserAccountSaveData());

        // アカウントの存在確認
        int userCount;
        NN_RESULT_DO(account::GetUserCount(&userCount));
        NN_RESULT_THROW_UNLESS(userCount > 0, ResultAccountNotFound());

        NN_RESULT_SUCCESS;
    }
}}
