﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <vector>
#include <nn/fs.h>
#include <nn/fs/fs_ImageDirectory.h>
#include <nn/ns/ns_ApplicationEntityApi.h>
#include <nn/ns/ns_ApplicationEntitySystemApi.h>
#include <nn/ns/ns_ApplicationRecordApi.h>
#include <nn/ns/ns_ApplicationVerificationApi.h>
#include <nn/ns/ns_Result.h>
#include <nn/mii/mii_PrivateDatabase.h>
#include <nn/mii/mii_Special.h>
#include <nn/mii/mii_StoreDataAccessor.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/rid/rid_PrepareApi.h>
#include <nn/rid/rid_Result.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>
#include "detail/rid_Settings.h"

namespace nn { namespace rid {

    namespace {
        Result DeleteAllAlbumsImpl(fs::ImageDirectoryId id) NN_NOEXCEPT
        {
            NN_RESULT_DO(fs::MountImageDirectory("image", id));
            NN_UTIL_SCOPE_EXIT{ fs::Unmount("image"); };

            const char* rootPath = "image:/";
            fs::DirectoryHandle handle;
            NN_RESULT_DO(fs::OpenDirectory(&handle, rootPath, fs::OpenDirectoryMode_All));
            NN_UTIL_SCOPE_EXIT{ fs::CloseDirectory(handle); };

            int64_t count;
            NN_RESULT_DO(fs::GetDirectoryEntryCount(&count, handle));
            if (count == 0)
            {
                NN_RESULT_SUCCESS;
            }

            std::vector<fs::DirectoryEntry> entryList;
            entryList.resize(static_cast<size_t>(count));

            int64_t tmpCount;
            NN_RESULT_DO(fs::ReadDirectory(&tmpCount, entryList.data(), handle, static_cast<int64_t>(entryList.size())));

            for (auto& e : entryList)
            {
                char path[128];
                util::SNPrintf(path, sizeof(path), "%s/%s", rootPath, e.name);
                if (e.directoryEntryType == fs::DirectoryEntryType_Directory)
                {
                    NN_RESULT_DO(fs::DeleteDirectoryRecursively(path));
                }
                else
                {
                    NN_RESULT_DO(fs::DeleteFile(path));
                }
            }

            NN_RESULT_SUCCESS;
        }
    }

    Result CheckContentsIntegrity() NN_NOEXCEPT
    {
        const size_t VerifyWorkBufferAlign = 4 * 1024;
        const size_t VerifyWorkBufferSize = 4 * 1024 * 1024;
        NN_ALIGNAS(VerifyWorkBufferAlign) static uint8_t buffer[VerifyWorkBufferSize];

        int offset = 0;

        while (NN_STATIC_CONDITION(true))
        {
            ns::ApplicationRecord records[16];
            int count = ns::ListApplicationRecord(records, 16, offset);
            if (count <= 0)
            {
                break;
            }

            for (int i = 0; i < count; i++)
            {
                // 試遊台の ApplicationId の場合は追加コンテンツのみ破損チェックする
                if (records[i].id == GetMenuApplicationId())
                {
                    ns::AsyncVerifyApplicationResult async;
                    NN_RESULT_DO(ns::RequestVerifyApplication(&async, records[i].id, ns::VerifyContentFlag_AddOnContent::Mask, buffer, VerifyWorkBufferSize));
                    NN_RESULT_TRY(async.Get())
                       NN_RESULT_CATCH(ns::ResultVerificationFailed)
                       {
                           NN_RESULT_TRY(ns::DeleteApplicationContentEntities(records[i].id, ns::ApplicationEntityFlag_AddOnContent::Mask | ns::ApplicationEntityFlag_SkipRunningCheck::Mask, ncm::StorageId::Any))
                               NN_RESULT_CATCH_ALL { NN_RESULT_THROW(ResultNandOrSdCardMaybeCorrupted()); }
                           NN_RESULT_END_TRY
                       }
                       NN_RESULT_CATCH(ns::ResultSdCardDataCorrupted)
                       {
                           NN_RESULT_TRY(ns::DeleteApplicationContentEntities(records[i].id, ns::ApplicationEntityFlag_AddOnContent::Mask | ns::ApplicationEntityFlag_SkipRunningCheck::Mask, ncm::StorageId::Any))
                               NN_RESULT_CATCH_ALL { NN_RESULT_THROW(ResultSdCardMaybeCorrupted()); }
                           NN_RESULT_END_TRY
                       }
                    NN_RESULT_END_TRY
                }
                // 試遊アプリケーションの ApplicationId の場合は全てのコンテンツをチェックする
                else
                {
                    ns::AsyncVerifyApplicationResult async;
                    NN_RESULT_DO(ns::RequestVerifyApplication(&async, records[i].id, buffer, VerifyWorkBufferSize));
                    NN_RESULT_TRY(async.Get())
                        NN_RESULT_CATCH(ns::ResultVerificationFailed)
                    {
                        NN_RESULT_TRY(ns::DeleteApplicationEntity(records[i].id))
                            NN_RESULT_CATCH_ALL {
                            NN_RESULT_THROW(ResultNandOrSdCardMaybeCorrupted());
                        }
                        NN_RESULT_END_TRY
                    }
                    NN_RESULT_CATCH(ns::ResultSdCardDataCorrupted)
                    {
                        NN_RESULT_TRY(ns::DeleteApplicationEntity(records[i].id))
                            NN_RESULT_CATCH_ALL {
                            NN_RESULT_THROW(ResultSdCardMaybeCorrupted());
                        }
                        NN_RESULT_END_TRY
                    }
                    NN_RESULT_END_TRY
                }
            }

            offset += count;
        }

        NN_RESULT_SUCCESS;
    }


    Result DeleteAllMii() NN_NOEXCEPT
    {
        // スペシャル mii の有効化
        mii::SetSpecialMiiKeyCode(mii::SpecialMiiKeyCode);

        mii::PrivateDatabase database;
        NN_RESULT_DO(database.Initialize());
        NN_UTIL_SCOPE_EXIT{ database.Finalize(); };

        int miiCount = database.GetCount(mii::SourceFlag_Database);

        if (miiCount > 0)
        {
            std::unique_ptr<mii::StoreData> miiList(new mii::StoreData[mii::DatabaseMiiCount]);
            int tmpCount;
            NN_RESULT_DO(database.Get(&tmpCount, miiList.get(), miiCount, mii::SourceFlag_Database));

            for (int i = 0; i < miiCount; i++)
            {
                mii::StoreDataAccessor accessor(miiList.get() + i);
                mii::CreateId createId = accessor.GetCreateId();

                // mii を削除
                NN_RESULT_DO(database.Delete(createId));
            }
        }

        NN_RESULT_SUCCESS;
    }

    Result DeleteAllAlbums() NN_NOEXCEPT
    {
        // Nand のアルバムを削除
        NN_RESULT_DO(DeleteAllAlbumsImpl(fs::ImageDirectoryId::Nand));

        // SDCard が使用できる状態であれば SDCard のアルバムを削除
        if (ns::CheckSdCardMountStatus().IsSuccess())
        {
            NN_RESULT_DO(DeleteAllAlbumsImpl(fs::ImageDirectoryId::SdCard));
        }

        NN_RESULT_SUCCESS;
    }
}}
