﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkLog.h>
#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/system/settings_Network.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_Uuid.h>
#include <nn/nifm/nifm_TypesNetworkProfile.h>
#include <sstream>
#include <string>
#include <vector>

#define RAPIDJSON_NO_INT64DEFINE
#define RAPIDJSON_ASSERT(x)             NN_SDK_ASSERT(x)
#define RAPIDJSON_HAS_CXX11_TYPETRAITS  1 //NOLINT(preprocessor/const)
#define RAPIDJSON_HAS_STDSTRING         1 //NOLINT(preprocessor/const)

#if defined(NN_BUILD_CONFIG_OS_WIN)
#pragma warning(push)
#pragma warning(disable : 4244)
#pragma warning(disable : 4668)
#endif
#include "rapidjson/document.h"
#if defined(NN_BUILD_CONFIG_OS_WIN)
#pragma warning(pop)
#endif

namespace nn {
    namespace repair {
        namespace {
            //!< IEEE 802.11 を表す NIC 種別
            const char* const NicTypeIeee80211 = "Ieee80211";

            //!< Ethernet を表す NIC 種別
            const char* const NicTypeEthernet = "Ethernet";

            //!< オープン認証を表す認証方式
            const char* const AuthenticationModeOpen = "Open";

            //!< 共有キー認証を表す認証方式
            const char* const AuthenticationModeShared = "Shared";

            //!< WPA-PSK 認証を表す認証方式
            const char* const AuthenticationModeWpaPsk = "WpaPsk";

            //!< WPA2-PSK 認証を表す認証方式
            const char* const AuthenticationModeWpa2Psk = "Wpa2Psk";

            //!< 暗号化無しを表す暗号化方式
            const char* const EncryptionModeNone = "None";

            //!< WEP 暗号化方式を表す暗号化方式
            const char* const EncryptionModeWep = "Wep";

            //!< AES 暗号化方式を表す暗号化方式
            const char* const EncryptionModeAes = "Aes";


            std::vector<std::string> Split(const std::string str, char delim) NN_NOEXCEPT
            {
                std::vector<std::string> results;
                std::stringstream ss(str);
                std::string item;
                while (std::getline(ss, item, delim))
                {
                    if (item.empty())
                    {
                        break;
                    }

                    results.push_back(item);
                }

                return results;
            }

            //!< IP アドレスをパースします。
            template<typename T>
            bool ParseIpAddress(T(&outBuffer)[4], const ::std::string& address) NN_NOEXCEPT
            {
                std::vector<std::string> numbers = Split(address, '.');

                if (numbers.size() != 4)
                {
                    return false;
                }

                for (const std::string& number : numbers)
                {
                    if (number.length() == 0)
                    {
                        return false;
                    }
                }

                T buffer[4] = {};

                for (int i = 0; i < 4; ++i)
                {
                    for (const char& value : numbers[i])
                    {
                        if (value < '0' || '9' < value)
                        {
                            return false;
                        }
                        else
                        {
                            buffer[i] = buffer[i] * 10 + (value - '0');
                        }
                    }
                }

                for (int i = 0; i < 4; ++i)
                {
                    outBuffer[i] = buffer[i];
                }

                return true;
            }

            bool SetNetWorkSettingsName(nn::nifm::NetworkProfileData* pSettings,
                const rapidjson::Value& value) NN_NOEXCEPT
            {
                std::string name = value.GetString();
                const int length = sizeof(pSettings->name);

                if (nn::util::Strlcpy(pSettings->name, name.c_str(), length) >= length)
                {
                    return false;
                }

                return true;
            }

            bool SetNetworkSettingsId(nn::nifm::NetworkProfileData* pSettings,
                const rapidjson::Value& value) NN_NOEXCEPT
            {
                pSettings->id.FromString(value.GetString());

                return true;
            }

            bool SetNetworkSettingsAutoConnection(nn::nifm::NetworkProfileData* pSettings,
                const rapidjson::Value& value) NN_NOEXCEPT
            {
                pSettings->isAutoConnect = value.GetBool();

                return true;
            }

            bool SetNetworkSettingsLargeCapacity(nn::nifm::NetworkProfileData* pSettings,
                const rapidjson::Value& value) NN_NOEXCEPT
            {
                pSettings->isLargeCapacity = value.GetBool();

                return true;
            }

            bool SetNetworkSettingsNicType(nn::nifm::NetworkProfileData* pSettings,
                const rapidjson::Value& value) NN_NOEXCEPT
            {
                std::string nicType = value.GetString();

                if (nicType == NicTypeIeee80211)
                {
                    pSettings->networkInterfaceType = nn::nifm::NetworkInterfaceType_Ieee80211;

                    return true;
                }

                if (nicType == NicTypeEthernet)
                {
                    pSettings->networkInterfaceType = nn::nifm::NetworkInterfaceType_Ethernet;

                    return true;
                }

                return false;
            }

            bool SetSsidSettingsName(nn::nifm::WirelessSettingData* pSettings,
                const rapidjson::Value& value) NN_NOEXCEPT
            {
                std::string name = value.GetString();
                const int length = static_cast<int>(name.length());

                if (length <= nn::nifm::Ssid::HexSize)
                {
                    std::copy(name.begin(),
                        name.end(),
                        reinterpret_cast<char*>(pSettings->ssidConfig.ssid.hex));

                    pSettings->ssidConfig.ssid.length = static_cast<uint8_t>(length);

                    return true;
                }

                return false;
            }

            bool SetSsidSettingsNonBroadcast(nn::nifm::WirelessSettingData* pSettings,
                const rapidjson::Value& value) NN_NOEXCEPT
            {
                pSettings->ssidConfig.nonBroadcast = value.GetBool();

                return true;
            }

            bool SetNetworkSettingsSsidConfig(nn::nifm::NetworkProfileData* pSettings,
                const rapidjson::Value& value) NN_NOEXCEPT
            {
                for (rapidjson::Value::ConstMemberIterator itr = value.MemberBegin();
                    itr != value.MemberEnd(); itr++)
                {
                    if (itr->name == "name")
                    {
                        SetSsidSettingsName(&pSettings->wirelessSetting, itr->value);
                    }
                    else if (itr->name == "non_broadcast")
                    {
                        SetSsidSettingsNonBroadcast(&pSettings->wirelessSetting, itr->value);
                    }
                    else
                    {
                        return false;
                    }
                }

                return true;
            }

            bool SetSecuritySettingsAuthentication(nn::nifm::WirelessSettingData* pSettings,
                const rapidjson::Value& value) NN_NOEXCEPT
            {
                std::string authenticationMode = value.GetString();

                if (authenticationMode == AuthenticationModeOpen)
                {
                    pSettings->security.authEncryption.authentication = nn::nifm::Authentication_Open;

                    return true;
                }

                if (authenticationMode == AuthenticationModeShared)
                {
                    pSettings->security.authEncryption.authentication = nn::nifm::Authentication_Shared;

                    return true;
                }

                if (authenticationMode == AuthenticationModeWpaPsk)
                {
                    pSettings->security.authEncryption.authentication = nn::nifm::Authentication_WpaPsk;

                    return true;
                }

                if (authenticationMode == AuthenticationModeWpa2Psk)
                {
                    pSettings->security.authEncryption.authentication = nn::nifm::Authentication_Wpa2Psk;

                    return true;
                }

                return false;
            }

            bool SetSecuritySettingsEncryption(nn::nifm::WirelessSettingData* pSettings,
                const rapidjson::Value& value) NN_NOEXCEPT
            {
                ::std::string encryptionMode = value.GetString();

                if (encryptionMode == EncryptionModeNone)
                {
                    pSettings->security.authEncryption.encryption = nn::nifm::Encryption_None;

                    return true;
                }

                if (encryptionMode == EncryptionModeWep)
                {
                    pSettings->security.authEncryption.encryption = nn::nifm::Encryption_Wep;

                    return true;
                }

                if (encryptionMode == EncryptionModeAes)
                {
                    pSettings->security.authEncryption.encryption = nn::nifm::Encryption_Aes;

                    return true;
                }

                return false;
            }

            bool SetSecuritySettingsKeyMaterial(nn::nifm::WirelessSettingData* pSettings,
                const rapidjson::Value& value) NN_NOEXCEPT
            {
                std::string keyMaterial = value.GetString();
                const int length = static_cast<int>(keyMaterial.length());

                if (length <= sizeof(pSettings->security.sharedKey.keyMaterial))
                {
                    for (char& c : pSettings->security.sharedKey.keyMaterial)
                    {
                        c = '\0';
                    }

                    std::copy(
                        keyMaterial.begin(), keyMaterial.end(), pSettings->security.sharedKey.keyMaterial);

                    pSettings->security.sharedKey.length = static_cast<uint8_t>(length);

                    return true;
                }

                return false;
            }


            bool SetNetworkSettingsSecurity(nn::nifm::NetworkProfileData* pSettings,
                const rapidjson::Value& value) NN_NOEXCEPT
            {
                bool valid = true;

                for (rapidjson::Value::ConstMemberIterator itr = value.MemberBegin();
                    itr != value.MemberEnd(); itr++)
                {
                    if (itr->name == "authentication")
                    {
                        valid &= SetSecuritySettingsAuthentication(&pSettings->wirelessSetting, itr->value);
                    }
                    else if (itr->name == "encryption")
                    {
                        valid &= SetSecuritySettingsEncryption(&pSettings->wirelessSetting, itr->value);
                    }
                    else if (itr->name == "key_material")
                    {
                        valid &= SetSecuritySettingsKeyMaterial(&pSettings->wirelessSetting, itr->value);
                    }
                    else
                    {
                        valid = false;
                    }
                }

                return valid;
            }

            bool SetIpSettingsAutoIp(nn::nifm::IpSettingData* pSettings,
                const rapidjson::Value& value) NN_NOEXCEPT
            {
                pSettings->ip.isAuto = value.GetBool();

                return true;
            }

            bool SetIpSettingsIpAddress(nn::nifm::IpSettingData* pSettings,
                const rapidjson::Value& value) NN_NOEXCEPT
            {
                std::string address = value.GetString();

                if (ParseIpAddress(pSettings->ip.ipAddress.data, address))
                {
                    return true;
                }

                return false;
            }

            bool SetIpSettingsSubnetMask(nn::nifm::IpSettingData* pSettings,
                const rapidjson::Value& value) NN_NOEXCEPT
            {
                std::string subnetMask = value.GetString();

                if (ParseIpAddress(pSettings->ip.subnetMask.data, subnetMask))
                {
                    return true;
                }

                return false;
            }

            bool SetIpSettingsAutoDns(nn::nifm::IpSettingData* pSettings,
                const rapidjson::Value& value) NN_NOEXCEPT
            {
                pSettings->dns.isAuto = value.GetBool();

                return true;
            }

            bool SetIpSettingsDefaultGateway(nn::nifm::IpSettingData* pSettings,
                const rapidjson::Value& value) NN_NOEXCEPT
            {
                std::string defaultGateway = value.GetString();

                if (ParseIpAddress(pSettings->ip.defaultGateway.data, defaultGateway))
                {
                    return true;
                }

                return false;
            }

            bool SetIpSettingsPreferredDns(nn::nifm::IpSettingData* pSettings,
                const rapidjson::Value& value) NN_NOEXCEPT
            {
                std::string preferredDns = value.GetString();

                if (ParseIpAddress(pSettings->dns.preferredDns.data, preferredDns))
                {
                    return true;
                }

                return false;
            }

            bool SetIpSettingsAlternateDns(nn::nifm::IpSettingData* pSettings,
                const rapidjson::Value& value) NN_NOEXCEPT

            {
                std::string alternateDns = value.GetString();

                if (ParseIpAddress(pSettings->dns.alternateDns.data, alternateDns))
                {
                    return true;
                }

                return false;
            }

            bool SetNetworkSettingsIpSettings(nn::nifm::NetworkProfileData* pSettings,
                const rapidjson::Value& value) NN_NOEXCEPT
            {
                bool valid = true;

                for (rapidjson::Value::ConstMemberIterator itr = value.MemberBegin();
                    itr != value.MemberEnd(); itr++)
                {
                    if (itr->name == "auto_ip")
                    {
                        valid &= SetIpSettingsAutoIp(&pSettings->ipSetting, itr->value);
                    }
                    else if (itr->name == "address")
                    {
                        valid &= SetIpSettingsIpAddress(&pSettings->ipSetting, itr->value);
                    }
                    else if (itr->name == "subnet_mask")
                    {
                        valid &= SetIpSettingsSubnetMask(&pSettings->ipSetting, itr->value);
                    }
                    else if (itr->name == "default_gateway")
                    {
                        valid &= SetIpSettingsDefaultGateway(&pSettings->ipSetting, itr->value);
                    }
                    else if (itr->name == "auto_dns")
                    {
                        valid &= SetIpSettingsAutoDns(&pSettings->ipSetting, itr->value);
                    }
                    else if (itr->name == "preferred_dns")
                    {
                        valid &= SetIpSettingsPreferredDns(&pSettings->ipSetting, itr->value);
                    }
                    else if (itr->name == "alternate_dns")
                    {
                        valid &= SetIpSettingsAlternateDns(&pSettings->ipSetting, itr->value);
                    }
                    else
                    {
                        valid = false;
                    }
                }

                return valid;
            }

            bool SetProxySettingsEnables(nn::nifm::IpSettingData* pSettings,
                const rapidjson::Value& value) NN_NOEXCEPT
            {
                pSettings->proxy.isEnabled = value.GetBool();

                return true;
            }

            bool SetProxySettingsHostName(nn::nifm::IpSettingData* pSettings,
                const rapidjson::Value& value) NN_NOEXCEPT
            {
                std::string hostName = value.GetString();

                for (char& c : pSettings->proxy.proxy)
                {
                    c = '\0';
                }

                std::copy(hostName.begin(), hostName.end(), pSettings->proxy.proxy);

                return true;
            }

            bool SetProxySettingsPort(nn::nifm::IpSettingData* pSettings,
                const rapidjson::Value& value) NN_NOEXCEPT
            {
                int port = value.GetInt();

                pSettings->proxy.port = static_cast<uint16_t>(port);

                return true;
            }

            bool SetProxySettingsAuthenticates(nn::nifm::IpSettingData* pSettings,
                const rapidjson::Value& value) NN_NOEXCEPT
            {
                bool authenticates = value.GetBool();

                pSettings->proxy.authentication.isEnabled = authenticates;

                return true;
            }

            bool SetProxySettingsUserName(nn::nifm::IpSettingData* pSettings,
                const rapidjson::Value& value) NN_NOEXCEPT
            {
                std::string userName = value.GetString();

                for (char& c : pSettings->proxy.authentication.username)
                {
                    c = '\0';
                }

                std::copy(userName.begin(), userName.end(), pSettings->proxy.authentication.username);

                return true;
            }

            bool SetProxySettingsPassword(nn::nifm::IpSettingData* pSettings,
                const rapidjson::Value& value) NN_NOEXCEPT
            {
                std::string password = value.GetString();

                for (char& c : pSettings->proxy.authentication.password)
                {
                    c = '\0';
                }

                std::copy(password.begin(), password.end(), pSettings->proxy.authentication.password);

                return true;
            }

            bool SetNetworkSettingsProxy(nn::nifm::NetworkProfileData* pSettings,
                const rapidjson::Value& value) NN_NOEXCEPT
            {
                bool valid = true;

                for (rapidjson::Value::ConstMemberIterator itr = value.MemberBegin();
                    itr != value.MemberEnd(); itr++)
                {
                    if (itr->name == "enables")
                    {
                        valid &= SetProxySettingsEnables(&pSettings->ipSetting, itr->value);
                    }
                    else if (itr->name == "hostname")
                    {
                        valid &= SetProxySettingsHostName(&pSettings->ipSetting, itr->value);
                    }
                    else if (itr->name == "port")
                    {
                        valid &= SetProxySettingsPort(&pSettings->ipSetting, itr->value);
                    }
                    else if (itr->name == "authenticates")
                    {
                        valid &= SetProxySettingsAuthenticates(&pSettings->ipSetting, itr->value);
                    }
                    else if (itr->name == "username")
                    {
                        valid &= SetProxySettingsUserName(&pSettings->ipSetting, itr->value);
                    }
                    else if (itr->name == "password")
                    {
                        valid &= SetProxySettingsPassword(&pSettings->ipSetting, itr->value);
                    }
                    else
                    {
                        valid = false;
                    }
                }

                return valid;
            }

            bool SetNetworkSettingsMtu(nn::nifm::NetworkProfileData* pSettings,
                const rapidjson::Value& value) NN_NOEXCEPT
            {
                int mtu = value.GetInt();

                pSettings->ipSetting.mtu = static_cast<uint16_t>(mtu);

                return true;
            }

            bool SetNetworkSettings(nn::nifm::NetworkProfileData* pSettings,
                const rapidjson::Value& value) NN_NOEXCEPT
            {
                bool valid = true;
                for (rapidjson::Value::ConstMemberIterator itr = value.MemberBegin();
                    itr != value.MemberEnd(); itr++)
                {
                    if (itr->name == "name")
                    {
                        valid &= SetNetWorkSettingsName(pSettings, itr->value);
                    }
                    else if (itr->name == "id")
                    {
                        valid &= SetNetworkSettingsId(pSettings, itr->value);
                    }
                    else if (itr->name == "auto_connection")
                    {
                        valid &= SetNetworkSettingsAutoConnection(pSettings, itr->value);
                    }
                    else if (itr->name == "large_capacity")
                    {
                        valid &= SetNetworkSettingsLargeCapacity(pSettings, itr->value);
                    }
                    else if (itr->name == "nic_type")
                    {
                        valid &= SetNetworkSettingsNicType(pSettings, itr->value);
                    }
                    else if (itr->name == "ssid_config")
                    {
                        valid &= SetNetworkSettingsSsidConfig(pSettings, itr->value);
                    }
                    else if (itr->name == "security")
                    {
                        valid &= SetNetworkSettingsSecurity(pSettings, itr->value);
                    }
                    else if (itr->name == "ip_settings")
                    {
                        valid &= SetNetworkSettingsIpSettings(pSettings, itr->value);
                    }
                    else if (itr->name == "proxy")
                    {
                        valid &= SetNetworkSettingsProxy(pSettings, itr->value);
                    }
                    else if (itr->name == "mtu")
                    {
                        valid &= SetNetworkSettingsMtu(pSettings, itr->value);
                    }
                    else
                    {
                        valid = false;
                    }
                }

                return valid;
            }
        }

        bool ImportNetworkSettings(nn::nifm::NetworkProfileData* pOutNetworkSettings, const char* json) NN_NOEXCEPT
        {
            rapidjson::Document document;
            document.Parse(json);

            if (document.HasParseError())
            {
                NN_SDK_LOG("parse error\n");
                return false;
            }

            pOutNetworkSettings->networkProfileType = nn::nifm::NetworkProfileType_Temporary;

            const rapidjson::Value& value = document["network_settings"];
            if (!SetNetworkSettings(pOutNetworkSettings, value))
            {
                NN_SDK_LOG("Invaild format\n");
                return false;
            }

            return true;
        }
    }
}
