﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/repair/repair_ProtectedFile.h>
#include <algorithm>

namespace {

}

namespace nn
{
namespace repair
{
    class TestProtectedFileEncryptor : public IProtectedFileEncryptor
    {
    public:
        TestProtectedFileEncryptor();
        virtual ~TestProtectedFileEncryptor() {}
        virtual nn::Result ProcessDataBlock(BlockHeader *pOutHeader, void * pOut, int64_t index, const void * buffer, size_t size) override;
        virtual nn::Result CreateNewBlock(void * pOut, int64_t index, size_t size) override;
        virtual nn::Result DeProcessDataBlock(void * pOut, const BlockHeader & header, int64_t index, const void * buffer, size_t size) override;
        virtual nn::Result CalculateCmac(Cmac * pOut, void * buffer, size_t size) override;
        virtual nn::Result LoadKey(Key128 encrpytedKey) override;
        virtual nn::Result GenerateEncryptedKey(Key128 * pOut) override;

    private:
        void FillRandomBytes(void *buffer, size_t size);

        std::mt19937 m_Random;
        std::uniform_int_distribution<uint32_t> m_RandomByteGeneartor;
        Key128 m_AesKey;
        Key128 m_FixedAesKey;
    };

    nn::Result CreateFixedKeyProtectedFileEncryptor(std::shared_ptr<IProtectedFileEncryptor>* pOut)
    {
        pOut->reset(new TestProtectedFileEncryptor());
        NN_RESULT_SUCCESS;
    }

    TestProtectedFileEncryptor::TestProtectedFileEncryptor()
        : m_Random(1234),
        m_RandomByteGeneartor(std::numeric_limits<uint8_t>::min(), std::numeric_limits<uint8_t>::max())
    {
        FillRandomBytes(m_FixedAesKey.data, m_FixedAesKey.KEY_SIZE);
        m_AesKey = Key128::MakeZero();
    }

    nn::Result TestProtectedFileEncryptor::ProcessDataBlock(BlockHeader * pOutHeader, void * pOut, int64_t index, const void * buffer, size_t size)
    {
        nn::crypto::Aes128CtrEncryptor encryptor;

        InitialVector iv;
        FillRandomBytes(iv.data, iv.IV_SIZE);

        encryptor.Initialize(m_AesKey.data, m_AesKey.KEY_SIZE, iv.data, iv.IV_SIZE);
        encryptor.Update(pOut, size, buffer, size);

        nn::crypto::Aes128CmacGenerator cmacGenerator;
        cmacGenerator.Initialize(m_AesKey.data, m_AesKey.KEY_SIZE);
        cmacGenerator.Update(&index, sizeof(index));
        cmacGenerator.Update(iv.data, iv.IV_SIZE);
        cmacGenerator.Update(pOut, size);

        Cmac cmac;
        cmacGenerator.GetMac(cmac.data, cmac.CMAC_SIZE);

        pOutHeader->initialVector = iv;
        pOutHeader->cmac = cmac;

        NN_RESULT_SUCCESS;
    }

    nn::Result TestProtectedFileEncryptor::CreateNewBlock(void * pOut, int64_t index, size_t size)
    {
        NN_UNUSED(index);

        FillRandomBytes(pOut, size);

        NN_RESULT_SUCCESS;
    }

    nn::Result TestProtectedFileEncryptor::DeProcessDataBlock(void * pOut, const BlockHeader & header, int64_t index, const void * buffer, size_t size)
    {
        nn::crypto::Aes128CmacGenerator cmacGenerator;
        cmacGenerator.Initialize(m_AesKey.data, m_AesKey.KEY_SIZE);
        cmacGenerator.Update(&index, sizeof(index));
        cmacGenerator.Update(header.initialVector.data, header.initialVector.IV_SIZE);
        cmacGenerator.Update(buffer, size);

        Cmac cmac;
        cmacGenerator.GetMac(cmac.data, cmac.CMAC_SIZE);

        if (cmac != header.cmac)
        {
            NN_RESULT_THROW(nn::fs::ResultDataCorrupted());
        }

        nn::crypto::Aes128CtrDecryptor decryptor;

        decryptor.Initialize(m_AesKey.data, m_AesKey.KEY_SIZE, header.initialVector.data, header.initialVector.IV_SIZE);
        decryptor.Update(pOut, size, buffer, size);

        NN_RESULT_SUCCESS;
    }

    nn::Result TestProtectedFileEncryptor::CalculateCmac(Cmac * pOut, void * buffer, size_t size)
    {
        nn::crypto::Aes128CmacGenerator cmacGenerator;
        cmacGenerator.Initialize(m_AesKey.data, m_AesKey.KEY_SIZE);
        cmacGenerator.Update(buffer, size);

        cmacGenerator.GetMac(pOut, pOut->CMAC_SIZE);

        NN_RESULT_SUCCESS;
    }

    nn::Result TestProtectedFileEncryptor::LoadKey(Key128 encrpytedKey)
    {
        nn::crypto::Aes128CtrDecryptor decryptor;
        InitialVector iv = {};
        decryptor.Initialize(m_FixedAesKey.data, m_FixedAesKey.KEY_SIZE, iv.data, iv.IV_SIZE);
        decryptor.Update(m_AesKey.data, m_AesKey.KEY_SIZE, encrpytedKey.data, encrpytedKey.KEY_SIZE);

        NN_RESULT_SUCCESS;
    }

    nn::Result TestProtectedFileEncryptor::GenerateEncryptedKey(Key128 * pOut)
    {
        FillRandomBytes(pOut, pOut->KEY_SIZE);

        NN_RESULT_SUCCESS;
    }

    void TestProtectedFileEncryptor::FillRandomBytes(void * buffer, size_t size)
    {
        uint8_t *byteBuffer = reinterpret_cast<uint8_t*>(buffer);
        for (size_t i = 0; i < size; i++)
        {
            byteBuffer[i] = static_cast<uint8_t>(m_RandomByteGeneartor(m_Random));
        }
    }
}
}
