﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/repair/repair_CryptUtility.h>
#include <nn/nn_Abort.h>
#include <nn/util/util_BitUtil.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fs_Base.h>
#include <nn/utilTool/utilTool_CommandLog.h>
#include <nn/crypto/crypto_Aes128CmacGenerator.h>
#include <nn/crypto/crypto_Sha256Generator.h>
#include <algorithm>
#include <memory>
#include <cstring>
#include <nn/crypto/crypto_Compare.h>

namespace
{
    bool IsHexChar(char c)
    {
        return
            ('0' <= c && c <= '9') ||
            ('a' <= c && c <= 'f') ||
            ('A' <= c && c <= 'F');
    }

    int FromHexChar(char c)
    {
        if ('0' <= c && c <= '9')
        {
            return c - '0';
        }
        else if ('a' <= c && c <= 'f')
        {
            return c - 'a' + 0xA;
        }
        else if ('A' <= c && c <= 'F')
        {
            return c - 'A' + 0xA;
        }
        else
        {
            NN_ABORT("invalid hex char");
        }
    }
}

namespace nn
{
    namespace repair
    {
        uint8_t PublicExponent[4] = { 0x00, 0x01, 0x00, 0x01 };

        InitialVector InitialVector::MakeZero()
        {
            InitialVector ret = {};
            return ret;
        }

        bool Cmac::operator==(const Cmac &cmac) const
        {
            return nn::crypto::IsSameBytes(this->data, cmac.data, this->CMAC_SIZE);
        }

        bool Cmac::operator!=(const Cmac & cmac) const
        {
            return !((*this) == cmac);
        }

        Cmac Cmac::MakeZero()
        {
            Cmac ret = {};
            return ret;
        }

        bool Sha256Hash::operator==(const Sha256Hash &hash) const
        {
            return nn::crypto::IsSameBytes(this->data, hash.data, this->SIZE);
        }

        bool Sha256Hash::operator!=(const Sha256Hash &hash) const
        {
            return !((*this) == hash);
        }

        Sha256Hash Sha256Hash::MakeZero()
        {
            return Sha256Hash();
        }

        Sha256Hash Sha256Hash::Make(void * buffer, size_t size)
        {
            Sha256Hash ret;
            nn::crypto::Sha256Generator sha256;
            sha256.Initialize();
            sha256.Update(buffer, size);
            sha256.GetHash(ret.data, ret.SIZE);

            return ret;
        }

        bool Key128::IsValid(const char *hexString)
        {
            if (strnlen(hexString, KEY_STRING_SIZE + 1) != KEY_STRING_SIZE)
            {
                return false;
            }

            for (int i = 0; i < KEY_STRING_SIZE; i++)
            {
                if (!IsHexChar(hexString[i]))
                {
                    return false;
                }
            }

            return true;

        }

        Key128 Key128::Make(uint8_t *data)
        {
            Key128 ret;
            for (int i = 0; i < KEY_SIZE; i++)
            {
                ret.data[i] = data[i];
            }
            return ret;
        }

        Key128 Key128::Make(const char *hexString)
        {
            NN_ABORT_UNLESS(Key128::IsValid(hexString), "Invalid key string.");
            Key128 ret;
            for (int i = 0; i < KEY_SIZE; i++)
            {
                ret.data[i] = static_cast<uint8_t>((FromHexChar(hexString[i * 2]) << 4) | FromHexChar(hexString[i * 2 + 1]));
            }
            return ret;
        }

        Key128 Key128::MakeZero()
        {
            Key128 ret = {};
            return ret;
        }

        Key128 Key128::EncryptKey(Key128 source, Key128 kek)
        {
            Key128 ret;

            for (int i = 0; i < KEY_SIZE; i++)
            {
                ret.data[i] = source.data[i] ^ kek.data[i];
            }

            return ret;
        }

        bool Key128::operator==(const Key128 & key) const
        {
            return nn::crypto::IsSameBytes(this->data, key.data, this->KEY_SIZE);
        }

        bool Key128::operator!=(const Key128 & key) const
        {
            return !((*this) == key);
        }


        bool Key2048::IsValid(const char *hexString)
        {
            if (strnlen(hexString, KEY_STRING_SIZE + 1) != KEY_STRING_SIZE)
            {
                return false;
            }

            for (int i = 0; i < KEY_STRING_SIZE; i++)
            {
                if (!IsHexChar(hexString[i]))
                {
                    return false;
                }
            }

            return true;
        }

        Key2048 Key2048::Make(uint8_t *data)
        {
            Key2048 ret;
            for (int i = 0; i < KEY_SIZE; i++)
            {
                ret.data[i] = data[i];
            }
            return ret;
        }

        Key2048 Key2048::Make(const char *hexString)
        {
            NN_ABORT_UNLESS(Key2048::IsValid(hexString), "Invalid key string.");
            Key2048 ret;
            for (int i = 0; i < KEY_SIZE; i++)
            {
                ret.data[i] = static_cast<uint8_t>((FromHexChar(hexString[i * 2]) << 4) | FromHexChar(hexString[i * 2 + 1]));
            }
            return ret;
        }

        bool Id128::operator==(const Id128 & id) const
        {
            return nn::crypto::IsSameBytes(this->data, id.data, this->SIZE);
        }

        bool Id128::operator!=(const Id128 & id) const
        {
            return !((*this) == id);
        }

        Id128 Id128::MakeZero()
        {
            Id128 ret = {};
            return ret;
        }
}
}
