﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/repair/repair_CryptUtility.h>
#include <nn/repair/repair_FileSystem.h>
#include <nn/repair/repair_Authentication.h>
#include <nn/repair/repair_CommandLineOption.h>
#include <nn/result/result_HandlingUtility.h>
#include <string>
#include <nn/manu/manu_Api.h>
#include <nn/nn_SdkLog.h>
#include <nn/spl/spl_Api.h>
#include <nn/os.h>

#include "../repair_Utility.h"
#include "repair_BlackList.h"
#include "repair_CachedArchiveFile.h"
#include "repair_Converter.h"
#include "repair_MessageReporter.h"
#include "repair_Record.h"
#include "repair_SaveDataEntry.h"
#include "repair_settingsFlag.h"
#include "repair_Transporter.h"


namespace nn { namespace repair { namespace detail {

const std::string Transporter::ArchiveName = "Archive.bin";

nn::Result Transporter::CreateArchiveFile(std::shared_ptr<CachedArchiveFile>* pOutFile) const NN_NOEXCEPT
{
    // セキュアモード時のみ認証を実施
    if (!m_IsSecurely)
    {
        // アーカイブファイルの作成
        NN_REPAIR_RESULT_DO(CachedArchiveFile::CreateArchive(pOutFile, m_ArchivePath.c_str()));
        NN_RESULT_SUCCESS;
    }

    nn::spl::InitializeForCrypto();

    // 鍵の生成
    Key128 key;
    {
        std::shared_ptr<nn::repair::IProtectedFileEncryptor> encryptor;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::repair::CreateProtectedFileEncryptor(&encryptor, "Spl"));
        NN_ABORT_UNLESS_RESULT_SUCCESS(encryptor->GenerateEncryptedKey(&key));
    }

    // 認証
    if (nn::os::GetHostArgc() < 3)
    {
        SendMessage("Invalid parameter\n");
        NN_RESULT_THROW(nn::fs::ResultUnsupportedOperation());
    }
    SendMessage("Begin authentication\n");
    AuthenticationArchiveContent authenticatedContent;
    NN_REPAIR_RESULT_DO(
        nn::repair::RequestBackupByCallback(&authenticatedContent, key,
    [](const nn::repair::BackupRequestMessage &message) -> nn::Result {
        SendMessage("Send a request\n");

        NN_REPAIR_RESULT_DO(
            nn::manu::WriteToHost(&message, sizeof(message), GetOptionArgument(RequestOption), 0, sizeof(message)));

        SendMessage("Finished to send the request.\n");

        NN_RESULT_SUCCESS;
    },
    [](nn::repair::BackupResponseMessage *pOut) -> nn::Result {
        auto response = GetOptionArgument(ResponseOption);
        while (true)
        {
            SendMessage("Wait a response.\n");

            size_t fileSize;
            auto result = nn::manu::GetFileSize(&fileSize, response);
            if (result.IsSuccess())
            {
                if (fileSize == sizeof(*pOut))
                {
                    break;
                }
            }

            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        }

        NN_REPAIR_RESULT_DO(
            nn::manu::ReadFromHost(pOut, sizeof(*pOut), response, 0, sizeof(*pOut)));

        SendMessage("Received the response.\n");

        NN_RESULT_SUCCESS;
    }));
    SendMessage("Succeeded the authentication.\n");

    // アーカイブファイルの作成
    NN_REPAIR_RESULT_DO(CachedArchiveFile::CreateArchive(pOutFile, m_ArchivePath.c_str(), key, authenticatedContent));
    NN_RESULT_SUCCESS;
}

nn::Result Transporter::Export(const BlackListStruct* pList) const NN_NOEXCEPT
{
    // fs のアボートを抑制
    nn::fs::SetEnabledAutoAbort(false);

    // バックアップ記録ファイルの作成
    NN_REPAIR_RESULT_DO(CreateRecord(BackupRecordName));

    // 破損記録ファイルの作成
    NN_REPAIR_RESULT_DO(CreateRecord(CorruptionRecordName));

    // アーカイブファイルを作成
    std::shared_ptr<CachedArchiveFile> pFile;
    NN_REPAIR_RESULT_DO(this->CreateArchiveFile(&pFile));

    // ブラックリストを作成
    BlackList blackList(pList);

    // バックアップデータに修理中フラグを立てる
    // （エラー終了しても立つ)
    NN_REPAIR_RESULT_DO(SetInRepairFlag( true ));

    // アカウント破損時はアルバムのみ救出
    // システムセーブデータはすべて正常時にのみ救出
    if (this->IsValidAccount())
    {
        if (this->IsValidSystem(&blackList))
        {
            NN_REPAIR_RESULT_DO(this->ExportSystemData(pFile, &blackList));
        }
        else
        {
            // システムデータが壊れているときはアカウントのデータだけ救う
            NN_REPAIR_RESULT_DO(this->ExportSystemData(
                        pFile, ACCOUNT_SAVEDATA_ID, &blackList));
        }

        NN_REPAIR_RESULT_DO(this->ExportUserData(pFile));
        NN_REPAIR_RESULT_DO(this->ExportThumbnailData(pFile, &blackList));
    }

    NN_REPAIR_RESULT_DO(this->ExportAlbumData(pFile));

    NN_REPAIR_RESULT_DO(pFile->Close());

    NN_RESULT_SUCCESS;
}

nn::Result Transporter::ExportSystemData(
        std::shared_ptr<CachedArchiveFile> pFile, const BlackList* pBlackList) const NN_NOEXCEPT
{
    SendMessage("export system save data\n");

    Converter converter(pFile, Converter::DataType::System);
    NN_REPAIR_RESULT_DO(converter.ToArchive(pBlackList));

    NN_RESULT_SUCCESS;
}

nn::Result Transporter::ExportSystemData(
        std::shared_ptr<CachedArchiveFile> pFile,
        nn::Bit64 id,
        const BlackList* pBlackList) const NN_NOEXCEPT
{
    SendMessage("export account save data\n");

    Converter converter(pFile, Converter::DataType::System);
    NN_REPAIR_RESULT_DO(converter.ToArchive(id, pBlackList));

    NN_RESULT_SUCCESS;
}

nn::Result Transporter::ExportUserData(std::shared_ptr<CachedArchiveFile> pFile) const NN_NOEXCEPT
{
    SendMessage("export user save data\n");

    Converter converter(pFile, Converter::DataType::User);
    NN_REPAIR_RESULT_DO(converter.ToArchive());

    NN_RESULT_SUCCESS;
}

nn::Result Transporter::ExportThumbnailData(
        std::shared_ptr<CachedArchiveFile> pFile, const BlackList* pBlackList) const NN_NOEXCEPT
{
    SendMessage("export thumbnail data\n");

    Converter converter(pFile, Converter::DataType::Thumbnail);
    NN_REPAIR_RESULT_DO(converter.ToArchive(pBlackList));

    NN_RESULT_SUCCESS;
}

nn::Result Transporter::ExportAlbumData(std::shared_ptr<CachedArchiveFile> pFile) const NN_NOEXCEPT
{
    SendMessage("export album data\n");

    Converter converter(pFile, Converter::DataType::Album);
    NN_REPAIR_RESULT_DO(converter.ToArchive());

    NN_RESULT_SUCCESS;
}

nn::Result Transporter::OpenArchiveFile(std::shared_ptr<CachedArchiveFile>* pOutFile) const NN_NOEXCEPT
{
    if (!m_IsSecurely)
    {
        // アーカイブファイルを開く
        return CachedArchiveFile::OpenArchive(pOutFile, m_ArchivePath.c_str());

        NN_RESULT_SUCCESS;
    }

    nn::spl::InitializeForCrypto();

    // アーカイブファイルを開く
    std::shared_ptr<FileSystem> fileSystem = nn::repair::FileSystem::Create("nnfs");

    // 鍵の生成
    AuthenticationArchiveContent authenticatedContent;
    NN_REPAIR_RESULT_DO(nn::repair::LoadEncryptedKey(&authenticatedContent, m_ArchivePath, fileSystem));

    // 認証
    if (nn::os::GetHostArgc() < 3)
    {
        SendMessage("Invalid parameter\n");
        NN_RESULT_THROW(nn::fs::ResultUnsupportedOperation());
    }

    SendMessage("Begin authentication\n");
    Key128 key;
    NN_REPAIR_RESULT_DO(
        nn::repair::RequestRestoreByCallback(&key, authenticatedContent,
    [](const nn::repair::RestoreRequestMessage &message) -> nn::Result {
        SendMessage("Send a request\n");

        NN_REPAIR_RESULT_DO(
            nn::manu::WriteToHost(&message, sizeof(message), GetOptionArgument(RequestOption), 0, sizeof(message)));

        SendMessage("Finished to send the request.\n");

        NN_RESULT_SUCCESS;
    },
    [](nn::repair::RestoreResponseMessage *pOut) -> nn::Result {

        auto response = GetOptionArgument(ResponseOption);
        while (true)
        {
            SendMessage("Wait a response.\n");

            size_t fileSize;
            auto result = nn::manu::GetFileSize(&fileSize, response);
            if (result.IsSuccess())
            {
                if (fileSize == sizeof(*pOut))
                {
                    break;
                }
            }

            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        }

        NN_REPAIR_RESULT_DO(
            nn::manu::ReadFromHost(pOut, sizeof(*pOut), response, 0, sizeof(*pOut)));

        SendMessage("Received the response.\n");

        NN_RESULT_SUCCESS;
    }));
    SendMessage("Succeeded the authentication.\n");

    // アーカイブファイルの作成
    NN_ABORT_UNLESS_RESULT_SUCCESS(CachedArchiveFile::OpenArchive(pOutFile, m_ArchivePath.c_str(), key));

    NN_RESULT_SUCCESS;
}

nn::Result Transporter::Import() const NN_NOEXCEPT
{
    // fs のアボートを抑制
    nn::fs::SetEnabledAutoAbort(false);

    // リストア記録用ファイルを作成
    NN_REPAIR_RESULT_DO(CreateRecord(RestoreRecordName));

    std::shared_ptr<CachedArchiveFile> pFile;
    NN_REPAIR_RESULT_DO(this->OpenArchiveFile(&pFile));

    int64_t size = 0;
    int64_t offset = 0;

    offset += size;
    NN_REPAIR_RESULT_DO(this->ImportSystemData(&size, offset, pFile));

    offset += size;
    NN_REPAIR_RESULT_DO(this->ImportUserData(&size, offset, pFile));

    offset += size;
    NN_REPAIR_RESULT_DO(this->ImportThumbnailData(&size, offset, pFile));

    offset += size;
    NN_REPAIR_RESULT_DO(this->ImportAlbumData(&offset, offset, pFile));

    NN_REPAIR_RESULT_DO(pFile->Close());

    // リストアが成功したら

    // 時計補正ツール未実行フラグを有効化する
    SetRequiresRunRepairTimeReviserFlag(true);

    // 修理フラグを落とす
    SetInRepairFlag(false);

    NN_RESULT_SUCCESS;
}

nn::Result Transporter::ImportSystemData(
        int64_t* pOutSize, int64_t offset, std::shared_ptr<CachedArchiveFile> pFile) const NN_NOEXCEPT
{
    SendMessage("import system save data\n");

    Converter converter(pFile, Converter::DataType::System);
    NN_REPAIR_RESULT_DO(converter.ToData(pOutSize, offset));

    NN_RESULT_SUCCESS;
}

nn::Result Transporter::ImportUserData(
        int64_t* pOutSize, int64_t offset, std::shared_ptr<CachedArchiveFile> pFile) const NN_NOEXCEPT
{
    SendMessage("import user save data\n");

    Converter converter(pFile, Converter::DataType::User);
    NN_REPAIR_RESULT_DO(converter.ToData(pOutSize, offset));

    NN_RESULT_SUCCESS;
}

nn::Result Transporter::ImportThumbnailData(
        int64_t* pOutSize, int64_t offset, std::shared_ptr<CachedArchiveFile> pFile) const NN_NOEXCEPT
{
    SendMessage("import thumbnail data\n");

    Converter converter(pFile, Converter::DataType::Thumbnail);
    NN_REPAIR_RESULT_DO(converter.ToData(pOutSize, offset));

    NN_RESULT_SUCCESS;
}

nn::Result Transporter::ImportAlbumData(
        int64_t* pOutSize, int64_t offset, std::shared_ptr<CachedArchiveFile> pFile) const NN_NOEXCEPT
{
    SendMessage("import album data\n");

    Converter converter(pFile, Converter::DataType::Album);
    NN_REPAIR_RESULT_DO(converter.ToData(pOutSize, offset));

    NN_RESULT_SUCCESS;
}

nn::Result Transporter::ExportedId(bool* pOut, nn::Bit64 id) const NN_NOEXCEPT
{
    RecordHandle handle;
    NN_REPAIR_RESULT_DO(OpenRecord(&handle, BackupRecordName));
    NN_UTIL_SCOPE_EXIT
    {
        CloseRecord(handle);
    };

    NN_REPAIR_RESULT_DO(IsIdRecorded(pOut, handle, id));

    NN_RESULT_SUCCESS;
}

nn::Result Transporter::ImportedId(bool* pOut, nn::Bit64 id) const NN_NOEXCEPT
{
    RecordHandle handle;
    NN_REPAIR_RESULT_DO(OpenRecord(&handle, RestoreRecordName));
    NN_UTIL_SCOPE_EXIT
    {
        CloseRecord(handle);
    };

    NN_REPAIR_RESULT_DO(IsIdRecorded(pOut, handle, id));

    NN_RESULT_SUCCESS;
}

bool Transporter::IsValidAccount() const NN_NOEXCEPT
{
    RecordHandle handle;
    NN_REPAIR_RESULT_DO(OpenRecord(&handle, CorruptionRecordName));
    NN_UTIL_SCOPE_EXIT
    {
        CloseRecord(handle);
    };

    SaveDataEntry entry(nn::fs::SaveDataSpaceId::ProperSystem, 0);
    auto isValid = entry.VerifyData(ACCOUNT_SAVEDATA_ID);

    // アカウントセーブデータ破損時はすべてのセーブデータを救わない
    if (!isValid)
    {
        SendRecordLog(handle, "all save data\n");
    }

    return isValid;
}

bool Transporter::IsValidSystem(const BlackList* pBlackList) const NN_NOEXCEPT
{
    RecordHandle handle;
    NN_REPAIR_RESULT_DO(OpenRecord(&handle, CorruptionRecordName));
    NN_UTIL_SCOPE_EXIT
    {
        CloseRecord(handle);
    };

    SaveDataEntry entry(nn::fs::SaveDataSpaceId::ProperSystem,0);
    auto isValid = entry.VerifyDataAll(pBlackList);

    // 一つでも破損データがあった場合はすべてのシステムセーブデータを救わない
    if (!isValid)
    {
        SendRecordLog(handle, "all system save data except for account\n");
    }

    return isValid;
}

nn::Result Transporter::ShowRecordedData(
        bool* pOutHasCorruption, const char* name,
        ReportFunctionPointer const pFunction, void* const pParameter) const NN_NOEXCEPT
{
    RecordHandle handle;
    NN_REPAIR_RESULT_DO(OpenRecord(&handle, name));
    NN_UTIL_SCOPE_EXIT
    {
        CloseRecord(handle);
    };

    NN_REPAIR_RESULT_DO(ShowRecorded(pOutHasCorruption, handle, pFunction, pParameter));

    NN_RESULT_SUCCESS;
}

}}} // namespace nn::repair::detail

