﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/fs.h>
#include <nn/fs/fs_SaveDataTypes.h>
#include <nn/nn_Result.h>
#include <nn/repair.h>
#include <string>

namespace nn { namespace repair { namespace detail {

//!< 記録ファイルのハンドル
typedef nn::fs::FileHandle RecordHandle;

//!< 記録ファイルを扱うクラスです。
class Record final
{
private:
    //!< 記録ファイルを残すディレクトリパス
    std::string m_Path;

public:
    Record() NN_NOEXCEPT
    {
    };

    //!< 記録ファイルを残すディレクトリパスをセットします。
    void SetDirectoryPath(const std::string& path) NN_NOEXCEPT;

    //!< 記録ファイルを作成します。
    nn::Result Create(const std::string& name) const NN_NOEXCEPT;

    //!< 開きます。
    nn::Result Open(RecordHandle* pOutHandle, const std::string& name) const NN_NOEXCEPT;

    //!< 閉じます。
    void Close(const RecordHandle& handle) const NN_NOEXCEPT;

    //!< ログを追記します。
    nn::Result Log(const RecordHandle& handle, const char* log) const NN_NOEXCEPT;

    //!< Application Id がログに含まれるか調べます。
    nn::Result Contain(bool* pOut, const RecordHandle& handle, nn::Bit64 id) const NN_NOEXCEPT;

    //!< Application Id と User Id の組み合わせがログに含まれるか調べます。
    nn::Result Contain(bool* pOut, const RecordHandle& handle, nn::Bit64 id, nn::fs::UserId userId) const NN_NOEXCEPT;

    //!< ログを表示します。
    nn::Result Show(
            bool* pOutHasCorruption, const RecordHandle& handle,
            ReportFunctionPointer const pFunction, void* const pParameter) const NN_NOEXCEPT;
};

//!< ディレクトリパスをセットします。
void SetRecordDirectoryPath(const char* path) NN_NOEXCEPT;

//!< 記録ファイルを作成します。
nn::Result CreateRecord(const char* name) NN_NOEXCEPT;

//!< 記録ファイルを開きます。
nn::Result OpenRecord(RecordHandle* pOutHandle, const char* name) NN_NOEXCEPT;

//!< 記録ファイルを閉じます。
void CloseRecord(const RecordHandle& handle) NN_NOEXCEPT;

//!< ログを追加します。
nn::Result SendRecordLog(const RecordHandle& handle, const char* pFormat, ...) NN_NOEXCEPT;

//!< 指定の Application Id がログに含まれるか調べます。
nn::Result IsIdRecorded(bool* pOut, const RecordHandle& handle, nn::Bit64 id) NN_NOEXCEPT;

//!< 指定の Application Id と User Id の組み合わせがログに含まれるか調べます。
nn::Result IsIdRecorded(bool* pOut, const RecordHandle& handle, nn::Bit64 id, nn::fs::UserId userId) NN_NOEXCEPT;

//!< 記録されたデータを表示します。
nn::Result ShowRecorded(
        bool* pOutHasCorruption, const RecordHandle& handle,
        ReportFunctionPointer const pFunction, void* const pParameter) NN_NOEXCEPT;

}}} // namespace nn::repair::detail

