﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory>
#include <nn/repair.h>
#include <nn/repair/repair_CryptUtility.h>
#include <nn/repair/repair_FileSystem.h>
#include "repair_BlackList.h"
#include "repair_CachedArchiveFile.h"

namespace nn { namespace repair { namespace detail {

//!< データを保存形式に変換するクラスです。
class Converter final
{
public:
    //!< データの種別を表す列挙型です。
    enum class DataType : uint64_t
    {
        System,    //!< システムセーブデータ
        User,      //!< ユーザセーブデータ
        Thumbnail, //!< サムネイルデータ
        Album,     //!< アルバムデータ
    };

private:
    //!< 旧ヘッダと区別するためのマジックナンバー
    static const uint32_t MAGIC_NUMBER;

    //!< 互換性チェックのためのバージョン
    static const uint32_t CURRENT_VERSION;

    //!< ヘッダ用の構造体です。
    struct Header final
    {
        //!< マジックナンバー
        const uint32_t magic = MAGIC_NUMBER;

        //!< バージョン
        const uint32_t version = CURRENT_VERSION;

        //!< エントリの種別
        DataType type;

        //!< エントリ数
        int64_t count;
    };

    //!< アーカイブファイルへのアクセスポインタ
    std::shared_ptr<CachedArchiveFile> m_pFile;

    //!< データの種別
    DataType m_Type;

public:
    Converter(std::shared_ptr<CachedArchiveFile> pFile, DataType type) NN_NOEXCEPT
        : m_pFile(pFile)
        , m_Type(type)
    {
    }

    //!< アーカイブへ変換します。
    nn::Result ToArchive(const BlackList* pList = nullptr) const NN_NOEXCEPT;

    //!< 指定した ID のデータをアーカイブへ変換します。
    nn::Result ToArchive(nn::Bit64 id, const BlackList* pList = nullptr) const NN_NOEXCEPT;

    //!< データへ変換します。
    nn::Result ToData(int64_t* pOutSize, int64_t offset) const NN_NOEXCEPT;

private:
    //!< ヘッダを書き込みます。
    nn::Result WriteHeader(const BlackList* pList, int64_t count = 0) const NN_NOEXCEPT;

    //!< ボディを書き込みます。。
    nn::Result WriteBody(const BlackList* pList, nn::Bit64 id = 0) const NN_NOEXCEPT;

    //!< エントリを数えます。
    nn::Result CountEntries(int64_t* pOutCount, const BlackList* pList) const NN_NOEXCEPT;
};

}}} // namespace nn::repair::detail

