﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/g3d/g3d_VisibilityAnimObj.h>
#include <algorithm>
#include <nw/g3d/ut/g3d_Flag.h>
#include <nw/g3d/ut/g3d_Perf.h>
#include <nw/g3d/g3d_ModelObj.h>

namespace nw { namespace g3d {

void VisibilityAnimObj::Sizer::Calc(const InitArg& arg)
{
    NW_G3D_ASSERT(arg.IsValid());

    int numTarget = 0;
    int numAnim = 0;
    if (arg.GetMaxBoneAnimCount() > 0)
    {
        numTarget = arg.GetMaxBoneCount();
        numAnim = arg.GetMaxBoneAnimCount();
    }
    if (arg.GetMaxMatAnimCount() > 0)
    {
        numTarget = std::max(numTarget, arg.GetMaxMatCount());
        numAnim = std::max(numAnim, arg.GetMaxMatAnimCount());
    }
    int numBind = std::max(numTarget, numAnim);
    int numCurve = arg.GetMaxCurveCount();

    // サイズ計算
    int idx = 0;
    chunk[idx++].size = Align(numAnim, 32) >> 3;
    chunk[idx++].size = sizeof(bit32) * numBind;
    chunk[idx++].size = arg.IsContextEnabled() ? Align(sizeof(AnimFrameCache) * numCurve) : 0;
    NW_G3D_ASSERT(idx == NUM_CHUNK);

    CalcOffset(chunk, NUM_CHUNK);
}

size_t VisibilityAnimObj::CalcBufferSize(const InitArg& arg)
{
    Sizer& sizer = arg.GetSizer();
    sizer.Calc(arg);
    return sizer.GetTotalSize();
}

bool VisibilityAnimObj::Init(const InitArg& arg, void* pBuffer, size_t bufferSize)
{
    NW_G3D_ASSERT(bufferSize == 0 || pBuffer);
    NW_G3D_WARNING(IsAligned(pBuffer, BUFFER_ALIGNMENT), "pBuffer must be aligned.");

    Sizer& sizer = arg.GetSizer();
    if (!sizer.IsValid())
    {
        // キャッシュが残っていない場合は再計算する。
        sizer.Calc(arg);
    }
    if (sizer.GetTotalSize() > bufferSize)
    {
        // バッファが必要なサイズに満たない場合は失敗。
        return false;
    }

    int numTarget = 0;
    int numAnim = 0;
    if (arg.GetMaxBoneAnimCount() > 0)
    {
        numTarget = arg.GetMaxBoneCount();
        numAnim = arg.GetMaxBoneAnimCount();
    }
    if (arg.GetMaxMatAnimCount() > 0)
    {
        numTarget = std::max(numTarget, arg.GetMaxMatCount());
        numAnim = std::max(numAnim, arg.GetMaxMatAnimCount());
    }
    int numBind = std::max(numTarget, numAnim);
    int numCurve = arg.GetMaxCurveCount();

    // メンバの初期化。
    void* ptr = pBuffer;
    SetBufferPtr(pBuffer);
    // フレーム関係はリソース設定時に初期化
    m_pRes = NULL;
    GetBindTable().Init(sizer.GetPtr<bit32>(ptr, Sizer::BIND_TABLE), numBind);
    GetContext().Init(sizer.GetPtr<AnimFrameCache>(ptr, Sizer::FRAMECACHE_ARRAY), numCurve);
    SetResultBuffer(sizer.GetPtr(ptr, Sizer::RESULT_BUFFER));
    m_MaxBoneAnim = static_cast<u16>(arg.GetMaxBoneAnimCount());
    m_MaxMatAnim = static_cast<u16>(arg.GetMaxMatAnimCount());

    return true;
}

void VisibilityAnimObj::SetResource(ResVisibilityAnim* pRes)
{
    NW_G3D_ASSERTMSG(IsAcceptable(pRes), "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));

    m_pRes = pRes;
    m_pCurveArray = pRes->ref().ofsCurveArray.to_ptr<ResAnimCurveData>();
    m_NumCurve = pRes->GetCurveCount();

    SetTargetUnbound();

    bool loop = (pRes->ref().flag & AnimFlag::PLAYPOLICY_LOOP) != 0;
    ResetFrameCtrl(pRes->GetFrameCount(), loop);
    GetBindTable().SetAnimCount(pRes->GetAnimCount());
    GetContext().SetCurveCount(pRes->GetCurveCount());
}

BindResult VisibilityAnimObj::Bind(const ResModel* pModel)
{
    NW_G3D_ASSERTMSG(IsAcceptable(pModel), "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));

    const ResDicType* pDic = NULL;
    int count = 0;
    if (m_pRes->GetVisibilityType() == ResVisibilityAnim::BONE_VISIBILITY)
    {
        const ResSkeleton* pSkeleton = pModel->GetSkeleton();
        pDic = NW_G3D_RES_DIC(pSkeleton->ref().ofsBoneDic);
        count = pSkeleton->GetBoneCount();
    }
    else
    {
        pDic = NW_G3D_RES_DIC(pModel->ref().ofsMaterialDic);
        count = pModel->GetMaterialCount();
    }
    NW_G3D_ASSERTMSG(count <= GetBindTable().GetSize(), "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));

    BindResult result;
    AnimBindTable& bindTable = GetBindTable();
    bindTable.ClearAll(count);
    const BinString* pNameArray = m_pRes->ref().ofsNameArray.to_ptr<BinString>();
    for (int idxAnim = 0, numAnim = bindTable.GetAnimCount(); idxAnim < numAnim; ++idxAnim)
    {
        const ResName* pName = pNameArray[idxAnim].GetResName();
        int idxTarget = pDic->FindIndex(pName);
        if (idxTarget >= 0)
        {
            bindTable.Bind(idxAnim, idxTarget);
            result |= BindResult::Bound();
        }
        else
        {
            result |= BindResult::NotBound();
        }
    }

    SetTargetBound();
    VisibilityAnimObj::ClearResult();
    return result;
}

BindResult VisibilityAnimObj::Bind(const ModelObj* pModelObj)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pModelObj, "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
    return Bind(pModelObj->GetResource());
}

void VisibilityAnimObj::BindFast(const ResModel* pModel)
{
    NW_G3D_ASSERTMSG(IsAcceptable(pModel), "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
    NW_G3D_ASSERTMSG(pModel == m_pRes->GetBindModel(), "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
    int count = m_pRes->GetVisibilityType() == ResVisibilityAnim::BONE_VISIBILITY ?
        pModel->GetSkeleton()->GetBoneCount() : pModel->GetMaterialCount();

    AnimBindTable& bindTable = GetBindTable();
    bindTable.ClearAll(count);
    bindTable.BindAll(m_pRes->ref().ofsBindIndexArray.to_ptr<u16>());

    SetTargetBound();
    VisibilityAnimObj::ClearResult();
}

void VisibilityAnimObj::ClearResult()
{
    NW_G3D_ASSERT_NOT_NULL(m_pRes);

    int sizeResult = Align(GetAnimCount(), 32) >> 3; // 1ビジビリティ1ビットでのサイズ。
    memcpy(GetResultBuffer(), m_pRes->ref().ofsBaseValueArray.to_ptr(), sizeResult);
}

void VisibilityAnimObj::Calc()
{
    NW_G3D_PERF_LEVEL1_FUNC();
    NW_G3D_ASSERT_NOT_NULL(m_pRes);
    NW_G3D_ASSERTMSG(IsTargetBound(), "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));

    if (IsFrameChanged())
    {
        float frame = GetFrameCtrl().GetFrame();
        const AnimBindTable& bindTable = GetBindTable();
        AnimContext& context = GetContext();
        bit32* pResultBuffer = static_cast<bit32*>(GetResultBuffer());
        if (context.IsFrameCacheValid())
        {
            for (int idxCurve = 0; idxCurve < m_NumCurve; ++idxCurve)
            {
                const ResAnimCurve* pCurve = GetCurve(idxCurve);
                int idxAnim = static_cast<int>(pCurve->ref().targetOffset);

                if (!bindTable.IsCalcEnabled(idxAnim))
                {
                    continue;
                }

                int visible = pCurve->EvalInt(frame, context.GetFrameCacheArray(idxCurve));
                SetBit(pResultBuffer, idxAnim, static_cast<bit32>(visible));
            }
        }
        else
        {
            for (int idxCurve = 0; idxCurve < m_NumCurve; ++idxCurve)
            {
                const ResAnimCurve* pCurve = GetCurve(idxCurve);
                int idxAnim = static_cast<int>(pCurve->ref().targetOffset);

                if (!bindTable.IsCalcEnabled(idxAnim))
                {
                    continue;
                }

                int visible = pCurve->EvalInt(frame);
                SetBit(pResultBuffer, idxAnim, static_cast<bit32>(visible));
            }
        }
        UpdateLastFrame();
    }
}

void VisibilityAnimObj::ApplyTo(ModelObj* pModelObj) const
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(m_pRes, "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
    NW_G3D_ASSERTMSG(IsTargetBound(), "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pModelObj, "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));

    const bit32* pResultBuffer = static_cast<const bit32*>(GetResultBuffer());
    const AnimBindTable& bindTable = GetBindTable();
    for (int idxAnim = 0, numAnim = bindTable.GetAnimCount(); idxAnim < numAnim; ++idxAnim)
    {
        if (!bindTable.IsApplyEnabled(idxAnim))
        {
            continue;
        }

        int  idxTarget = bindTable.GetTargetIndex(idxAnim);
        bool visible   = IsBitOn<bool>(pResultBuffer, idxAnim);
        int  type      = m_pRes->GetVisibilityType();
        switch(type)
        {
            case ResVisibilityAnim::BONE_VISIBILITY:
                pModelObj->SetBoneVisibility(idxTarget, visible);
                break;
            case ResVisibilityAnim::MAT_VISIBILITY:
                pModelObj->SetMatVisibility(idxTarget, visible);
                break;
            default:
                break;
        }
    }
}

}} // namespace nw::g3d
