﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/g3d/g3d_SkeletonObj.h>
#include <nw/g3d/ut/g3d_Flag.h>
#include <nw/g3d/ut/g3d_Perf.h>
#include <nw/g3d/fnd/g3d_GfxManage.h>
#include <nw/g3d/g3d_Billboard.h>

namespace nw { namespace g3d {

// メンバ関数ポインタはサイズの解釈が一定でないためヘッダに出さない
class SkeletonObj::Impl
{
public:
    static void (SkeletonObj::* const s_pFuncCalcWorld[])(const Mtx34&);
    static void (SkeletonObj::* const s_pFuncCalcWorldWithCallback[])(const Mtx34&);
};

class CalcWorldNoScale;
class CalcWorldStd;
class CalcWorldMaya;
class CalcWorldSoftimage;

void (SkeletonObj::* const SkeletonObj::Impl::s_pFuncCalcWorld[])(const Mtx34&) = {
    &SkeletonObj::CalcWorldImpl<CalcWorldNoScale, false>,
    &SkeletonObj::CalcWorldImpl<CalcWorldStd, false>,
    &SkeletonObj::CalcWorldImpl<CalcWorldMaya, false>,
    &SkeletonObj::CalcWorldImpl<CalcWorldSoftimage, false>
};

void (SkeletonObj::* const SkeletonObj::Impl::s_pFuncCalcWorldWithCallback[])(const Mtx34&) = {
    &SkeletonObj::CalcWorldImpl<CalcWorldNoScale, true>,
    &SkeletonObj::CalcWorldImpl<CalcWorldStd, true>,
    &SkeletonObj::CalcWorldImpl<CalcWorldMaya, true>,
    &SkeletonObj::CalcWorldImpl<CalcWorldSoftimage, true>
};

void SkeletonObj::Sizer::Calc(const InitArg& arg)
{
    NW_G3D_ASSERT(arg.GetBufferingCount() > 0);
    ResSkeleton* pRes = arg.GetResource();
    int numBone = pRes->GetBoneCount();
    int scaleMode = pRes->GetScaleMode();

    // サイズ計算
    int idx = 0;
    chunk[idx++].size = Align(sizeof(Mtx34) * numBone, LL_CACHE_FETCH_SIZE);
    chunk[idx++].size = Align(sizeof(LocalMtx) * numBone, LL_CACHE_FETCH_SIZE);
    chunk[idx++].size = scaleMode == SCALE_SOFTIMAGE ? sizeof(Vec3) * numBone : 0;
    NW_G3D_ASSERT(idx == NUM_CHUNK);

    CalcOffset(chunk, NUM_CHUNK);
}

size_t SkeletonObj::CalcBufferSize(const InitArg& arg)
{
    Sizer& sizer = arg.GetSizer();
    sizer.Calc(arg);
    return sizer.GetTotalSize();
}

bool SkeletonObj::Init(const InitArg& arg, void* pBuffer, size_t bufferSize)
{
    NW_G3D_ASSERT(bufferSize == 0 || pBuffer); // ボーンが無い場合は 0 になる。
    NW_G3D_WARNING(IsAligned(pBuffer, BUFFER_ALIGNMENT), "pBuffer must be aligned.");

    Sizer& sizer = arg.GetSizer();
    if (!sizer.IsValid())
    {
        // キャッシュが残っていない場合は再計算する。
        sizer.Calc(arg);
    }
    if (sizer.GetTotalSize() > bufferSize)
    {
        // バッファが必要なサイズに満たない場合は失敗。
        return false;
    }

    ResSkeleton* pRes = arg.GetResource();

    // メンバの初期化。
    void* ptr = pBuffer;
    m_pRes = pRes;
    m_Flag = static_cast<bit16>(pRes->ref().flag |
        BLOCK_BUFFER_SWAP | CACHE_LOCAL_MTX | CACHE_WORLD_MTX);
    m_NumBuffering = static_cast<u8>(arg.GetBufferingCount());
    m_pBoneArray = pRes->ref().ofsBoneArray.to_ptr<ResBoneData>();
    m_pLocalMtxArray = sizer.GetPtr<LocalMtx>(ptr, Sizer::LOCAL_MTX);
    m_pMemLocalMtxArray = m_pLocalMtxArray;
    m_pWorldMtxArray = sizer.GetPtr<Mtx34>(ptr, Sizer::WORLD_MTX);
    m_pMemWorldMtxArray = m_pWorldMtxArray;
    m_pScaleArray = sizer.GetPtr<Vec3>(ptr, Sizer::SCALE);
    m_NumBone = static_cast<u16>(pRes->GetBoneCount());
    m_CallbackBone = ICalcWorldCallback::INVALID_BONE;
    m_pCallback = NULL;
    m_pLCMtxBlock = NULL;
    m_pUserPtr = NULL;
    m_pBufferPtr = pBuffer;
    m_pBlockBuffer = NULL;
    memset(&m_MtxBlock, 0, sizeof(GfxBuffer));

    ClearLocalMtx();

    // 各バッファは計算結果が参照されるべきなのでここでは初期化しない。

    return true;
}

size_t SkeletonObj::CalcBlockBufferSize() const
{
    return Align(sizeof(Mtx34) * m_pRes->GetMtxCount(), BLOCK_BUFFER_ALIGNMENT) * m_NumBuffering;
}

bool SkeletonObj::SetupBlockBuffer(void* pBuffer, size_t bufferSize)
{
    NW_G3D_ASSERT(bufferSize == 0 || pBuffer);
    NW_G3D_ASSERT_ADDR_ALIGNMENT(pBuffer, BLOCK_BUFFER_ALIGNMENT);
    NW_G3D_ASSERT((m_Flag & BLOCK_BUFFER_VALID) == 0);

    size_t size = CalcBlockBufferSize();

    if (size > bufferSize)
    {
        // バッファが必要なサイズに満たない場合は失敗。
        return false;
    }

    m_pBlockBuffer = pBuffer;

    // size は複数バッファ分のサイズなので 1 枚分に変換して SetData します。
    m_MtxBlock.SetData(pBuffer, size / m_NumBuffering, m_NumBuffering);
    m_MtxBlock.Setup();
#if NW_G3D_IS_HOST_CAFE
    // LCMountMtxBlock() を使う場合に備えてキャッシュを捨てます。
    // LC を使わない場合は DCZeroRange() してから書き込むのでキャッシュミスは起きません。
    DCInvalidateRange(m_MtxBlock.pData, size);
#endif

    // 各バッファは計算結果が参照されるべきなのでここでは初期化しない。

    m_Flag |= BLOCK_BUFFER_VALID;
    return true;
}

void SkeletonObj::CleanupBlockBuffer()
{
    NW_G3D_ASSERT(m_Flag & BLOCK_BUFFER_VALID);

    m_MtxBlock.Cleanup();
    m_Flag &= ~BLOCK_BUFFER_VALID;
}

//--------------------------------------------------------------------------------------------------

void SkeletonObj::ClearLocalMtx()
{
    LocalMtx* pLocalMtxArray = GetLocalMtxArray();
    if (ROT_QUAT != GetRotateMode())
    {
        // Euler
        int numBone = m_pRes->GetBoneCount();
        for (int idxBone = 0; idxBone < numBone; ++idxBone)
        {
            ResBone* pBone = m_pRes->GetBone(idxBone);
            LocalMtx& local = pLocalMtxArray[idxBone];
            local.flag = pBone->ref().flag & (ResBone::TRANSFORM_MASK | ResBone::BILLBOARD_MASK);
            local.scale.Set(pBone->GetScale());
            local.mtxRT.SetR(pBone->GetRotateEuler());
            local.mtxRT.SetT(pBone->GetTranslate());
        }
    }
    else
    {
        // Quaternion
        int numBone = m_pRes->GetBoneCount();
        for (int idxBone = 0; idxBone < numBone; ++idxBone)
        {
            ResBone* pBone = m_pRes->GetBone(idxBone);
            LocalMtx& local = pLocalMtxArray[idxBone];
            local.flag = pBone->ref().flag & (ResBone::TRANSFORM_MASK | ResBone::BILLBOARD_MASK);
            local.scale.Set(pBone->GetScale());
            local.mtxRT.SetR(pBone->GetRotateQuat());
            local.mtxRT.SetT(pBone->GetTranslate());
        }
    }
}

void SkeletonObj::CalcBillboardMtx(
    Mtx34* pMtx, const Mtx34& cameraMtx, int boneIndex, bool combineWorld) const
{
    NW_G3D_PERF_LEVEL1_FUNC();
    NW_G3D_ASSERTMSG(pMtx != NULL, "%s\n", NW_G3D_RES_GET_NAME(GetBone(boneIndex), GetName()));

    const Mtx34* pWorldMtxArray = GetWorldMtxArray();
    bit32 bbMode = GetBone(boneIndex)->GetBillboardMode();
    const Mtx34& world = pWorldMtxArray[boneIndex];

    Mtx34 wv;
    wv.Mul(cameraMtx, world);
    Billboard::Calc(bbMode, &wv, cameraMtx, world, m_pLocalMtxArray[boneIndex].mtxRT);

    if ((bbMode & ~ResBone::BILLBOARD_CHILD) == ResBone::BILLBOARD_SCREEN_VIEWVECTOR)
    {
        // スクリーンビルボード
        const_cast<bit16&>(m_Flag) |= CACHE_LOCAL_MTX | CACHE_WORLD_MTX;
    }
    else
    {
        const_cast<bit16&>(m_Flag) |= CACHE_WORLD_MTX;
    }

    // 階層ビルボードのためビルボード行列単体を取り出す。
    Mtx34 inverse;
    float det;

    inverse.Inverse(&det, cameraMtx); // TODO: 最適化
    pMtx->Mul(inverse, wv);

    if (combineWorld)
    {
        return;
    }

    inverse.Inverse(&det, world); // TODO: det == 0 の場合の扱い
    pMtx->Mul(*pMtx, inverse);
}

void SkeletonObj::CalcWorldMtx(const Mtx34& baseMtx)
{
    NW_G3D_PERF_LEVEL1_FUNC();
    int funcIndex = (m_Flag & SCALE_MASK) >> (SCALE_SHIFT);
    NW_G3D_ASSERT_INDEX_BOUNDS(funcIndex, sizeof(Impl::s_pFuncCalcWorld) / sizeof(*Impl::s_pFuncCalcWorld));

    m_Flag |= CACHE_LOCAL_MTX | CACHE_WORLD_MTX;
#if NW_G3D_IS_HOST_CAFE
    if (m_pWorldMtxArray == m_pMemWorldMtxArray)
    {
        // ワールド行列の領域をキャッシュに乗せます。サイズはアライメントまで切り上げられます。
        DCZeroRange(m_pWorldMtxArray, sizeof(Mtx34) * GetBoneCount());
    }
#endif
    if (m_pCallback && m_CallbackBone != ICalcWorldCallback::INVALID_BONE)
    {
        (this->*Impl::s_pFuncCalcWorldWithCallback[funcIndex])(baseMtx);
    }
    else
    {
        (this->*Impl::s_pFuncCalcWorld[funcIndex])(baseMtx);
    }
}

void SkeletonObj::CalcMtxBlock(int bufferIndex /*= 0*/)
{
    NW_G3D_PERF_LEVEL1_FUNC();

    GfxBuffer& block = GetMtxBlock();
    if (block.GetSize() == 0)
    {
        return; // 更新不要。
    }

    m_Flag |= CACHE_WORLD_MTX;

    // TODO: 最適化
    const s16* pMtxToBoneTable = m_pRes->ref().ofsMtxToBoneTable.to_ptr<s16>();
    const Mtx34* pInvModelMatrixArray = m_pRes->ref().ofsInvModelMatrixArray.to_ptr<Mtx34>();
    int idxMtx = 0;
    Mtx34* pMtxArray = NULL;
    if (m_pLCMtxBlock == NULL)
    {
        // メモリ
        pMtxArray = static_cast<Mtx34*>(block.GetData(bufferIndex));
#if NW_G3D_IS_HOST_CAFE
        size_t blockSize = block.GetSize();
        if (CPUCache::IsValid(pMtxArray, blockSize))
        {
            // HDP 領域はキャッシュレスなので除外する必要がある。
            DCZeroRange(pMtxArray, blockSize);
        }
#endif
    }
    else
    {
        // LockedCache
        pMtxArray = static_cast<Mtx34*>(m_pLCMtxBlock);
    }

    if (IsBlockSwapEnabled())
    {
        // スムーススキン
        int smoothSkinMtxCount = m_pRes->GetSmoothMtxCount();
        for (; idxMtx < smoothSkinMtxCount; ++idxMtx)
        {
            int idxBone = pMtxToBoneTable[idxMtx];
            const Mtx34& worldMtx = m_pWorldMtxArray[idxBone];
            const Mtx34& invModelMtx = pInvModelMatrixArray[idxMtx];
            Mtx34 mtx;
            mtx.Mul(worldMtx, invModelMtx);
            Copy32<true>(&pMtxArray[idxMtx], &mtx, sizeof(Mtx34) >> 2);
        }

        // リジッドスキン
        int mtxCount = m_pRes->GetMtxCount();
        for (; idxMtx < mtxCount; ++idxMtx)
        {
            int idxBone = pMtxToBoneTable[idxMtx];
            Copy32<true>(&pMtxArray[idxMtx], &m_pWorldMtxArray[idxBone], sizeof(Mtx34) >> 2);
        }
    }
    else
    {
        // スムーススキン
        int smoothSkinMtxCount = m_pRes->GetSmoothMtxCount();
        for (; idxMtx < smoothSkinMtxCount; ++idxMtx)
        {
            int idxBone = pMtxToBoneTable[idxMtx];
            const Mtx34& worldMtx = m_pWorldMtxArray[idxBone];
            const Mtx34& invModelMtx = pInvModelMatrixArray[idxMtx];
            pMtxArray[idxMtx].Mul(worldMtx, invModelMtx);
        }

        // リジッドスキン
        int mtxCount = m_pRes->GetMtxCount();
        for (; idxMtx < mtxCount; ++idxMtx)
        {
            int idxBone = pMtxToBoneTable[idxMtx];
            pMtxArray[idxMtx] = m_pWorldMtxArray[idxBone];
        }
    }

    if (m_pLCMtxBlock == NULL)
    {
        // メモリ
        block.DCFlush(bufferIndex);
    }
}

size_t SkeletonObj::LCMountLocalMtx(void* pLC, size_t size, bool load)
{
#if NW_G3D_IS_HOST_CAFE
    NW_G3D_ASSERTMSG(m_pLocalMtxArray == m_pMemLocalMtxArray, "Mounted twice.");
    NW_G3D_ASSERT_ADDR_ALIGNMENT(pLC, CACHE_BLOCK_SIZE);
    size_t sizeLC = CalcLocalMtxLCSize();
    if (size < sizeLC)
    {
        return 0;
    }
    m_pLocalMtxArray = static_cast<LocalMtxData*>(pLC);
    if (load)
    {
        if (m_Flag & MASK_LOCAL_MTX) // キャッシュに乗っていたらフラッシュ。
        {
            DCFlushRange(m_pMemLocalMtxArray, sizeLC);
            m_Flag &= ~MASK_LOCAL_MTX;
        }
        LockedCache::LoadDMABlocks(m_pLocalMtxArray, m_pMemLocalMtxArray, sizeLC >> 5);
        LockedCache::WaitDMAQueue(0);
    }
    else
    {
        m_Flag |= (m_Flag & CACHE_LOCAL_MTX) << 1; // キャッシュを残す場合は不正扱い。
    }
    return size;
#else
    NW_G3D_UNUSED(pLC);
    NW_G3D_UNUSED(size);
    NW_G3D_UNUSED(load);
    return 0;
#endif
}

void SkeletonObj::LCUnmountLocalMtx(bool store)
{
#if NW_G3D_IS_HOST_CAFE
    if (m_pLocalMtxArray == m_pMemLocalMtxArray)
    {
        return;
    }
    if (store)
    {
        size_t sizeLC = CalcLocalMtxLCSize();
        if (m_Flag & INVALID_LOCAL_MTX) // 不正なキャッシュは破棄。
        {
            DCInvalidateRange(m_pMemLocalMtxArray, sizeLC);
            m_Flag &= ~MASK_LOCAL_MTX;
        }
        LockedCache::StoreDMABlocks(m_pMemLocalMtxArray, m_pLocalMtxArray, sizeLC >> 5);
        LockedCache::WaitDMAQueue(0);
    }
    m_Flag &= ~CACHE_LOCAL_MTX; // マウント中に触ってもキャッシュに乗っていない。
    m_pLocalMtxArray = static_cast<LocalMtxData*>(m_pMemLocalMtxArray);
#else
    NW_G3D_UNUSED(store);
#endif
}

size_t SkeletonObj::LCMountWorldMtx(void* pLC, size_t size, bool load)
{
#if NW_G3D_IS_HOST_CAFE
    NW_G3D_ASSERTMSG(m_pWorldMtxArray == m_pMemWorldMtxArray, "Mounted twice.");
    NW_G3D_ASSERT_ADDR_ALIGNMENT(pLC, CACHE_BLOCK_SIZE);
    size_t sizeLC = CalcWorldMtxLCSize();
    if (size < sizeLC)
    {
        return 0;
    }
    m_pWorldMtxArray = static_cast<Mtx34*>(pLC);
    if (load)
    {
        if (m_Flag & MASK_WORLD_MTX) // キャッシュに乗っていたらフラッシュ。
        {
            DCFlushRange(m_pMemWorldMtxArray, sizeLC);
            m_Flag &= ~MASK_WORLD_MTX;
        }
        LockedCache::LoadDMABlocks(m_pWorldMtxArray, m_pMemWorldMtxArray, sizeLC >> 5);
        LockedCache::WaitDMAQueue(0);
    }
    else
    {
        m_Flag |= (m_Flag & CACHE_WORLD_MTX) << 1; // キャッシュを残す場合は不正扱い。
    }
    return size;
#else
    NW_G3D_UNUSED(pLC);
    NW_G3D_UNUSED(size);
    NW_G3D_UNUSED(load);
    return 0;
#endif
}

void SkeletonObj::LCUnmountWorldMtx(bool store)
{
#if NW_G3D_IS_HOST_CAFE
    if (m_pWorldMtxArray == m_pMemWorldMtxArray)
    {
        return;
    }
    if (store)
    {
        size_t sizeLC = CalcWorldMtxLCSize();
        if (m_Flag & INVALID_WORLD_MTX) // 不正なキャッシュは破棄。
        {
            DCInvalidateRange(m_pMemWorldMtxArray, sizeLC);
            m_Flag &= ~MASK_WORLD_MTX;
        }
        LockedCache::StoreDMABlocks(m_pMemWorldMtxArray, m_pWorldMtxArray, sizeLC >> 5);
        LockedCache::WaitDMAQueue(0);
    }
    m_Flag &= ~CACHE_WORLD_MTX; // マウント中に触ってもキャッシュに乗っていない。
    m_pWorldMtxArray = static_cast<Mtx34*>(m_pMemWorldMtxArray);
#else
    NW_G3D_UNUSED(store);
#endif
}

size_t SkeletonObj::LCMountMtxBlock(void* pLC, size_t size, bool load, int bufferIndex /*= 0*/)
{
#if NW_G3D_IS_HOST_CAFE
    NW_G3D_ASSERTMSG(m_pLCMtxBlock == NULL, "Mounted twice.");
    NW_G3D_ASSERT_ADDR_ALIGNMENT(pLC, CACHE_BLOCK_SIZE);
    size_t sizeLC = CalcMtxBlockLCSize();
    if (sizeLC == 0 || size < sizeLC) // 行列パレットは存在しないことがある。
    {
        return 0;
    }
    if (load)
    {
        LockedCache::LoadDMABlocks(pLC, m_MtxBlock.GetData(bufferIndex), sizeLC >> 5);
        LockedCache::WaitDMAQueue(0);
    }
    m_pLCMtxBlock = pLC;
    return size;
#else
    NW_G3D_UNUSED(pLC);
    NW_G3D_UNUSED(size);
    NW_G3D_UNUSED(load);
    NW_G3D_UNUSED(bufferIndex);
    return 0;
#endif
}

void SkeletonObj::LCUnmountMtxBlock(bool store, int bufferIndex /*= 0*/)
{
#if NW_G3D_IS_HOST_CAFE
    if (m_pLCMtxBlock == NULL)
    {
        return;
    }
    void* pLC = m_pLCMtxBlock;
    m_pLCMtxBlock = NULL;
    if (store)
    {
        size_t sizeLC = CalcMtxBlockLCSize();
        LockedCache::StoreDMABlocks(m_MtxBlock.GetData(bufferIndex), pLC, sizeLC >> 5);
        LockedCache::WaitDMAQueue(0);
    }
#else
    NW_G3D_UNUSED(store);
    NW_G3D_UNUSED(bufferIndex);
#endif
}

//--------------------------------------------------------------------------------------------------

template<typename CalcType, bool useCallback>
void SkeletonObj::CalcWorldImpl(const Mtx34& baseMtx)
{
    int numBone = m_NumBone;
    NW_G3D_ASSERT(numBone > 0);

    CalcType op(GetLocalMtxArray(), m_pWorldMtxArray, m_pScaleArray);

    // ベース変換の都合上、ルートボーンは別処理にする。
    op.CalcRootBone(baseMtx);
    int idxBone = 0;
    ICalcWorldCallback::CallbackArg callbackArg(idxBone, m_CallbackBone);

    if (NW_G3D_STATIC_CONDITION(useCallback) && callbackArg.GetCallbackBoneIndex() == idxBone)
    {
        WorldMtxManip manip(
            &m_pWorldMtxArray[idxBone], op.GetScale(idxBone), &m_pLocalMtxArray[idxBone].flag);
        m_pCallback->Exec(callbackArg, manip);
    }

    for (++idxBone; idxBone < numBone; ++idxBone)
    {
        ResBone* pBone = GetBone(idxBone);

        // モード毎の処理
        op.CalcBone(pBone);

        if (NW_G3D_STATIC_CONDITION(useCallback) && callbackArg.GetCallbackBoneIndex() == idxBone)
        {
            WorldMtxManip manip(
                &m_pWorldMtxArray[idxBone], op.GetScale(idxBone), &m_pLocalMtxArray[idxBone].flag);
            m_pCallback->Exec(callbackArg, manip);
        }
    }

    op.CalcScale(numBone);
}

// CalcWorld の雛形クラス
class CalcWorldBase
{
public:
    CalcWorldBase(LocalMtx* pLocalMtxArray, Mtx34* pWorldMtxArray, Vec3* pScaleArray)
        : m_pLocalMtxArray(pLocalMtxArray)
        , m_pWorldMtxArray(pWorldMtxArray)
        , m_pScaleArray(pScaleArray)
    {
    }

    //! ルートボーン以外のワールド計算
    NW_G3D_FORCE_INLINE void CalcBone( const ResBone* /*bone*/) { NW_G3D_ASSERT(false); }

    //! ルートボーンのワールド計算
    NW_G3D_FORCE_INLINE
    void CalcRootBone(const Mtx34& baseMtx)
    {
        LocalMtx& local = m_pLocalMtxArray[0];
        if (CheckFlag(local.flag, ResBone::ROTTRANS_ZERO))
        {
            m_pWorldMtxArray[0].Set(baseMtx);
        }
        else
        {
            m_pWorldMtxArray[0].Mul(baseMtx, local.mtxRT);
        }

        // baseMtx は単位スケール扱いとする。
        AccumulateFlag(local.flag, ResBone::SCALE_ONE | ResBone::HI_SCALE_ONE);
    }

    //! 各ボーンへのスケールの反映
    NW_G3D_FORCE_INLINE
    void CalcScale(int numBone)
    {
        for (int idxBone = 0; idxBone < numBone; ++idxBone)
        {
            const LocalMtx& local = m_pLocalMtxArray[idxBone];
            if (!CheckFlag(local.flag, ResBone::SCALE_ONE))
            {
                Mtx34& worldMtx = m_pWorldMtxArray[idxBone];
                worldMtx.ScaleBases(worldMtx, local.scale);
            }
        }
    }

    //! スケールの取得
    NW_G3D_FORCE_INLINE
    Vec3* GetScale(int idxBone)
    {
        return &m_pLocalMtxArray[idxBone].scale;
    }

protected:
    //! SRT のフラグを累積する
    NW_G3D_FORCE_INLINE void AccumulateFlag(bit32& flag, bit32 parentFlag)
    {
        flag &= ~ResBone::HI_IDENTITY;
        flag |= (parentFlag & (flag << ResBone::HIERARCHY_SHIFT)) &
            static_cast<bit32>(ResBone::HI_IDENTITY);
    }

    LocalMtx* m_pLocalMtxArray;
    Mtx34* m_pWorldMtxArray;
    Vec3* m_pScaleArray;
};

//--------------------------------------------------------------------------------------------------

class CalcWorldNoScale : public CalcWorldBase
{
public:
    CalcWorldNoScale(LocalMtx* pLocalMtxArray, Mtx34* pWorldMtxArray, Vec3* pScaleArray)
        : CalcWorldBase(pLocalMtxArray, pWorldMtxArray, pScaleArray)
    {
    }

    NW_G3D_FORCE_INLINE
    void CalcBone(const ResBone* bone)
    {
        NW_G3D_PERF_LEVEL2_FUNC();
        int index = bone->GetIndex();
        int parentIndex = bone->GetParentIndex();
        Mtx34& worldMtx = m_pWorldMtxArray[index];
        LocalMtx& local = m_pLocalMtxArray[index];
        const Mtx34& parentMtx = m_pWorldMtxArray[parentIndex];
        const LocalMtx& parent = m_pLocalMtxArray[parentIndex];
        AccumulateFlag(local.flag, parent.flag);

        if (CheckFlag(local.flag, ResBone::ROTTRANS_ZERO))
        {
            worldMtx.Set(parentMtx);
        }
        else
        {
            worldMtx.Mul(parentMtx, local.mtxRT);
        }
    }

    NW_G3D_FORCE_INLINE
    void CalcScale(int numBone)
    {
        (void)numBone;
    }

    //! スケールの取得
    NW_G3D_FORCE_INLINE
    Vec3* GetScale(int idxBone)
    {
        (void)idxBone;
        return NULL;
    }
};

class CalcWorldStd : public CalcWorldBase
{
public:
    CalcWorldStd(LocalMtx* pLocalMtxArray, Mtx34* pWorldMtxArray, Vec3* pScaleArray)
        : CalcWorldBase(pLocalMtxArray, pWorldMtxArray, pScaleArray)
    {
    }

    NW_G3D_FORCE_INLINE
    void CalcBone(const ResBone* bone)
    {
        NW_G3D_PERF_LEVEL2_FUNC();
        int index = bone->GetIndex();
        int parentIndex = bone->GetParentIndex();
        Mtx34& worldMtx = m_pWorldMtxArray[index];
        LocalMtx& local = m_pLocalMtxArray[index];
        const Mtx34& parentMtx = m_pWorldMtxArray[parentIndex];
        const LocalMtx& parent = m_pLocalMtxArray[parentIndex];
        AccumulateFlag(local.flag, parent.flag);

        if (CheckFlag(parent.flag, ResBone::SCALE_ONE))
        {
            if (CheckFlag(local.flag, ResBone::ROTTRANS_ZERO))
            {
                worldMtx.Set(parentMtx);
            }
            else
            {
                worldMtx.Mul(parentMtx, local.mtxRT);
            }
        }
        else
        {
            worldMtx.ScaleBases(parentMtx, parent.scale);
            if (!CheckFlag(local.flag, ResBone::ROTTRANS_ZERO))
            {
                worldMtx.Mul(worldMtx, local.mtxRT);
            }
        }
    }
};

class CalcWorldMaya : public CalcWorldBase
{
public:
    CalcWorldMaya(LocalMtx* pLocalMtxArray, Mtx34* pWorldMtxArray, Vec3* pScaleArray)
        : CalcWorldBase(pLocalMtxArray, pWorldMtxArray, pScaleArray)
    {
    }

    NW_G3D_FORCE_INLINE
    void CalcBone(const ResBone* bone)
    {
        NW_G3D_PERF_LEVEL2_FUNC();
        int index = bone->GetIndex();
        int parentIndex = bone->GetParentIndex();
        Mtx34& worldMtx = m_pWorldMtxArray[index];
        LocalMtx& local = m_pLocalMtxArray[index];
        const Mtx34& parentMtx = m_pWorldMtxArray[parentIndex];
        const LocalMtx& parent = m_pLocalMtxArray[parentIndex];
        AccumulateFlag(local.flag, parent.flag);

        if (CheckFlag(local.flag, ResBone::ROTTRANS_ZERO))
        {
            if ((local.flag & ResBone::SEGMENT_SCALE_COMPENSATE) ||
                CheckFlag(parent.flag, ResBone::SCALE_ONE))
            {
                worldMtx.Set(parentMtx); // T がゼロなので、SSC でなければスケールの処理も省略する。
            }
            else
            {
                worldMtx.ScaleBases(parentMtx, parent.scale);
            }
        }
        else
        {
            if (CheckFlag(parent.flag, ResBone::SCALE_ONE))
            {
                worldMtx.Mul(parentMtx, local.mtxRT);
            }
            else if (local.flag & ResBone::SEGMENT_SCALE_COMPENSATE)
            {
                worldMtx.Set(local.mtxRT);
                worldMtx.m03 *= parent.scale.x;
                worldMtx.m13 *= parent.scale.y;
                worldMtx.m23 *= parent.scale.z;
                worldMtx.Mul(parentMtx, worldMtx);
            }
            else
            {
                worldMtx.ScaleBases(parentMtx, parent.scale);
                worldMtx.Mul(worldMtx, local.mtxRT);
            }
        }
    }
};

class CalcWorldSoftimage : public CalcWorldBase
{
public:
    CalcWorldSoftimage(
        LocalMtx* pLocalMtxArray, Mtx34* pWorldMtxArray, Vec3* pScaleArray)
        : CalcWorldBase(pLocalMtxArray, pWorldMtxArray, pScaleArray)
    {
    }

    NW_G3D_FORCE_INLINE
    void CalcRootBone(const Mtx34& baseMtx)
    {
        // baseMtx は Hierarchical Scaling ではなく標準の SRT として扱う。
        // この挙動が問題となる場合はルートボーンを baseMtx の代用として回避する。

        CalcWorldBase::CalcRootBone(baseMtx);
        LocalMtx& local = m_pLocalMtxArray[0];
        if (CheckFlag(local.flag, ResBone::SCALE_ONE))
        {
            m_pScaleArray[0].Set(1.0f, 1.0f, 1.0f);
        }
        else
        {
            m_pScaleArray[0].Set(local.scale);
        }
    }

    NW_G3D_FORCE_INLINE
    void CalcBone(const ResBone* bone)
    {
        NW_G3D_PERF_LEVEL2_FUNC();
        int index = bone->GetIndex();
        int parentIndex = bone->GetParentIndex();
        Vec3& scale = m_pScaleArray[index];
        Mtx34& worldMtx = m_pWorldMtxArray[index];
        LocalMtx& local = m_pLocalMtxArray[index];
        const Vec3& parentScale = m_pScaleArray[parentIndex];
        const Mtx34& parentMtx = m_pWorldMtxArray[parentIndex];
        const LocalMtx& parent = m_pLocalMtxArray[parentIndex];
        AccumulateFlag(local.flag, parent.flag);

        if (CheckFlag(local.flag, ResBone::ROTTRANS_ZERO))
        {
            // T がゼロなので、スケールの処理も省略する。
            worldMtx.Set(parentMtx);
        }
        else
        {
            if (CheckFlag(parent.flag, ResBone::HI_SCALE_ONE))
            {
                worldMtx.Mul(parentMtx, local.mtxRT);
            }
            else
            {
                worldMtx.Set(local.mtxRT);
                worldMtx.m03 *= parentScale.x;
                worldMtx.m13 *= parentScale.y;
                worldMtx.m23 *= parentScale.z;
                worldMtx.Mul(parentMtx, worldMtx);
            }
        }

        if (CheckFlag(local.flag, ResBone::SCALE_ONE))
        {
            scale.Set(parentScale);
        }
        else
        {
            scale.Mul(parentScale, local.scale);
        }
    }

    NW_G3D_FORCE_INLINE
    void CalcScale(int numBone)
    {
        for (int idxBone = 0; idxBone < numBone; ++idxBone)
        {
            const LocalMtx& local = m_pLocalMtxArray[idxBone];
            if (!CheckFlag(local.flag, ResBone::HI_SCALE_ONE))
            {
                Mtx34& worldMtx = m_pWorldMtxArray[idxBone];
                worldMtx.ScaleBases(worldMtx, m_pScaleArray[idxBone]);
            }
        }
    }

    //! スケールの取得
    NW_G3D_FORCE_INLINE
    Vec3* GetScale(int idxBone)
    {
        return &m_pScaleArray[idxBone];
    }
};

}} // namespace nw::g3d
