﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/g3d/fnd/g3d_GLUtility.h>
#include <nw/g3d/ut/g3d_Inlines.h>
#include <memory>

#if NW_G3D_IS_HOST_WIN
#ifndef WIN32_LEAN_AND_MEAN
#define WIN32_LEAN_AND_MEAN
#endif
#ifndef NOMINMAX
#define NOMINMAX
#endif
#include <windows.h>
#elif defined(NW_G3D_IS_GL_ES)
#include <nw/g3d/fnd/g3d_EglUtility.h>
#endif

#if !defined( NW_STRIP_GL )

#if defined(__APPLE__)
#define GL_STACK_OVERFLOW   (0x0503)
#define GL_STACK_UNDERFLOW  (0x0504)
#endif

namespace nw { namespace g3d { namespace fnd {

void SetSwapInterval(int interval)
{
#if !defined(NW_G3D_IS_GL_ES)
    typedef BOOL (WINAPI * PFNWGLSWAPINTERVALEXTPROC) (int interval);
    static const PFNWGLSWAPINTERVALEXTPROC wglSwapInterval =
        (PFNWGLSWAPINTERVALEXTPROC)wglGetProcAddress("wglSwapIntervalEXT");
    wglSwapInterval(interval);
#else
#if defined(ANDROID)
    eglSwapInterval(detail::GetEGLDisplay(), interval);
#endif
#endif
}

bool IsScalarType(GLenum type)
{
    switch (type)
    {
    case GL_FLOAT:
#if !defined(NW_G3D_IS_GL_ES)
    case GL_DOUBLE:
#endif
    case GL_INT:
    case GL_UNSIGNED_INT:
    case GL_BOOL:
        return true;
    default:
        return false;
    }
}

bool IsVectorType(GLenum type)
{
    switch (type)
    {
    case GL_FLOAT_VEC2:
    case GL_FLOAT_VEC3:
    case GL_FLOAT_VEC4:
#if !defined(NW_G3D_IS_GL_ES)
    case GL_DOUBLE_VEC2:
    case GL_DOUBLE_VEC3:
    case GL_DOUBLE_VEC4:
#endif
    case GL_INT_VEC2:
    case GL_INT_VEC3:
    case GL_INT_VEC4:
    case GL_UNSIGNED_INT_VEC2:
    case GL_UNSIGNED_INT_VEC3:
    case GL_UNSIGNED_INT_VEC4:
    case GL_BOOL_VEC2:
    case GL_BOOL_VEC3:
    case GL_BOOL_VEC4:
        return true;
    default:
        return false;
    }
}

bool IsMatrixType(GLenum type)
{
    switch (type)
    {
    case GL_FLOAT_MAT2:
    case GL_FLOAT_MAT3:
    case GL_FLOAT_MAT4:
    case GL_FLOAT_MAT2x3:
    case GL_FLOAT_MAT2x4:
    case GL_FLOAT_MAT3x2:
    case GL_FLOAT_MAT3x4:
    case GL_FLOAT_MAT4x2:
    case GL_FLOAT_MAT4x3:
#if !defined(NW_G3D_IS_GL_ES)
    case GL_DOUBLE_MAT2:
    case GL_DOUBLE_MAT3:
    case GL_DOUBLE_MAT4:
#endif
#if 0
    case GL_DOUBLE_MAT2x3:
    case GL_DOUBLE_MAT2x4:
    case GL_DOUBLE_MAT3x2:
    case GL_DOUBLE_MAT3x4:
    case GL_DOUBLE_MAT4x2:
    case GL_DOUBLE_MAT4x3:
#endif
        return true;
    default:
        return false;
    }
}

bool IsSamplerType(GLenum type)
{
    switch (type)
    {
#if !defined(NW_G3D_IS_GL_ES)
    case GL_SAMPLER_1D:
#endif
    case GL_SAMPLER_2D:
    case GL_SAMPLER_3D:
    case GL_SAMPLER_CUBE:
#if !defined(NW_G3D_IS_GL_ES)
    case GL_SAMPLER_1D_SHADOW:
#endif
    case GL_SAMPLER_2D_SHADOW:
#if !defined(NW_G3D_IS_GL_ES)
    case GL_SAMPLER_1D_ARRAY:
#endif
    case GL_SAMPLER_2D_ARRAY:
#if !defined(NW_G3D_IS_GL_ES)
    case GL_SAMPLER_CUBE_MAP_ARRAY:
    case GL_SAMPLER_1D_ARRAY_SHADOW:
#endif
    case GL_SAMPLER_2D_ARRAY_SHADOW:
#if !defined(NW_G3D_IS_GL_ES)
    case GL_SAMPLER_2D_MULTISAMPLE:
    case GL_SAMPLER_2D_MULTISAMPLE_ARRAY:
#endif
    case GL_SAMPLER_CUBE_SHADOW:
#if !defined(NW_G3D_IS_GL_ES)
    case GL_SAMPLER_CUBE_MAP_ARRAY_SHADOW:
    case GL_SAMPLER_BUFFER:
    case GL_SAMPLER_2D_RECT:
    case GL_SAMPLER_2D_RECT_SHADOW:
    case GL_INT_SAMPLER_1D:
#endif
    case GL_INT_SAMPLER_2D:
    case GL_INT_SAMPLER_3D:
    case GL_INT_SAMPLER_CUBE:
#if !defined(NW_G3D_IS_GL_ES)
    case GL_INT_SAMPLER_1D_ARRAY:
#endif
    case GL_INT_SAMPLER_2D_ARRAY:
#if !defined(NW_G3D_IS_GL_ES)
    case GL_INT_SAMPLER_CUBE_MAP_ARRAY:
    case GL_INT_SAMPLER_2D_MULTISAMPLE:
    case GL_INT_SAMPLER_2D_MULTISAMPLE_ARRAY:
    case GL_INT_SAMPLER_BUFFER:
    case GL_INT_SAMPLER_2D_RECT:
    case GL_UNSIGNED_INT_SAMPLER_1D:
#endif
    case GL_UNSIGNED_INT_SAMPLER_2D:
    case GL_UNSIGNED_INT_SAMPLER_3D:
    case GL_UNSIGNED_INT_SAMPLER_CUBE:
#if !defined(NW_G3D_IS_GL_ES)
    case GL_UNSIGNED_INT_SAMPLER_1D_ARRAY:
#endif
    case GL_UNSIGNED_INT_SAMPLER_2D_ARRAY:
#if !defined(NW_G3D_IS_GL_ES)
    case GL_UNSIGNED_INT_SAMPLER_CUBE_MAP_ARRAY:
    case GL_UNSIGNED_INT_SAMPLER_2D_MULTISAMPLE:
    case GL_UNSIGNED_INT_SAMPLER_2D_MULTISAMPLE_ARRAY:
    case GL_UNSIGNED_INT_SAMPLER_BUFFER:
    case GL_UNSIGNED_INT_SAMPLER_2D_RECT:
#endif
        return true;
    default:
        return false;
    }
}

#define TYPE_CASE(err) case err: return #err;

const char* GetGLErrorString(GLenum err)
{
    switch (err)
    {
        TYPE_CASE(GL_NO_ERROR);
        TYPE_CASE(GL_INVALID_ENUM);
        TYPE_CASE(GL_INVALID_VALUE);
        TYPE_CASE(GL_INVALID_OPERATION);
        TYPE_CASE(GL_STACK_OVERFLOW);
        TYPE_CASE(GL_STACK_UNDERFLOW);
        TYPE_CASE(GL_OUT_OF_MEMORY);
        TYPE_CASE(GL_INVALID_FRAMEBUFFER_OPERATION);
    default:
        return "Unkown Error";
    }
}

const char* GetGLDebugString(GLenum debug)
{
    switch (debug)
    {
#if !defined(NW_G3D_IS_GL_ES)
        TYPE_CASE(GL_DEBUG_SOURCE_API_ARB);
        TYPE_CASE(GL_DEBUG_SOURCE_WINDOW_SYSTEM_ARB);
        TYPE_CASE(GL_DEBUG_SOURCE_SHADER_COMPILER_ARB);
        TYPE_CASE(GL_DEBUG_SOURCE_THIRD_PARTY_ARB);
        TYPE_CASE(GL_DEBUG_SOURCE_APPLICATION_ARB);
        TYPE_CASE(GL_DEBUG_SOURCE_OTHER_ARB);
        TYPE_CASE(GL_DEBUG_TYPE_ERROR_ARB);
        TYPE_CASE(GL_DEBUG_TYPE_DEPRECATED_BEHAVIOR_ARB);
        TYPE_CASE(GL_DEBUG_TYPE_UNDEFINED_BEHAVIOR_ARB);
        TYPE_CASE(GL_DEBUG_TYPE_PORTABILITY_ARB);
        TYPE_CASE(GL_DEBUG_TYPE_PERFORMANCE_ARB);
        TYPE_CASE(GL_DEBUG_TYPE_OTHER_ARB);
        TYPE_CASE(GL_DEBUG_LOGGED_MESSAGES_ARB);
        TYPE_CASE(GL_DEBUG_SEVERITY_HIGH_ARB);
        TYPE_CASE(GL_DEBUG_SEVERITY_MEDIUM_ARB);
        TYPE_CASE(GL_DEBUG_SEVERITY_LOW_ARB);
#endif
    default:
        return "Unknown Debug";
    }
}

const char* GetGLTypeString(GLenum type)
{
    switch (type)
    {
        TYPE_CASE(GL_NONE);
        TYPE_CASE(GL_FLOAT);
        TYPE_CASE(GL_FLOAT_VEC2);
        TYPE_CASE(GL_FLOAT_VEC3);
        TYPE_CASE(GL_FLOAT_VEC4);
#if !defined(NW_G3D_IS_GL_ES)
        TYPE_CASE(GL_DOUBLE);
        TYPE_CASE(GL_DOUBLE_VEC2);
        TYPE_CASE(GL_DOUBLE_VEC3);
        TYPE_CASE(GL_DOUBLE_VEC4);
#endif
        TYPE_CASE(GL_INT);
        TYPE_CASE(GL_INT_VEC2);
        TYPE_CASE(GL_INT_VEC3);
        TYPE_CASE(GL_INT_VEC4);
        TYPE_CASE(GL_UNSIGNED_INT);
        TYPE_CASE(GL_UNSIGNED_INT_VEC2);
        TYPE_CASE(GL_UNSIGNED_INT_VEC3);
        TYPE_CASE(GL_UNSIGNED_INT_VEC4);
        TYPE_CASE(GL_BOOL);
        TYPE_CASE(GL_BOOL_VEC2);
        TYPE_CASE(GL_BOOL_VEC3);
        TYPE_CASE(GL_BOOL_VEC4);
        TYPE_CASE(GL_FLOAT_MAT2);
        TYPE_CASE(GL_FLOAT_MAT3);
        TYPE_CASE(GL_FLOAT_MAT4);
        TYPE_CASE(GL_FLOAT_MAT2x3);
        TYPE_CASE(GL_FLOAT_MAT2x4);
        TYPE_CASE(GL_FLOAT_MAT3x2);
        TYPE_CASE(GL_FLOAT_MAT3x4);
        TYPE_CASE(GL_FLOAT_MAT4x2);
        TYPE_CASE(GL_FLOAT_MAT4x3);
#if !defined(NW_G3D_IS_GL_ES)
        TYPE_CASE(GL_DOUBLE_MAT2);
        TYPE_CASE(GL_DOUBLE_MAT3);
        TYPE_CASE(GL_DOUBLE_MAT4);
#endif
#if 0
        TYPE_CASE(GL_DOUBLE_MAT2x3);
        TYPE_CASE(GL_DOUBLE_MAT2x4);
        TYPE_CASE(GL_DOUBLE_MAT3x2);
        TYPE_CASE(GL_DOUBLE_MAT3x4);
        TYPE_CASE(GL_DOUBLE_MAT4x2);
        TYPE_CASE(GL_DOUBLE_MAT4x3);
#endif
#if !defined(NW_G3D_IS_GL_ES)
        TYPE_CASE(GL_SAMPLER_1D);
#endif
        TYPE_CASE(GL_SAMPLER_2D);
        TYPE_CASE(GL_SAMPLER_3D);
        TYPE_CASE(GL_SAMPLER_CUBE);
#if !defined(NW_G3D_IS_GL_ES)
        TYPE_CASE(GL_SAMPLER_1D_SHADOW);
#endif
        TYPE_CASE(GL_SAMPLER_2D_SHADOW);
#if !defined(NW_G3D_IS_GL_ES)
        TYPE_CASE(GL_SAMPLER_1D_ARRAY);
#endif
        TYPE_CASE(GL_SAMPLER_2D_ARRAY);
#if !defined(NW_G3D_IS_GL_ES)
        TYPE_CASE(GL_SAMPLER_CUBE_MAP_ARRAY);
        TYPE_CASE(GL_SAMPLER_1D_ARRAY_SHADOW);
#endif
        TYPE_CASE(GL_SAMPLER_2D_ARRAY_SHADOW);
#if !defined(NW_G3D_IS_GL_ES)
        TYPE_CASE(GL_SAMPLER_2D_MULTISAMPLE);
        TYPE_CASE(GL_SAMPLER_2D_MULTISAMPLE_ARRAY);
#endif
        TYPE_CASE(GL_SAMPLER_CUBE_SHADOW);
#if !defined(NW_G3D_IS_GL_ES)
        TYPE_CASE(GL_SAMPLER_CUBE_MAP_ARRAY_SHADOW);
        TYPE_CASE(GL_SAMPLER_BUFFER);
        TYPE_CASE(GL_SAMPLER_2D_RECT);
        TYPE_CASE(GL_SAMPLER_2D_RECT_SHADOW);
        TYPE_CASE(GL_INT_SAMPLER_1D);
#endif
        TYPE_CASE(GL_INT_SAMPLER_2D);
        TYPE_CASE(GL_INT_SAMPLER_3D);
        TYPE_CASE(GL_INT_SAMPLER_CUBE);
#if !defined(NW_G3D_IS_GL_ES)
        TYPE_CASE(GL_INT_SAMPLER_1D_ARRAY);
#endif
        TYPE_CASE(GL_INT_SAMPLER_2D_ARRAY)
#if !defined(NW_G3D_IS_GL_ES)
        TYPE_CASE(GL_INT_SAMPLER_CUBE_MAP_ARRAY);
        TYPE_CASE(GL_INT_SAMPLER_2D_MULTISAMPLE);
        TYPE_CASE(GL_INT_SAMPLER_2D_MULTISAMPLE_ARRAY);
        TYPE_CASE(GL_INT_SAMPLER_BUFFER);
        TYPE_CASE(GL_INT_SAMPLER_2D_RECT);
        TYPE_CASE(GL_UNSIGNED_INT_SAMPLER_1D);
#endif
        TYPE_CASE(GL_UNSIGNED_INT_SAMPLER_2D);
        TYPE_CASE(GL_UNSIGNED_INT_SAMPLER_3D);
        TYPE_CASE(GL_UNSIGNED_INT_SAMPLER_CUBE);
#if !defined(NW_G3D_IS_GL_ES)
        TYPE_CASE(GL_UNSIGNED_INT_SAMPLER_1D_ARRAY);
#endif
        TYPE_CASE(GL_UNSIGNED_INT_SAMPLER_2D_ARRAY);
#if !defined(NW_G3D_IS_GL_ES)
        TYPE_CASE(GL_UNSIGNED_INT_SAMPLER_CUBE_MAP_ARRAY);
        TYPE_CASE(GL_UNSIGNED_INT_SAMPLER_2D_MULTISAMPLE);
        TYPE_CASE(GL_UNSIGNED_INT_SAMPLER_2D_MULTISAMPLE_ARRAY);
        TYPE_CASE(GL_UNSIGNED_INT_SAMPLER_BUFFER);
        TYPE_CASE(GL_UNSIGNED_INT_SAMPLER_2D_RECT);
#endif
    default:
        return "Unkown Type";
    }
}

const char* GetFramebufferStatusString(GLenum status)
{
    switch (status)
    {
        TYPE_CASE(GL_FRAMEBUFFER_COMPLETE);
        TYPE_CASE(GL_FRAMEBUFFER_UNDEFINED);
        TYPE_CASE(GL_FRAMEBUFFER_INCOMPLETE_ATTACHMENT);
        TYPE_CASE(GL_FRAMEBUFFER_INCOMPLETE_MISSING_ATTACHMENT);
#if !defined(NW_G3D_IS_GL_ES)
        TYPE_CASE(GL_FRAMEBUFFER_INCOMPLETE_DRAW_BUFFER);
        TYPE_CASE(GL_FRAMEBUFFER_INCOMPLETE_READ_BUFFER);
#endif
        TYPE_CASE(GL_FRAMEBUFFER_UNSUPPORTED);
        TYPE_CASE(GL_FRAMEBUFFER_INCOMPLETE_MULTISAMPLE);
#if !defined(NW_G3D_IS_GL_ES)
        TYPE_CASE(GL_FRAMEBUFFER_INCOMPLETE_LAYER_TARGETS);
#endif
    default:
        return "Unkown Status";
    }
}

#undef TYPE_CASE

bool CheckShaderStatus(GLuint hShader)
{
    GLint status = 0;
    glGetShaderiv(hShader, GL_COMPILE_STATUS, &status);
    if (!status)
    {
        GLint infoLen = 0;
        glGetShaderiv(hShader, GL_INFO_LOG_LENGTH, &infoLen);
        if (infoLen > 1)
        {
#if NW_G3D_IS_HOST_WIN
            GLchar* str = static_cast<GLchar*>(_malloca(infoLen));
#else
            // 呼び出し戻りで解放される。
            GLchar* str = static_cast<GLchar*>(alloca(infoLen));
#endif
            glGetShaderInfoLog(hShader, infoLen, &infoLen, str);
            NW_G3D_WARNING(0, "%s\n", str);
#if NW_G3D_IS_HOST_WIN
            _freea(str);
#endif
        }
        NW_G3D_WARNING(0, "glCompileShader failed.\n");
        return false;
    }
    return true;
}

bool CheckProgramStatus(GLuint hProgram)
{
    GLint status = 0;
    glGetProgramiv(hProgram, GL_LINK_STATUS, &status);
    if (!status)
    {
        GLint infoLen = 0;
        glGetProgramiv(hProgram, GL_INFO_LOG_LENGTH, &infoLen);
        if (infoLen > 1)
        {
#if NW_G3D_IS_HOST_WIN
            GLchar* str = static_cast<GLchar*>(_malloca(infoLen));
#else
            // 呼び出し戻りで解放される。
            GLchar* str = static_cast<GLchar*>(alloca(infoLen));
#endif
            glGetProgramInfoLog(hProgram, infoLen, &infoLen, str);
            NW_G3D_WARNING(0, "%s\n", str);
#if NW_G3D_IS_HOST_WIN
            _freea(str);
#endif
        }
        NW_G3D_WARNING(0, "glLinkProgram failed.\n");
        return false;
    }
    return true;
}

bool CheckFramebufferStatus(GLuint hFramebuffer)
{
#if NW_G3D_GL_PORTABILITY < NW_G3D_GL_LEVEL2
    GLenum status = glCheckNamedFramebufferStatusEXT(hFramebuffer, GL_FRAMEBUFFER);
#else
    (void)hFramebuffer;
    GLenum status = glCheckFramebufferStatus(GL_FRAMEBUFFER);
#endif
    if (status != GL_FRAMEBUFFER_COMPLETE)
    {
        NW_G3D_WARNING(0, "Framebuffer is incomplete.\n\t%s (0x%x)\n",
            GetFramebufferStatusString(status), status);
        return false;
    }
    return true;
}

GLuint CreateShaderProgram()
{
    return glCreateProgram();
}

void AttachShader(GLuint hProgram, GLenum type, const GLchar* string)
{
    return AttachShader(hProgram, type, &string, 1);
}

void AttachShader(GLuint hProgram, GLenum type, const GLchar** strings, GLsizei numStrings)
{
    GLuint hShader = glCreateShader(type);
    glShaderSource(hShader, numStrings, strings, NULL);
    glCompileShader(hShader);
    bool result = CheckShaderStatus(hShader);
    NW_G3D_ASSERT(result);
    (void)result;
    glAttachShader(hProgram, hShader);
}

void LinkShaderProgram(GLuint hProgram)
{
    glLinkProgram(hProgram);
    bool result = CheckProgramStatus(hProgram);
    NW_G3D_ASSERT(result);
#if !defined(NW_G3D_IS_GL_ES)
    if (result == true)
    {
        DestroyAttachedShader(hProgram);
    }
#else
    NW_G3D_UNUSED(result);
#endif
}

void DestroyAttachedShader(GLuint hProgram)
{
    GLuint hShader;
    GLint count = 0;
    glGetAttachedShaders(hProgram, 1, &count, &hShader);
    while (count > 0)
    {
        glDetachShader(hProgram, hShader);
        glDeleteShader(hShader);
        glGetAttachedShaders(hProgram, 1, &count, &hShader);
    }
}

void DestroyShaderProgram(GLuint hProgram)
{
    DestroyAttachedShader(hProgram);
    glDeleteProgram(hProgram);
}

//--------------------------------------------------------------------------------------------------

namespace detail {

void GL::BindTexture(GLenum unit, GLenum target, GLuint handle)
{
#if NW_G3D_GL_PORTABILITY == NW_G3D_GL_LEVEL0
    glBindMultiTextureEXT(GL_TEXTURE0 + unit, target, handle);
#else
    glActiveTexture(GL_TEXTURE0 + unit);
    glBindTexture(target, handle);
#endif
}

void GL::TexParameter(GLuint hTexture, GLenum target, GLenum name, GLint value)
{
    NW_G3D_GL_BIND_TEXTURE(target, hTexture);
#if NW_G3D_GL_PORTABILITY < NW_G3D_GL_LEVEL2
    glTextureParameteriEXT(hTexture, target, name, value);
#else
    glTexParameteri(target, name, value);
#endif
}

void GL::TexParameter(GLuint hTexture, GLenum target, GLenum name, const GLint* value)
{
    NW_G3D_GL_BIND_TEXTURE(target, hTexture);
#if NW_G3D_GL_PORTABILITY < NW_G3D_GL_LEVEL2
    glTextureParameterivEXT(hTexture, target, name, value);
#else
    glTexParameteriv(target, name, value);
#endif
}

void GL::TexImage(GLuint hTexture, GLenum target, GLint mipLevel, GLint detailFormat,
    GLsizei width, GLsizei height, GLsizei depth, GLenum format, GLenum type,
    GLboolean compressed, GLsizei imageSize, const GLvoid* data)
{
    const int CUBEMAP_SIZE = 6;
    const GLint BORDER_SIZE = 0;
    NW_G3D_GL_BIND_TEXTURE(target, hTexture);
    switch (target)
    {
#if !defined(NW_G3D_IS_GL_ES)
    case GL_TEXTURE_1D:
        if (compressed)
        {
#if NW_G3D_GL_PORTABILITY < NW_G3D_GL_LEVEL2
            glCompressedTextureImage1DEXT(hTexture, target, mipLevel, detailFormat,
                width, BORDER_SIZE,
                imageSize, data);
#else
            glCompressedTexImage1D(target, mipLevel, detailFormat,
                width, BORDER_SIZE, imageSize, data);
#endif
        }
        else
        {
#if NW_G3D_GL_PORTABILITY < NW_G3D_GL_LEVEL2
            glTextureImage1DEXT(hTexture, target, mipLevel, detailFormat,
                width, BORDER_SIZE,
                format, type, data);
#else
            glTexImage1D(target, mipLevel, detailFormat,
                width, BORDER_SIZE, format, type, data);
#endif
        }
        break;
#endif
    case GL_TEXTURE_2D:
#if !defined(NW_G3D_IS_GL_ES)
    case GL_TEXTURE_1D_ARRAY:
#endif
        if (compressed)
        {
#if NW_G3D_GL_PORTABILITY < NW_G3D_GL_LEVEL2
            glCompressedTextureImage2DEXT(hTexture, target, mipLevel, detailFormat,
                width, height, BORDER_SIZE,
                imageSize, data);
#else
            glCompressedTexImage2D(target, mipLevel, detailFormat,
                width, height, BORDER_SIZE, imageSize, data);
#endif
        }
        else
        {
#if NW_G3D_GL_PORTABILITY < NW_G3D_GL_LEVEL2
            glTextureImage2DEXT(hTexture, target, mipLevel, detailFormat,
                width, height, BORDER_SIZE,
                format, type, data);
#else
            glTexImage2D(target, mipLevel, detailFormat,
                width, height, BORDER_SIZE, format, type, data);
#endif
        }
        break;
    case GL_TEXTURE_3D:
    case GL_TEXTURE_2D_ARRAY:
#if !defined(NW_G3D_IS_GL_ES)
    case GL_TEXTURE_CUBE_MAP_ARRAY:
#endif
        if (compressed)
        {
#if NW_G3D_GL_PORTABILITY < NW_G3D_GL_LEVEL2
            glCompressedTextureImage3DEXT(hTexture, target, mipLevel, detailFormat,
                width, height, depth, BORDER_SIZE,
                imageSize, data);
#else
            glCompressedTexImage3D(target, mipLevel, detailFormat,
                width, height, depth, BORDER_SIZE, imageSize, data);
#endif
        }
        else
        {
#if NW_G3D_GL_PORTABILITY < NW_G3D_GL_LEVEL2
            glTextureImage3DEXT(hTexture, target, mipLevel, detailFormat,
                width, height, depth, BORDER_SIZE, format, type, data);
#else
            glTexImage3D(target, mipLevel, detailFormat,
                width, height, depth, BORDER_SIZE, format, type, data);
#endif
        }
        break;
    case GL_TEXTURE_CUBE_MAP:
        if (compressed)
        {
            const GLvoid *pData = data;
            const GLsizei faceSize = imageSize / CUBEMAP_SIZE;
            for (int face = GL_TEXTURE_CUBE_MAP_POSITIVE_X;
                face < GL_TEXTURE_CUBE_MAP_POSITIVE_X + CUBEMAP_SIZE; ++face)
            {
#if NW_G3D_GL_PORTABILITY < NW_G3D_GL_LEVEL2
                glCompressedTextureImage2DEXT(hTexture, face, mipLevel, detailFormat,
                    width, height, BORDER_SIZE, faceSize, pData);
#else
                glCompressedTexImage2D(face, mipLevel, detailFormat,
                    width, height, BORDER_SIZE, faceSize, pData);
#endif
                if (pData)
                {
                    pData = AddOffset(pData, faceSize);
                }
            }
        }
        else
        {
            const GLvoid *pData = data;
            const GLsizei faceSize = imageSize / CUBEMAP_SIZE;
            for (int face = GL_TEXTURE_CUBE_MAP_POSITIVE_X;
                face < GL_TEXTURE_CUBE_MAP_POSITIVE_X + CUBEMAP_SIZE; ++face)
            {
#if NW_G3D_GL_PORTABILITY < NW_G3D_GL_LEVEL2
                glTextureImage2DEXT(hTexture, face, mipLevel, detailFormat,
                    width, height, BORDER_SIZE, format, type, pData);
#else
                glTexImage2D(face, mipLevel, detailFormat,
                    width, height, BORDER_SIZE, format, type, pData);
#endif
                if (pData)
                {
                    pData = AddOffset(pData, faceSize);
                }
            }
        }
        break;
#if !defined(NW_G3D_IS_GL_ES)
    case GL_TEXTURE_2D_MULTISAMPLE:
        NW_G3D_NOT_IMPLEMENTED();
        break;
    case GL_TEXTURE_2D_MULTISAMPLE_ARRAY:
        NW_G3D_NOT_IMPLEMENTED();
        break;
#endif
    default:
        NW_G3D_NOT_SUPPORTED();
    }
}

void GL::TexSubImage(GLuint hTexture, GLenum target, GLint mipLevel, GLint detailFormat,
    GLsizei width, GLsizei height, GLsizei depth, GLenum format, GLenum type,
    GLboolean compressed, GLsizei imageSize, const GLvoid* data)
{
    const int CUBEMAP_SIZE = 6;
    const GLint X_OFFSET = 0;
    const GLint Y_OFFSET = 0;
    const GLint Z_OFFSET = 0;
    NW_G3D_GL_BIND_TEXTURE(target, hTexture);
    switch (target)
    {
#if !defined(NW_G3D_IS_GL_ES)
    case GL_TEXTURE_1D:
        if (compressed)
        {
#if NW_G3D_GL_PORTABILITY < NW_G3D_GL_LEVEL2
            glCompressedTextureSubImage1DEXT(hTexture, target, mipLevel, X_OFFSET,
                width, detailFormat, imageSize, data);
#else
            glCompressedTexSubImage1D(target, mipLevel, X_OFFSET,
                width, detailFormat, imageSize, data);
#endif
        }
        else
        {
#if NW_G3D_GL_PORTABILITY < NW_G3D_GL_LEVEL2
            glTextureSubImage1DEXT(hTexture, target, mipLevel, X_OFFSET,
                width, format, type, data);
#else
            glTexSubImage1D(target, mipLevel, X_OFFSET,
                width, format, type, data);
#endif
        }
        break;
#endif
    case GL_TEXTURE_2D:
#if !defined(NW_G3D_IS_GL_ES)
    case GL_TEXTURE_1D_ARRAY:
#endif
        if (compressed)
        {
#if NW_G3D_GL_PORTABILITY < NW_G3D_GL_LEVEL2
            glCompressedTextureSubImage2DEXT(hTexture, target,
                mipLevel, X_OFFSET, Y_OFFSET,
                width, height, detailFormat, imageSize, data);
#else
            glCompressedTexSubImage2D(target, mipLevel, X_OFFSET, Y_OFFSET,
                width, height, detailFormat, imageSize, data);
#endif
        }
        else
        {
#if NW_G3D_GL_PORTABILITY < NW_G3D_GL_LEVEL2
            glTextureSubImage2DEXT(hTexture, target, mipLevel, X_OFFSET, Y_OFFSET,
                width, height, format, type, data);
#else
            glTexSubImage2D(target, mipLevel, X_OFFSET, Y_OFFSET,
                width, height, format, type, data);
#endif
        }
        break;
    case GL_TEXTURE_3D:
    case GL_TEXTURE_2D_ARRAY:
#if !defined(NW_G3D_IS_GL_ES)
    case GL_TEXTURE_CUBE_MAP_ARRAY:
#endif
        if (compressed)
        {
#if NW_G3D_GL_PORTABILITY < NW_G3D_GL_LEVEL2
            glCompressedTextureSubImage3DEXT(hTexture, target,
                mipLevel, X_OFFSET, Y_OFFSET, Z_OFFSET,
                width, height, depth, detailFormat, imageSize, data);
#else
            glCompressedTexSubImage3D(target,
                mipLevel, X_OFFSET, Y_OFFSET, Z_OFFSET,
                width, height, depth, detailFormat, imageSize, data);
#endif
        }
        else
        {
#if NW_G3D_GL_PORTABILITY < NW_G3D_GL_LEVEL2
            glTextureSubImage3DEXT(hTexture, target,
                mipLevel, X_OFFSET, Y_OFFSET, Z_OFFSET,
                width, height, depth, format, type, data);
#else
            glTexSubImage3D(target,
                mipLevel, X_OFFSET, Y_OFFSET, Z_OFFSET,
                width, height, depth, format, type, data);
#endif
        }
        break;
    case GL_TEXTURE_CUBE_MAP:
        if (compressed)
        {
            const GLvoid *pData = data;
            const GLsizei faceSize = imageSize / CUBEMAP_SIZE;
            for (int face = GL_TEXTURE_CUBE_MAP_POSITIVE_X;
                face < GL_TEXTURE_CUBE_MAP_POSITIVE_X + CUBEMAP_SIZE; ++face)
            {
#if NW_G3D_GL_PORTABILITY < NW_G3D_GL_LEVEL2
                glCompressedTextureSubImage2DEXT(hTexture, face, mipLevel, X_OFFSET, Y_OFFSET,
                    width, height, detailFormat, faceSize, pData);
#else
                glCompressedTexSubImage2D(face, mipLevel, X_OFFSET, Y_OFFSET,
                    width, height, detailFormat, faceSize, pData);
#endif
                if (pData)
                {
                    pData = AddOffset(pData, faceSize);
                }
            }
        }
        else
        {
            const GLvoid *pData = data;
            const GLsizei faceSize = imageSize / CUBEMAP_SIZE;
            for (int face = GL_TEXTURE_CUBE_MAP_POSITIVE_X;
                face < GL_TEXTURE_CUBE_MAP_POSITIVE_X + CUBEMAP_SIZE; ++face)
            {
#if NW_G3D_GL_PORTABILITY < NW_G3D_GL_LEVEL2
                glTextureSubImage2DEXT(hTexture, face, mipLevel, X_OFFSET, Y_OFFSET,
                    width, height, format, type, pData);
#else
                glTexSubImage2D(face, mipLevel, X_OFFSET, Y_OFFSET,
                    width, height, format, type, pData);
#endif
                if (pData)
                {
                    pData = AddOffset(pData, faceSize);
                }
            }
        }
        break;
    default:
        NW_G3D_NOT_SUPPORTED();
    }
}

void GL::GetTexImage(GLuint hTexture, GLenum target, GLint mipLevel, GLenum format, GLenum type,
    GLsizei imageSize, GLvoid* data)
{
    const int CUBEMAP_SIZE = 6;
    NW_G3D_GL_BIND_TEXTURE(target, hTexture);
    switch (target)
    {
#if !defined(NW_G3D_IS_GL_ES)
    case GL_TEXTURE_1D:
    case GL_TEXTURE_2D:
    case GL_TEXTURE_1D_ARRAY:
    case GL_TEXTURE_3D:
    case GL_TEXTURE_2D_ARRAY:
    case GL_TEXTURE_CUBE_MAP_ARRAY:
#if NW_G3D_GL_PORTABILITY < NW_G3D_GL_LEVEL2
        glGetTextureImageEXT(hTexture, target, mipLevel, format, type, data);
#else
        glGetTexImage(target, mipLevel, format, type, data);
#endif
        break;
    case GL_TEXTURE_CUBE_MAP:
        {
            GLvoid *pData = data;
            for (int face = GL_TEXTURE_CUBE_MAP_POSITIVE_X;
                face < GL_TEXTURE_CUBE_MAP_POSITIVE_X + CUBEMAP_SIZE; ++face)
            {
#if NW_G3D_GL_PORTABILITY < NW_G3D_GL_LEVEL2
                glGetTextureImageEXT(hTexture, face, mipLevel, format, type, pData);
#else
                glGetTexImage(face, mipLevel, format, type, pData);
#endif
                if (pData)
                {
                    pData = AddOffset(pData, imageSize / CUBEMAP_SIZE);
                }
            }
        }
        break;
#endif
    default:
        NW_G3D_NOT_SUPPORTED();
    }
}

void GL::EnableVertexAttribArray(GLuint hVertexArray, GLuint location, GLboolean enabled)
{
#if (NW_G3D_GL_PORTABILITY == NW_G3D_GL_LEVEL1) || (NW_G3D_GL_PORTABILITY == NW_G3D_GL_LEVEL2)
    VertexArrayBinder vertexArrayBinder(hVertexArray);
#else
    (void)hVertexArray;
#endif
    if (enabled)
    {
#if NW_G3D_GL_PORTABILITY < NW_G3D_GL_LEVEL1
        glEnableVertexArrayAttribEXT(hVertexArray, location);
#else
        glEnableVertexAttribArray(location);
#endif
    }
    else
    {
#if NW_G3D_GL_PORTABILITY < NW_G3D_GL_LEVEL1
        glDisableVertexArrayAttribEXT(hVertexArray, location);
#else
        glDisableVertexAttribArray(location);
#endif
    }
}

void GL::VertexAttribOffset(GLuint hVertexArray, GLuint hVertexBuffer, GLuint location,
    GLint size, GLenum type, GLboolean integer, GLboolean normalized,
    GLsizei stride, const GLvoid* offset)
{
#if (NW_G3D_GL_PORTABILITY == NW_G3D_GL_LEVEL1) || (NW_G3D_GL_PORTABILITY == NW_G3D_GL_LEVEL2)
    VertexArrayBinder vertexArrayBinder(hVertexArray);
#else
    (void)hVertexArray;
#endif
    if (integer)
    {
#if NW_G3D_GL_PORTABILITY < NW_G3D_GL_LEVEL1
        glVertexArrayVertexAttribIOffsetEXT(hVertexArray, hVertexBuffer, location,
            size, type, stride, reinterpret_cast<GLintptr>(offset));
#else
        BufferBinder bufferBinder(GL_ARRAY_BUFFER, hVertexBuffer);
        glVertexAttribIPointer(location, size, type, stride, offset);
#endif
    }
    else
    {
#if NW_G3D_GL_PORTABILITY < NW_G3D_GL_LEVEL1
        glVertexArrayVertexAttribOffsetEXT(hVertexArray, hVertexBuffer, location,
            size, type, normalized, stride, reinterpret_cast<GLintptr>(offset));
#else
        BufferBinder bufferBinder(GL_ARRAY_BUFFER, hVertexBuffer);
        glVertexAttribPointer(location, size, type, normalized, stride, offset);
#endif
    }
}

void GL::VertexAttribDivisor(GLuint hVertexArray, GLuint location, GLuint divisor)
{
#if NW_G3D_GL_PORTABILITY <= NW_G3D_GL_LEVEL2
    VertexArrayBinder vertexArrayBinder(hVertexArray);
#else
    (void)hVertexArray;
#endif
    glVertexAttribDivisor(location, divisor);
}

void GL::ProgramUniform(GLuint hProgram, GLint location, GLint value)
{
#if NW_G3D_GL_PORTABILITY == NW_G3D_GL_LEVEL0
    glProgramUniform1i(hProgram, location, value);
#elif NW_G3D_GL_PORTABILITY == NW_G3D_GL_LEVEL1
    glProgramUniform1iEXT(hProgram, location, value);
#else
    glUseProgram(hProgram);
    glUniform1i(location, value);
    glUseProgram(0);
#endif
}

void GL::ProgramUniform(GLuint hProgram, GLint location, GLfloat value)
{
#if NW_G3D_GL_PORTABILITY == NW_G3D_GL_LEVEL0
    glProgramUniform1f(hProgram, location, value);
#elif NW_G3D_GL_PORTABILITY == NW_G3D_GL_LEVEL1
    glProgramUniform1fEXT(hProgram, location, value);
#else
    glUseProgram(hProgram);
    glUniform1f(location, value);
    glUseProgram(0);
#endif
}

void GL::Enable(GLenum cap, bool enabled)
{
    if (enabled)
    {
        glEnable(cap);
    }
    else
    {
        glDisable(cap);
    }
}

void GL::Enable(GLenum cap, GLuint targetIndex, bool enabled)
{
#if (NW_G3D_GL_PORTABILITY < NW_G3D_GL_LEVEL3) && !defined(NW_G3D_IS_GL_ES)
    if (enabled)
    {
        glEnablei(cap, targetIndex);
    }
    else
    {
        glDisablei(cap, targetIndex);
    }
#else
    if (targetIndex == 0)
    {
        Enable(cap, enabled);
    }
#endif
}

void GL::BlendEquation(GLuint targetIndex, GLenum modeRGB, GLenum modeAlpha)
{
#if (NW_G3D_GL_PORTABILITY < NW_G3D_GL_LEVEL3) && !defined(NW_G3D_IS_GL_ES)
    glBlendEquationSeparatei(targetIndex, modeRGB, modeAlpha);
#else
    if (targetIndex == 0)
    {
        glBlendEquationSeparate(modeRGB, modeAlpha);
    }
#endif
}

void GL::BlendFunc(GLuint targetIndex,
    GLenum srcRGB, GLenum dstRGB, GLenum srcAlpha, GLenum dstAlpha)
{
#if (NW_G3D_GL_PORTABILITY < NW_G3D_GL_LEVEL3) &&  !defined(NW_G3D_IS_GL_ES)
    glBlendFuncSeparatei(targetIndex, srcRGB, dstRGB, srcAlpha, dstAlpha);
#else
    if (targetIndex == 0)
    {
        glBlendFuncSeparate(srcRGB, dstRGB, srcAlpha, dstAlpha);
    }
#endif
}

void GL::ColorMask(GLuint targetIndex,
    GLboolean red, GLboolean green, GLboolean blue, GLboolean alpha)
{
#if (NW_G3D_GL_PORTABILITY < NW_G3D_GL_LEVEL3) && !defined(NW_G3D_IS_GL_ES)
    glColorMaski(targetIndex, red, green, blue, alpha);
#else
    if (targetIndex == 0)
    {
        glColorMask(red, green, blue, alpha);
    }
#endif
}

void GL::FrameBufferTexture(
    GLuint hFramebuffer, GLenum attachment, GLuint hTexture, GLint mipLevel)
{
    (void)hFramebuffer;
    if (hTexture)
    {
#if NW_G3D_GL_PORTABILITY < NW_G3D_GL_LEVEL2
        glNamedFramebufferTextureEXT(hFramebuffer, attachment, hTexture, mipLevel);
#else
#if !defined(NW_G3D_IS_GL_ES)
        glFramebufferTexture(GL_FRAMEBUFFER, attachment, hTexture, mipLevel);
#else
        // 2Dテクスチャのみ対応
        glFramebufferTexture2D(GL_FRAMEBUFFER, attachment, GL_TEXTURE_2D, hTexture, mipLevel);
#endif
#endif
    }
    else
    {
#if NW_G3D_GL_PORTABILITY < NW_G3D_GL_LEVEL2
        glNamedFramebufferRenderbufferEXT(hFramebuffer, attachment, GL_RENDERBUFFER, 0);
#else
        glFramebufferRenderbuffer(GL_FRAMEBUFFER, attachment, GL_RENDERBUFFER, 0);
#endif
    }
}

void GL::FrameBufferTexture(
    GLuint hFramebuffer, GLenum attachment, GLuint hTexture, GLint mipLevel, GLint layer)
{
    (void)hFramebuffer;
    if (hTexture)
    {
#if NW_G3D_GL_PORTABILITY < NW_G3D_GL_LEVEL2
        glNamedFramebufferTextureLayerEXT(hFramebuffer, attachment, hTexture, mipLevel, layer);
#else
        glFramebufferTextureLayer(GL_FRAMEBUFFER, attachment, hTexture, mipLevel, layer);
#endif
    }
    else
    {
#if NW_G3D_GL_PORTABILITY < NW_G3D_GL_LEVEL2
        glNamedFramebufferRenderbufferEXT(hFramebuffer, attachment, GL_RENDERBUFFER, 0);
#else
        glFramebufferRenderbuffer(GL_FRAMEBUFFER, attachment, GL_RENDERBUFFER, 0);
#endif
    }
}

void GL::FrameBufferDrawBuffers(GLuint hFramebuffer, GLsizei num, const GLenum* bufs)
{
    (void)hFramebuffer;
#if NW_G3D_GL_PORTABILITY < NW_G3D_GL_LEVEL2
    glFramebufferDrawBuffersEXT(hFramebuffer, num, bufs);
#else
    glDrawBuffers(num, bufs);
#endif
}

} // detail

}}} // nw::g3d::fnd

#endif // !defined( NW_STRIP_GL )
