﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "g3d_EditSocket.h"

#if NW_G3D_CONFIG_USE_HOSTIO

#if NW_G3D_IS_HOST_WIN

#include <winsock.h>
#pragma comment(lib, "wsock32.lib ")

#else

namespace {

const u32 HOST_IO_MAX_COUNT = 128; // 最大500バイト内で、32バイトの倍数

} // anonymous namespace

#endif

namespace nw { namespace g3d { namespace edit { namespace detail {

#if NW_G3D_IS_HOST_WIN

struct EditSocket::Impl
{
    SOCKET          socket;
    SOCKET          clientSocket;
    u_short         portNumber;
};

#endif

bool
EditSocket::Open()
{
#if NW_G3D_IS_HOST_WIN
    if (m_Impl->socket != INVALID_SOCKET)
    {
        return false;
    }

    // ソケットの作成
    m_Impl->socket = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
    if (m_Impl->socket == INVALID_SOCKET)
    {
        return false;
    }

    // ノンブロッキングモードに設定
    DWORD  nonBlock = 1;
    ioctlsocket(m_Impl->socket, FIONBIO, &nonBlock);

    // ソケットの設定
    struct sockaddr_in addr;
    addr.sin_family = AF_INET;
    addr.sin_port = htons(m_Impl->portNumber);
    addr.sin_addr.S_un.S_addr = inet_addr("127.0.0.1");
    bind(m_Impl->socket, reinterpret_cast<struct sockaddr*>(&addr), sizeof(addr));

    // TCPクライアントからの接続要求を待てる状態にする
    int listenResult = listen(m_Impl->socket, 5);
    if (listenResult == SOCKET_ERROR)
    {
        NW_G3D_WARNING(false, "Socket open error %d\n", WSAGetLastError());
        closesocket(m_Impl->socket);
        m_Impl->socket = INVALID_SOCKET;
        return false;
    }
#else
    if (m_Handle != HIO_HANDLE_INVALID)
    {
        return false;
    }

    m_Handle = HIOOpen(m_ChannelName, ConnectionCallback, this, HIO_CHANNEL_OPTION_READ_WRITE, 0);
    if (m_Handle == HIO_HANDLE_INVALID)
    {
        NW_G3D_WARNING(false, "HIO Handle open error %d\n", m_Handle);
        return false;
    }

#endif
    return true;
}

void
EditSocket::Close()
{
#if NW_G3D_IS_HOST_WIN
    if (m_Impl->clientSocket != INVALID_SOCKET)
    {
        closesocket(m_Impl->clientSocket);
        m_Impl->clientSocket = INVALID_SOCKET;
    }

    if (m_Impl->socket != INVALID_SOCKET)
    {
        closesocket(m_Impl->socket);
        m_Impl->socket = INVALID_SOCKET;
    }
#else
    if (m_Handle == HIO_HANDLE_INVALID)
    {
        return;
    }

    HIOClose(m_Handle);
    m_Handle = HIO_HANDLE_INVALID;
#endif
    ResetReadFlag();
    ResetWriteFlag();
}

bool
EditSocket::WriteASync(void* buf, u32 size)
{
    NW_G3D_ASSERT_NOT_NULL( buf );
    NW_G3D_ASSERT( size != 0 );

    if (IsWriting())
    {
        return false;
    }

    if (!m_IsWriteEnd)
    {
        return false;
    }

#if NW_G3D_IS_HOST_WIN
    if (m_Impl->clientSocket == INVALID_SOCKET)
    {
        return false;
    }

    m_IsWriteEnd = false;
    m_pWriteBuffer = static_cast<char*>(buf);
    m_WriteBufferSize = size;

    m_WriteSize = size;
    m_TotalWritingSize = m_WritingSize = 0;

    int bufferSize = static_cast<int>(m_WriteBufferSize);
    int result = send(m_Impl->clientSocket, m_pWriteBuffer, bufferSize, 0);
    if (result <= 0)
    {
        // ソケットが使えなくなっている場合は、ソケットを閉じる
        bool isGracefulClose = result == 0;
        if ((WSAGetLastError() != WSAEWOULDBLOCK) || isGracefulClose)
        {
            ResetWriteFlag();
            closesocket(m_Impl->clientSocket);
            m_Impl->clientSocket = INVALID_SOCKET;
            return false;
        }
        result = 0;
    }

    m_WritingSize += static_cast<u32>(result);
    m_TotalWritingSize = m_WritingSize;
#else
    if (m_Handle == HIO_HANDLE_INVALID)
    {
        return false;
    }

    if (!m_IsConnected)
    {
        return false;
    }

    m_IsWritingEnd = false;
    m_IsWriteEnd = false;
    m_pWriteBuffer = buf;
    m_WriteBufferSize = size;

    m_WriteSize = size;
    m_WritingSize = 0;
    m_TotalWritingSize = 0;

    if (m_WriteSize >= HOST_IO_MAX_COUNT)
    {
        m_WriteSize = HOST_IO_MAX_COUNT;
    }

    HIOWriteAsync(m_Handle, m_WriteSize, m_pWriteBuffer, WriteCallback, this);
#endif
    return true;
}

bool
EditSocket::ReadASync(void* buf, u32 size)
{
    NW_G3D_ASSERT_NOT_NULL(buf);
    NW_G3D_ASSERT(size != 0);

    if (IsReading())
    {
        return false;
    }

    if (!m_IsReadEnd)
    {
        return false;
    }
#if NW_G3D_IS_HOST_WIN

    if (m_Impl->clientSocket == INVALID_SOCKET)
    {
        return false;
    }

    m_IsReadEnd = false;
    m_pReadBuffer = static_cast<char*>(buf);
    m_ReadBufferSize = size;

    m_ReadSize = size;
    m_TotalReadingSize = m_ReadingSize = 0;

    int result = recv(m_Impl->clientSocket, m_pReadBuffer, m_ReadBufferSize, 0);
    if (result <= 0)
    {
        // ソケットが使えなくなっている場合は、ソケットを閉じる
        bool isGracefulClose = result == 0;
        if ((WSAGetLastError() != WSAEWOULDBLOCK) || isGracefulClose)
        {
            ResetReadFlag();
            closesocket(m_Impl->clientSocket);
            m_Impl->clientSocket = INVALID_SOCKET;
            return false;
        }

        result = 0;
    }

    m_ReadingSize += static_cast<u32>(result);
    m_TotalReadingSize = m_ReadingSize;
#else
    if (m_Handle == HIO_HANDLE_INVALID)
    {
        return false;
    }

    if (!m_IsConnected)
    {
        return false;
    }

    m_IsReadingEnd = false;
    m_IsReadEnd = false;
    m_pReadBuffer = buf;
    m_ReadBufferSize = size;

    m_ReadSize = size;
    m_ReadingSize = 0;
    m_TotalReadingSize = 0;

    if (m_ReadSize >= HOST_IO_MAX_COUNT)
    {
        m_ReadSize = HOST_IO_MAX_COUNT;
    }

    HIOReadAsync(m_Handle, m_ReadSize, m_pReadBuffer, ReadCallback, this);
#endif
    return true;
}

bool
EditSocket::WriteSync(void* buf, u32 size)
{
    bool result = WriteASync(buf, size);

    u32 oldWriteSize = m_WritingSize;

    if (size != oldWriteSize)
    {
        result = false;
    }

    do
    {
        if (!IsConnected())
        {
            break;
        }

        WriteASyncInternal();

        if (!IsWriting())
        {
            result = true;
            break;
        }
    } while (!result);

    u32 writeSize = m_WritingSize;
    if (size != writeSize)
    {
        result = false;
    }
    return result;
}

bool
EditSocket::IsWriting() const
{
    return m_WriteBufferSize > m_TotalWritingSize;
}

bool
EditSocket::IsReading() const
{
    return m_ReadBufferSize > m_TotalReadingSize;
}

void
EditSocket::ResetReadFlag()
{
    m_TotalReadingSize = m_ReadSize = m_ReadingSize = 0;
    m_IsReadEnd = true;
    m_ReadBufferSize = 0;
    m_pReadBuffer = NULL;
}

void
EditSocket::ResetWriteFlag()
{
    m_TotalWritingSize = m_WriteSize = m_WritingSize = 0;
    m_IsWriteEnd = true;
    m_WriteBufferSize = 0;
    m_pWriteBuffer = NULL;
}

void
EditSocket::Poll()
{
#if NW_G3D_IS_HOST_WIN
    // ソケットが無効ならば、処理を抜ける
    if (m_Impl->socket == INVALID_SOCKET)
    {
        return;
    }

    // クライアントの接続ができている場合は処理を抜ける
    if (m_Impl->clientSocket != INVALID_SOCKET)
    {
        ReadASyncInternal();
        WriteASyncInternal();
        return;
    }

    // TCPクライアントからの接続要求を受け付ける
    struct sockaddr_in client;
    int len = sizeof(client);
    SOCKET clientSocket = accept(m_Impl->socket, reinterpret_cast<struct sockaddr*>(&client), &len);
    if (clientSocket != INVALID_SOCKET)
    {
        m_Impl->clientSocket = clientSocket;
    }
#else
    // ハンドルが無効ならば、処理を抜ける
    if (m_Handle == HIO_HANDLE_INVALID)
    {
        return;
    }

    if (m_IsChangedToNoConnection)
    {
        // 接続が途絶えたときにHIOReadAsyncのコールバックが呼ばれなくなり、処理が進まなくなることがある
        // TODO: クローズしてオープンすると回避できたので、正しい対処がわかったら置き換える
        Close();
        Open();
        m_IsChangedToNoConnection = false;
    }

    ReadASyncInternal();
    WriteASyncInternal();
#endif
}

bool
EditSocket::IsConnected() const
{
#if NW_G3D_IS_HOST_WIN

    return m_Impl->clientSocket != INVALID_SOCKET;

#else

    return m_IsConnected;

#endif
}

void
EditSocket::ReadASyncInternal()
{
    // 読み込みが終了している場合は処理を抜ける
    if (m_IsReadEnd)
    {
        return;
    }

#if NW_G3D_IS_HOST_WIN
    NW_G3D_ASSERT(m_Impl->clientSocket != INVALID_SOCKET);

    int bufferSize = static_cast<int>(m_ReadBufferSize - m_ReadingSize);
    int result = 0;
    if (bufferSize > 0)
    {
        result = recv(m_Impl->clientSocket, m_pReadBuffer + m_ReadingSize, bufferSize, 0);
        if (result <= 0)
        {
            // ソケットが使えなくなっている場合は、ソケットを閉じる
            bool isGracefulClose = result == 0;
            int errorCode = WSAGetLastError();
            if ((errorCode != WSAEWOULDBLOCK) || isGracefulClose)
            {
                ResetReadFlag();
                closesocket(m_Impl->clientSocket);
                m_Impl->clientSocket = INVALID_SOCKET;
            }

            return;
        }
    }

    m_ReadingSize += static_cast<u32>(result);
    m_TotalReadingSize += m_ReadingSize;
    if (m_ReadBufferSize <= m_TotalReadingSize)
    {
        m_IsReadEnd = true;
    }
#else

    NW_G3D_ASSERT(m_Handle != HIO_HANDLE_INVALID);

    // 分割読み込みが終了していないので、処理を抜ける
    if (!m_IsReadingEnd)
    {
        return;
    }

    m_ReadSize = m_ReadBufferSize - m_TotalReadingSize;
    if (m_ReadSize >= HOST_IO_MAX_COUNT)
    {
        m_ReadSize = HOST_IO_MAX_COUNT;
    }

    // 指定サイズをすべて読み込み済みならば、処理を抜ける
    if (m_ReadSize == 0)
    {
        m_IsReadEnd = true;
        m_IsReadingEnd = true;
        return;
    }

    m_IsReadingEnd = false;

    HIOReadAsync(m_Handle, m_ReadSize, AddOffset(m_pReadBuffer, m_TotalReadingSize), ReadingCallback, this);

#endif
}

void
EditSocket::WriteASyncInternal()
{
    // 書き込みが終了している場合は処理を抜ける
    if (m_IsWriteEnd)
    {
        return;
    }

#if NW_G3D_IS_HOST_WIN
    if (m_Impl->clientSocket == INVALID_SOCKET) // ツール側から通信を切った時の対処
    {
        return;
    }

    int bufferSize = static_cast<int>(m_WriteBufferSize - m_WritingSize);
    int result = 0;

    if (bufferSize > 0)
    {
        result = send(m_Impl->clientSocket, m_pWriteBuffer + m_WritingSize, bufferSize, 0);
        if (result <= 0)
        {
            // ソケットが使えなくなっている場合は、ソケットを閉じる
            bool isGracefulClose = result == 0;
            if ((WSAGetLastError() != WSAEWOULDBLOCK) || isGracefulClose)
            {
                ResetWriteFlag();
                closesocket(m_Impl->clientSocket);
                m_Impl->clientSocket = INVALID_SOCKET;
            }
            return;
        }
    }

    m_WritingSize += static_cast<u32>(result);
    m_TotalWritingSize += m_WritingSize;
    if (m_WriteBufferSize <= m_TotalWritingSize)
    {
        m_IsWriteEnd = true;
    }
#else
    if (m_Handle == HIO_HANDLE_INVALID) // ツール側から通信を切った時の対処
    {
        return;
    }

    // 分割書き込みが終了していないので、処理を抜ける
    if (!m_IsWritingEnd)
    {
        return;
    }

    m_WriteSize = m_WriteBufferSize - m_TotalWritingSize;
    if (m_WriteSize >= HOST_IO_MAX_COUNT)
    {
        m_WriteSize = HOST_IO_MAX_COUNT;
    }

    // 指定サイズをすべて読み込み済みならば、処理を抜ける
    if (m_WriteSize == 0)
    {
        m_IsWriteEnd = true;
        m_IsWritingEnd = true;
        return;
    }

    m_IsWritingEnd = false;

    HIOWriteAsync(m_Handle, m_WriteSize, AddOffset(m_pWriteBuffer, m_TotalWritingSize), WritingCallback, this);
#endif
}

#if NW_G3D_IS_HOST_CAFE

/*static*/void
EditSocket::ConnectionCallback(HIOStatus status, void* context)
{
    EditSocket* socket = static_cast<EditSocket*>( context );
    socket->m_Status = status;

    if (status == HIO_STATUS_NEW_CONNECTION)
    {
        socket->m_IsConnected = true;
        NW_G3D_EDIT_PRINT("HIO_STATUS_NEW_CONNECTION\n");
    }
    else if (status == HIO_STATUS_NO_CONNECTIONS)
    {
        socket->m_IsConnected = false;
        socket->m_IsChangedToNoConnection = true;
        NW_G3D_EDIT_PRINT("HIO_STATUS_NO_CONNECTIONS\n");
    }
}

/*static*/void
EditSocket::WriteCallback(HIOStatus status, void* context)
{
    EditSocket* socket = static_cast<EditSocket*>( context );
    socket->m_WriteStatus = status;

    switch(status)
    {
    case HIO_STATUS_NEW_CONNECTION:
        break;

    case HIO_STATUS_OK:
        break;

    case HIO_STATUS_INVALID_CHANNEL_NAME:
        break;

    case HIO_STATUS_NO_CONNECTIONS:
        break;

    case HIO_STATUS_NO_CHANNELS:
        break;

    case HIO_STATUS_INVALID_HANDLE:
        break;

    case HIO_STATUS_NO_CLIENT_TXN_BUF_AVAILABLE:
        break;

    default:
        if (status > 0)
        {
            u32 size = static_cast<u32>(status);
            socket->m_TotalWritingSize += size;
            socket->m_WritingSize += size;

            if (socket->m_WritingSize >= socket->m_WriteSize)
            {
                if (socket->m_WritingSize >= socket->m_WriteBufferSize)
                {
                    socket->m_IsWriteEnd = true;
                }
                else
                {
                    NW_G3D_EDIT_WARNING(false, "Divide Writing Start %d\n", socket->m_WriteBufferSize);
                    // 分割書き込みを開始
                    socket->m_IsWritingEnd = true;
                }
            }
        }
        break;
    }
}

/*static*/void
EditSocket::WritingCallback(HIOStatus status, void* context)
{
    EditSocket* socket = static_cast<EditSocket*>( context );
    socket->m_WriteStatus = status;

    switch(status)
    {
    case HIO_STATUS_NEW_CONNECTION:
        break;

    case HIO_STATUS_OK:
        break;

    case HIO_STATUS_INVALID_CHANNEL_NAME:
        break;

    case HIO_STATUS_NO_CONNECTIONS:
        break;

    case HIO_STATUS_NO_CHANNELS:
        break;

    case HIO_STATUS_INVALID_HANDLE:
        break;

    case HIO_STATUS_NO_CLIENT_TXN_BUF_AVAILABLE:
        break;

    default:
        if (status >= 0)
        {
            u32 size = static_cast<u32>(status);
            socket->m_WritingSize += size;
            socket->m_TotalWritingSize += size;
            if (socket->m_WritingSize >= socket->m_WriteSize)
            {
                socket->m_IsWritingEnd = true;
                if (socket->m_TotalWritingSize >= socket->m_WriteBufferSize)
                {
                    socket->m_IsWriteEnd = true;
                }
            }
        }
        break;
    }
}

/*static*/void
EditSocket::ReadCallback(HIOStatus status, void* context)
{
    EditSocket* socket = static_cast<EditSocket*>( context );
    socket->m_ReadStatus = status;

    switch(status)
    {
    case HIO_STATUS_NEW_CONNECTION:
        break;

    case HIO_STATUS_OK:
        break;

    case HIO_STATUS_INVALID_CHANNEL_NAME:
        break;

    case HIO_STATUS_NO_CONNECTIONS:
        break;

    case HIO_STATUS_NO_CHANNELS:
        break;

    case HIO_STATUS_INVALID_HANDLE:
        break;

    case HIO_STATUS_NO_CLIENT_TXN_BUF_AVAILABLE:
        break;

    default:
        if (status > 0)
        {
            u32 size = static_cast<u32>(status);;
            socket->m_TotalReadingSize += size;
            socket->m_ReadingSize += size;

            if (socket->m_ReadingSize >= socket->m_ReadSize)
            {
                if (socket->m_ReadingSize >= socket->m_ReadBufferSize)
                {
                    socket->m_IsReadEnd = true;
                }
                else
                {
                    NW_G3D_EDIT_WARNING(false, "Divide Reading Start %d\n", socket->m_ReadBufferSize);
                    // 分割読み込みを開始
                    socket->m_IsReadingEnd = true;
                }
            }
        }
        break;
    }
}

/*static*/void
EditSocket::ReadingCallback(HIOStatus status, void* context)
{
    EditSocket* socket = static_cast<EditSocket*>( context );
    socket->m_ReadStatus = status;

    switch(status)
    {
    case HIO_STATUS_NEW_CONNECTION:
        break;

    case HIO_STATUS_OK:
        break;

    case HIO_STATUS_INVALID_CHANNEL_NAME:
        break;

    case HIO_STATUS_NO_CONNECTIONS:
        break;

    case HIO_STATUS_NO_CHANNELS:
        break;

    case HIO_STATUS_INVALID_HANDLE:
        break;

    case HIO_STATUS_NO_CLIENT_TXN_BUF_AVAILABLE:
        break;

    default:
        if (status >= 0)
        {
            u32 size = static_cast<u32>(status);
            socket->m_ReadingSize += size;
            socket->m_TotalReadingSize += size;
            if (socket->m_ReadingSize >= socket->m_ReadSize)
            {
                socket->m_IsReadingEnd = true;
                if (socket->m_TotalReadingSize >= socket->m_ReadBufferSize)
                {
                    socket->m_IsReadEnd = true;
                }
            }
        }
        break;
    }
}

#endif

EditSocket::EditSocket()
#if NW_G3D_IS_HOST_WIN
: m_Impl(NULL)
#else
: m_Handle(HIO_HANDLE_INVALID)
, m_Status(HIO_STATUS_OK)
, m_IsConnected(false)
, m_ReadStatus(HIO_STATUS_OK)
, m_IsReadingEnd(true)
#endif
, m_pWriteBuffer(NULL)
, m_pReadBuffer(NULL)
, m_WriteBufferSize(0)
, m_ReadBufferSize(0)

, m_WriteSize(0)
, m_WritingSize(0)
, m_TotalWritingSize(0)
, m_IsWriteEnd(true)

, m_ReadSize(0)
, m_ReadingSize(0)
, m_TotalReadingSize(0)
, m_IsReadEnd(true)
{
#if NW_G3D_IS_HOST_WIN
    NW_G3D_ASSERT(sizeof(m_SocketMemberBuffer) >= sizeof(Impl));
    memset(&m_SocketMemberBuffer, 0, sizeof(m_SocketMemberBuffer));
    m_Impl = reinterpret_cast<Impl*>(&m_SocketMemberBuffer);
    m_Impl->socket = INVALID_SOCKET;
    m_Impl->clientSocket = INVALID_SOCKET;
#endif
}

void
EditSocket::Setup(const SetupArg& arg)
{
#if NW_G3D_IS_HOST_WIN
    m_Impl = reinterpret_cast<Impl*>(&m_SocketMemberBuffer);
    memset(&m_SocketMemberBuffer, 0, sizeof(m_SocketMemberBuffer));
    m_Impl->socket = INVALID_SOCKET;
    m_Impl->clientSocket = INVALID_SOCKET;
    m_Impl->portNumber = arg.portNumber;
#else
    m_Handle = HIO_HANDLE_INVALID;
    m_Status = HIO_STATUS_OK;
    m_IsConnected = false;
    m_ReadStatus = HIO_STATUS_OK;
    m_IsWritingEnd = true;
    m_IsReadingEnd = true;
    strncpy(m_ChannelName, arg.channelName, CHANNEL_NAME_LENGTH);
    m_ChannelName[CHANNEL_NAME_LENGTH] = '\0';
#endif

    m_pWriteBuffer = NULL;
    m_pReadBuffer = NULL;
    m_WriteBufferSize = 0;
    m_ReadBufferSize = 0;

    m_WriteSize = 0;
    m_WritingSize = 0;
    m_TotalWritingSize = 0;
    m_IsWriteEnd = true;

    m_ReadSize = 0;
    m_ReadingSize = 0;
    m_TotalReadingSize = 0;
    m_IsReadEnd = true;
}

}}}} // namespace nw::g3d::edit::detail

#endif // NW_G3D_CONFIG_USE_HOSTIO
