﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "g3d_EditRenderInfo.h"

#if NW_G3D_CONFIG_USE_HOSTIO

#include <nw/g3d/ut/g3d_Inlines.h>
#include "g3d_EditSocket.h"
#include "g3d_EditUtility.h"

namespace nw { namespace g3d { namespace edit { namespace detail {

using namespace edit::ut::detail;

bool
EditRenderInfo::PushChoice(const char* labelName, const char* itemName, const char* aliasItemName)
{
    NW_G3D_ASSERT_NOT_NULL(labelName);
    NW_G3D_ASSERT_NOT_NULL(itemName);

    int labelIndex = FindLabelIndex(labelName);
    if (labelIndex == -1)
    {
        if (!PushLabel(labelName, ResRenderInfo::STRING))
        {
            return false;
        }
    }
    labelIndex = FindLabelIndex(labelName);
    NW_G3D_ASSERT(labelIndex >= 0);

    return PushItem(labelIndex, itemName, aliasItemName);
}

bool
EditRenderInfo::PushChoice(const char* labelName, s32 minValue, s32 maxValue)
{
    NW_G3D_ASSERT_NOT_NULL(labelName);

    int labelIndex = FindLabelIndex(labelName);
    if (labelIndex == -1)
    {
        if (!PushLabel(labelName, ResRenderInfo::INT))
        {
            return false;
        }
    }
    labelIndex = FindLabelIndex(labelName);
    NW_G3D_ASSERT(labelIndex >= 0);
    RenderInfoLabel* label = m_Labels.UnsafeAt(labelIndex);

    // タイプが違う場合は不正な処理なので止める
    NW_G3D_ASSERT(label->m_Type == ResRenderInfo::INT);
    label->m_Min.iValue = minValue;
    label->m_Max.iValue = maxValue;
    return true;
}

bool
EditRenderInfo::PushChoice(const char* labelName, f32 minValue, f32 maxValue)
{
    NW_G3D_ASSERT_NOT_NULL(labelName);

    int labelIndex = FindLabelIndex(labelName);
    if (labelIndex == -1)
    {
        if (!PushLabel(labelName, ResRenderInfo::FLOAT))
        {
            return false;
        }
    }
    labelIndex = FindLabelIndex(labelName);
    NW_G3D_ASSERT(labelIndex >= 0);
    RenderInfoLabel* label = m_Labels.UnsafeAt(labelIndex);

    // タイプが違う場合は不正な処理なので止める
    NW_G3D_ASSERT(label->m_Type == ResRenderInfo::FLOAT);
    label->m_Min.fValue = minValue;
    label->m_Max.fValue = maxValue;
    return true;
}

bool
EditRenderInfo::PushDefault(const char* labelName, const char* value)
{
    NW_G3D_ASSERT_NOT_NULL(labelName);
    NW_G3D_ASSERT_NOT_NULL(value);

    int labelIndex = FindLabelIndex(labelName);
    if (labelIndex == -1)
    {
        if (!PushLabel(labelName, ResRenderInfo::STRING))
        {
            return false;
        }
    }
    labelIndex = FindLabelIndex(labelName);
    NW_G3D_ASSERT(labelIndex >= 0);

    return PushValue(labelIndex, value);
}

bool
EditRenderInfo::PushDefault(const char* labelName, s32 value)
{
    NW_G3D_ASSERT_NOT_NULL(labelName);

    int labelIndex = FindLabelIndex(labelName);
    if (labelIndex == -1)
    {
        if (!PushLabel(labelName, ResRenderInfo::INT))
        {
            return false;
        }
    }
    labelIndex = FindLabelIndex(labelName);
    NW_G3D_ASSERT(labelIndex >= 0);

    return PushValue(labelIndex, value);
}

bool
EditRenderInfo::PushDefault(const char* labelName, f32 value)
{
    NW_G3D_ASSERT_NOT_NULL(labelName);

    int labelIndex = FindLabelIndex(labelName);
    if (labelIndex == -1)
    {
        if (!PushLabel(labelName, ResRenderInfo::FLOAT))
        {
            return false;
        }
    }
    labelIndex = FindLabelIndex(labelName);
    NW_G3D_ASSERT(labelIndex >= 0);

    return PushValue(labelIndex, value);
}

void
EditRenderInfo::Clear()
{
    {
        int size = m_Values.Size();
        for (int i = 0; i < size; ++i)
        {
            RenderInfoValue* value = m_Values.UnsafeAt(i);
            value->~RenderInfoValue();
            m_pAllocator->Free(value);
        }
        m_Values.Clear();
    }

    {
        int size = m_Items.Size();
        for (int i = 0;i < size; ++i)
        {
            RenderInfoItem* item = m_Items.UnsafeAt(i);
            item->~RenderInfoItem();
            m_pAllocator->Free(item);
        }
        m_Items.Clear();
    }

    {
        int size = m_Labels.Size();
        for (int i = 0; i < size; ++i)
        {
            RenderInfoLabel* label = m_Labels.UnsafeAt(i);
            label->~RenderInfoLabel();
            m_pAllocator->Free(label);
        }
        m_Labels.Clear();
    }
}

bool
EditRenderInfo::MakeRenderInfoPacket()
{
    size_t bufferSize = CalcWorkBufferSize();
    if (bufferSize == 0)
    {
        return false;
    }

    if (!m_WorkBuffer.Resize(bufferSize))
    {
        return false;
    }

    m_RenderInfoBufferSize = bufferSize;
    MakeRenderInfoBuffer();
    return true;
}

bool
EditRenderInfo::SendRenderInfo(EditSocket* socket)
{
    NW_G3D_ASSERT_NOT_NULL(socket);

    bool success = socket->WriteSync(m_WorkBuffer.GetWorkBufferPtr(), m_RenderInfoBufferSize);

    return success;
}

bool
EditRenderInfo::PushLabel(const char* labelName, ResRenderInfo::Type type)
{
    NW_G3D_ASSERT_NOT_NULL(labelName);
    // 指定したlabelName が存在しない場合に実行するので存在チェックは行わない。

    size_t bufferSize = sizeof(RenderInfoLabel);
    void* buffer = m_pAllocator->Alloc(bufferSize, DEFAULT_ALIGNMENT);
    if (buffer == NULL)
    {
        return false;
    }

    // ラベル登録作業
    {
        RenderInfoLabel* label = new (buffer) RenderInfoLabel(m_pAllocator);
        if (!label->Setup(labelName, type))
        {
            label->~RenderInfoLabel();
            m_pAllocator->Free(label);
            return false;
        }

        if (!m_Labels.PushBack(label))
        {
            label->~RenderInfoLabel();
            m_pAllocator->Free(label);
            return false;
        }
    }
    return true;
}

int
EditRenderInfo::FindLabelIndex(const char* labelName) const
{
    int size = m_Labels.Size();
    for (int i = 0; i < size; ++i)
    {
        RenderInfoLabel* label = m_Labels.UnsafeAt(i);
        if (label->IsEqual(labelName))
        {
            return i;
        }
    }
    return -1;
}

bool
EditRenderInfo::PushItem(int labelIndex, const char* itemName, const char* aliasItemName)
{
    if (FindItemIndex(labelIndex, itemName) >= 0)
    {
        return true;
    }

    size_t bufferSize = sizeof(RenderInfoItem);
    void* buffer = m_pAllocator->Alloc(bufferSize, DEFAULT_ALIGNMENT);
    if (buffer == NULL)
    {
        return false;
    }

    RenderInfoItem* item = new (buffer) RenderInfoItem(m_pAllocator);
    if (!item->Setup(labelIndex, itemName, aliasItemName))
    {
        item->~RenderInfoItem();
        m_pAllocator->Free(item);
        return false;
    }

    if (!m_Items.PushBack(item))
    {
        item->~RenderInfoItem();
        m_pAllocator->Free(item);
        return false;
    }
    return true;
}

int
EditRenderInfo::FindItemIndex(int labelIndex, const char* itemName) const
{
    int size = m_Items.Size();
    for (int i = 0; i < size; ++i)
    {
        RenderInfoItem* item = m_Items.UnsafeAt(i);
        if (item->IsEqual(labelIndex, itemName))
        {
            return i;
        }
    }
    return -1;
}

bool
EditRenderInfo::PushValue(int labelIndex, const char* value)
{
    size_t bufferSize = sizeof(RenderInfoValue);
    void* buffer = m_pAllocator->Alloc(bufferSize, DEFAULT_ALIGNMENT);
    if (buffer == NULL)
    {
        return false;
    }

    RenderInfoValue* renderInfoValue = new (buffer) RenderInfoValue(ResRenderInfo::STRING, m_pAllocator);
    if (!renderInfoValue->Setup(labelIndex, value))
    {
        renderInfoValue->~RenderInfoValue();
        m_pAllocator->Free(renderInfoValue);
        return false;
    }

    if (!m_Values.PushBack(renderInfoValue))
    {
        renderInfoValue->~RenderInfoValue();
        m_pAllocator->Free(renderInfoValue);
        return false;
    }
    return true;
}

bool
EditRenderInfo::PushValue(int labelIndex, s32 value)
{
    size_t bufferSize = sizeof(RenderInfoValue);
    void* buffer = m_pAllocator->Alloc(bufferSize, DEFAULT_ALIGNMENT);
    if (buffer == NULL)
    {
        return false;
    }

    RenderInfoValue* renderInfoValue = new (buffer) RenderInfoValue(ResRenderInfo::INT, m_pAllocator);
    if (!renderInfoValue->Setup(labelIndex, value))
    {
        renderInfoValue->~RenderInfoValue();
        m_pAllocator->Free(renderInfoValue);
        return false;
    }

    if (!m_Values.PushBack(renderInfoValue))
    {
        renderInfoValue->~RenderInfoValue();
        m_pAllocator->Free(renderInfoValue);
        return false;
    }
    return true;
}

bool
EditRenderInfo::PushValue(int labelIndex, f32 value)
{
    size_t bufferSize = sizeof(RenderInfoValue);
    void* buffer = m_pAllocator->Alloc(bufferSize, DEFAULT_ALIGNMENT);
    if (buffer == NULL)
    {
        return false;
    }

    RenderInfoValue* renderInfoValue = new (buffer) RenderInfoValue(ResRenderInfo::FLOAT, m_pAllocator);
    if (!renderInfoValue->Setup(labelIndex, value))
    {
        renderInfoValue->~RenderInfoValue();
        m_pAllocator->Free(renderInfoValue);
        return false;
    }

    if (!m_Values.PushBack(renderInfoValue))
    {
        renderInfoValue->~RenderInfoValue();
        m_pAllocator->Free(renderInfoValue);
        return false;
    }
    return true;
}

size_t
EditRenderInfo::CalcWorkBufferSize()
{
    int labelSize = m_Labels.Size();
    int itemSize = m_Items.Size();
    int valueSize = m_Values.Size();

    if (labelSize <= 0)
    {
        return 0;
    }

    size_t size = sizeof(PacketHeader);
    size += sizeof(RenderInfoSendInfo);

    size_t itemOffsetBufferSize = sizeof(RenderInfoChoiceInfo) * itemSize;
    size_t valueOffsetBufferSize = sizeof(u32) * valueSize;
    size += sizeof(RenderInfoLabelInfo) * labelSize;

    size_t itemOffsetArray = size;
    size += itemOffsetBufferSize;

    size_t valueOffsetArray = size;
    size += valueOffsetBufferSize;

    // Choiceの前処理設定
    {
        for (int i = 0; i < itemSize; ++i)
        {
            RenderInfoItem* item = m_Items.UnsafeAt(i);
            RenderInfoLabel* label = m_Labels.UnsafeAt(item->GetLabelIndex());
            item->m_ItemIndex = label->m_ItemNum;
            ++label->m_ItemNum;
        }
    }

    // Defaultの前処理設定
    {
        for (int i = 0; i < valueSize; ++i)
        {
            RenderInfoValue* value = m_Values.UnsafeAt(i);
            RenderInfoLabel* label = m_Labels.UnsafeAt(value->GetLabelIndex());
            value->m_ValueIndex = label->m_ValueNum;
            ++label->m_ValueNum;
        }
    }

    {
        for (int i = 0; i < labelSize; ++i)
        {
            RenderInfoLabel* label = m_Labels.UnsafeAt(i);

            label->m_Offset = size;
            size += label->m_LabelBinarySize;

            label->m_ValueOffset = valueOffsetArray;
            valueOffsetArray += sizeof(u32) * label->m_ValueNum;

            // RenderInfoのタイプが STRING 以外はアイテムが存在しないので処理を飛ばす
            if (label->m_Type != ResRenderInfo::STRING)
            {
                continue;
            }

            label->m_ItemOffset = itemOffsetArray;
            itemOffsetArray += sizeof(RenderInfoChoiceInfo) * label->m_ItemNum;
        }
    }

    // Choice のオフセット決定
    {
        for (int i = 0; i < itemSize; ++i)
        {
            RenderInfoItem* item = m_Items.UnsafeAt(i);
            item->m_Offset = size;
            size += item->m_ItemBinarySize;
        }
    }

    // Choice のAliasオフセット決定
    {
        for (int i = 0; i < itemSize; ++i)
        {
            RenderInfoItem* item = m_Items.UnsafeAt(i);
            if (item->m_AliasBinarySize > 0)
            {
                item->m_AliasOffset = size;
                size += item->m_AliasBinarySize;
            }
        }
    }

    // Default のオフセット決定
    {
        for (int i = 0; i < valueSize; ++i)
        {
            RenderInfoValue* value = m_Values.UnsafeAt(i);
            if (value->m_DefaultBinarySize > 0) // 文字列型のみオフセットを設定
            {
                value->m_Offset = size;
                size += value->m_DefaultBinarySize;
            }
        }
    }

    return size;
}

void
EditRenderInfo::MakeRenderInfoBuffer()
{
    // CalcWorkBufferSize が成功している前提で実行

    void* buffer = m_WorkBuffer.GetWorkBufferPtr();
    NW_G3D_ASSERT_NOT_NULL(buffer);

    PacketHeader* header = static_cast<PacketHeader*>(buffer);
    header->magic = NW_G3D_EDIT_MAGIC;
    header->verWord = NW_G3D_EDIT_VERSION;
    header->command = EDIT_SEND_RENDER_INFO_COMMAND_FLAG;
    header->dataSize = m_RenderInfoBufferSize - sizeof(PacketHeader);

    RenderInfoSendInfo* sendInfo = g3d::ut::AddOffset<RenderInfoSendInfo>(buffer, sizeof(PacketHeader));

    int labelSize = m_Labels.Size();
    int itemSize = m_Items.Size();
    int valueSize = m_Values.Size();

    sendInfo->labelInfoNum = static_cast<u32>(labelSize);
    sendInfo->modelKey = m_ModelKey;
    sendInfo->materialIndex = m_MaterialIndex;
    sendInfo->padding = 0;

    size_t labelInfoSize = sizeof(RenderInfoLabelInfo);
    RenderInfoLabelInfo* firstLabelInfo = g3d::ut::AddOffset<RenderInfoLabelInfo>(sendInfo, sizeof(RenderInfoSendInfo));
    {
        for (int i = 0; i < labelSize; ++i)
        {
            RenderInfoLabelInfo* labelInfo = g3d::ut::AddOffset<RenderInfoLabelInfo>(firstLabelInfo, i * labelInfoSize);
            RenderInfoLabel* label = m_Labels.UnsafeAt(i);

            labelInfo->labelOffset = label->m_Offset;
            labelInfo->renderInfoType = label->m_Type;

            labelInfo->valueNum = label->m_ValueNum;
            labelInfo->valueOffset = label->m_ValueOffset;

            switch(label->m_Type)
            {
            case ResRenderInfo::STRING:
                labelInfo->itemNum = label->m_ItemNum;
                labelInfo->itemOffset = label->m_ItemOffset;
                break;
            case ResRenderInfo::INT:
                labelInfo->iMinValue = label->m_Min.iValue;
                labelInfo->iMaxValue = label->m_Max.iValue;
                break;
            case ResRenderInfo::FLOAT:
                labelInfo->fMinValue = label->m_Min.fValue;
                labelInfo->fMaxValue = label->m_Max.fValue;
                break;

            default:
                NW_G3D_EDIT_UNEXPECTED_DEFAULT;
            }
        }
    }

    // Choice のオフセット配列設定
    {
        for (int i = 0; i < itemSize; ++i)
        {
            RenderInfoItem* item = m_Items.UnsafeAt(i);
            RenderInfoLabel* label = m_Labels.UnsafeAt(item->GetLabelIndex());
            NW_G3D_ASSERT(label->m_Type == ResRenderInfo::STRING);

            RenderInfoChoiceInfo* itemOffsetArray = g3d::ut::AddOffset<RenderInfoChoiceInfo>(buffer, label->m_ItemOffset);
            RenderInfoChoiceInfo* itemOffset = g3d::ut::AddOffset<RenderInfoChoiceInfo>(itemOffsetArray, sizeof(RenderInfoChoiceInfo) * item->m_ItemIndex);
            itemOffset->choiceOffset = item->m_Offset;
            itemOffset->padding = 0;
            if (item->m_AliasBinarySize > 0)
            {
                itemOffset->aliasOffset = item->m_AliasOffset;
                itemOffset->aliasSize = item->m_AliasLength;
            }
            else
            {
                itemOffset->aliasOffset = 0;
                itemOffset->aliasSize = 0;
            }
        }
    }

    // Default のオフセット配列設定
    {
        for (int i = 0; i < valueSize; ++i)
        {
            RenderInfoValue* value = m_Values.UnsafeAt(i);
            RenderInfoLabel* label = m_Labels.UnsafeAt(value->GetLabelIndex());

            if (label->m_ValueNum == 0) // Defaultがない場合はスキップ
            {
                continue;;
            }

            u32* valueOffsetArray = g3d::ut::AddOffset<u32>(buffer, label->m_ValueOffset);
            u32* valueOffset = g3d::ut::AddOffset<u32>(valueOffsetArray, sizeof(u32) * value->m_ValueIndex);
            *valueOffset = value->m_Offset; // union なので、型を無視して値を設定
        }
    }

    {
        for (int i = 0; i < labelSize; ++i)
        {
            RenderInfoLabelInfo* labelInfo = g3d::ut::AddOffset<RenderInfoLabelInfo>(firstLabelInfo, i * labelInfoSize);
            RenderInfoLabel* label = m_Labels.UnsafeAt(i);

            labelInfo->labelOffset = label->m_Offset;// 必要？
            labelInfo->valueNum = label->m_ValueNum;
            labelInfo->valueOffset = label->m_ValueOffset;

            if (label->m_Type != ResRenderInfo::STRING)
            {
                continue;;
            }

            labelInfo->itemNum = label->m_ItemNum;
            labelInfo->itemOffset = label->m_ItemOffset;
        }
    }

    // 描画情報名を出力
    {
        for (int i = 0; i < labelSize; ++i)
        {
            RenderInfoLabel* label = m_Labels.UnsafeAt(i);
            char* name = g3d::ut::AddOffset<char>(buffer, label->m_Offset);
            memcpy(name, label->GetName(), label->GetNameLength());
            name[label->GetNameLength()] = 0;
        }
    }

    // Choiceの文字列を出力
    {
        for (int i = 0; i < itemSize; ++i)
        {
            RenderInfoItem* item = m_Items.UnsafeAt(i);
            char* name = g3d::ut::AddOffset<char>(buffer, item->m_Offset);
            memcpy(name, item->GetName(), item->GetNameLength());
            name[item->GetNameLength()] = 0;
        }
    }

    // Choice の Aliasを出力
    {
        for (int i = 0;i < itemSize; ++i)
        {
            RenderInfoItem* item = m_Items.UnsafeAt(i);
            if (item->m_AliasBinarySize == 0)
            {
                continue;
            }

            u8* alias = g3d::ut::AddOffset<u8>(buffer, item->m_AliasOffset);
            memcpy(alias, item->m_Alias.GetWorkBufferPtr(), item->m_AliasLength);
            alias[item->m_AliasLength] = 0;
        }
    }

    // 文字列型のDefaultの文字列を出力
    {
        for (int i = 0; i < valueSize; ++i)
        {
            RenderInfoValue* value = m_Values.UnsafeAt(i);
            if (value->m_Type != ResRenderInfo::STRING)
            {
                continue;
            }

            char* name = g3d::ut::AddOffset<char>(buffer, value->m_Offset);
            memcpy(name, value->GetStringDefault(), value->GetStringDefaultLength());
            name[value->GetStringDefaultLength()] = 0;
        }
    }
} // NOLINT (readability/fn_size)

bool
EditRenderInfo::RenderInfoLabel::Setup(const char* labelName, ResRenderInfo::Type type)
{
    NW_G3D_ASSERT_NOT_NULL(labelName);
    if (!m_Label.Assign(labelName))
    {
        return false;
    }

    m_LabelBinarySize = nw::g3d::ut::Align(m_Label.GetLength() + 1, DEFAULT_ALIGNMENT);
    m_Type = type;
    return true;
}

bool
EditRenderInfo::RenderInfoLabel::IsEqual(const char* labelName) const
{
    if (strcmp(m_Label.GetStr(), labelName) != 0)
    {
        return false;
    }
    return true;
}

bool
EditRenderInfo::RenderInfoItem::Setup(int labelIndex, const char* itemName, const char* aliasItemName)
{
    NW_G3D_ASSERT(labelIndex >= 0);
    NW_G3D_ASSERT_NOT_NULL(itemName);
    if (!m_Item.Assign(itemName))
    {
        return false;
    }

    if (aliasItemName != NULL)
    {
        size_t length = strlen(aliasItemName);
        m_AliasBinarySize = nw::g3d::ut::Align(length + 1, DEFAULT_ALIGNMENT);
        if (!m_Alias.Resize(m_AliasBinarySize))
        {
            m_Item.Clear();
            return false;
        }
        void* buffer = m_Alias.GetWorkBufferPtr();
        memcpy(buffer, aliasItemName, length + 1);
        m_AliasLength = length;
    }

    m_LabelIndex = labelIndex;
    m_ItemBinarySize = nw::g3d::ut::Align(m_Item.GetLength() + 1, DEFAULT_ALIGNMENT);
    return true;
}

bool
EditRenderInfo::RenderInfoItem::IsEqual(int labelIndex, const char* itemName) const
{
    if (m_LabelIndex != labelIndex)
    {
        return false;
    }
    if (strcmp(m_Item.GetStr(), itemName) != 0)
    {
        return false;
    }
    return true;
}

bool
EditRenderInfo::RenderInfoValue::Setup(int labelIndex, const char* value)
{
    NW_G3D_ASSERT(labelIndex >= 0);
    NW_G3D_ASSERT(m_Type == ResRenderInfo::STRING);// タイプが違う場合は不正な処理なので止める
    NW_G3D_ASSERT_NOT_NULL(value);
    if (!m_DefaultString.Assign(value))
    {
        return false;
    }
    m_Offset = 0;
    m_DefaultBinarySize = nw::g3d::ut::Align(m_DefaultString.GetLength() + 1, DEFAULT_ALIGNMENT);
    m_LabelIndex = labelIndex;
    return true;
}

bool
EditRenderInfo::RenderInfoValue::Setup(int labelIndex, s32 value)
{
    NW_G3D_ASSERT(labelIndex >= 0);
    NW_G3D_ASSERT(m_Type == ResRenderInfo::INT);// タイプが違う場合は不正な処理なので止める
    m_iValue = value;
    m_DefaultBinarySize = 0;
    m_LabelIndex = labelIndex;
    return true;
}

bool
EditRenderInfo::RenderInfoValue::Setup(int labelIndex, f32 value)
{
    NW_G3D_ASSERT(labelIndex >= 0);
    NW_G3D_ASSERT(m_Type == ResRenderInfo::FLOAT);// タイプが違う場合は不正な処理なので止める
    m_fValue = value;
    m_DefaultBinarySize = 0;
    m_LabelIndex = labelIndex;
    return true;
}

}}}} // namespace nw::g3d::edit::detail

#endif // NW_G3D_CONFIG_USE_HOSTIO
