﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "g3d_EditMaterialObj.h"

#if NW_G3D_CONFIG_USE_HOSTIO

#include <nw/g3d/g3d_ModelObj.h>
#include <nw/g3d/g3d_MaterialObj.h>
#include <nw/g3d/res/g3d_ResMaterial.h>
#include <nw/g3d/edit/g3d_IAllocator.h>

namespace nw { namespace g3d { namespace edit { namespace detail {

using namespace ut::detail;

void EditMaterialObj::Setup()
{
    // 初期値を保存
    m_MaterialVisibilityEnable = m_pOwnerModel->IsMatVisible(m_Index);
    m_CullFront = m_pMaterial->GetResRenderState()->GetPolygonCtrl().GetCullFront();
    m_CullBack = m_pMaterial->GetResRenderState()->GetPolygonCtrl().GetCullBack();

    for (int i = 0, end = m_SamplerArray.Size(); i < end; ++i)
    {
        EditSampler* sampler = m_SamplerArray.UnsafeAt(i);
        sampler->Setup();
    }
}

void EditMaterialObj::Cleanup()
{
    m_RenderInfo.Cleanup();

    nw::g3d::MaterialObj::InitArg arg(m_pOriginalMaterial);
    arg.BufferingCount(m_OriginalBufferingCount);

    void* userPtr = m_pMaterial->GetUserPtr();// UserPtr を保持するために一時退避
    m_pOriginalMaterial->Cleanup();
    m_pOriginalMaterial->Setup();

    m_pMaterial->CleanupBlockBuffer();
    m_pMaterial->Init(arg, m_pOriginalBuffer, m_OriginalBufferSize);
    m_pMaterial->SetUserPtr(userPtr);
    m_pMaterial->SetupBlockBuffer(m_pOriginalBlockBuffer, m_OriginalBlockBufferSize);

    // 初期値に戻します。
    m_pOwnerModel->SetMatVisibility(m_Index, m_MaterialVisibilityEnable);
    m_pMaterial->GetResRenderState()->GetPolygonCtrl().SetCullFront(m_CullFront);
    m_pMaterial->GetResRenderState()->GetPolygonCtrl().SetCullBack(m_CullBack);

    CleanupSamplers();
}

void EditMaterialObj::UpdateRenderInfo(const void* pRenderInfoDic, size_t dataSize)
{
    m_RenderInfo.Update(pRenderInfoDic, dataSize);
}

EditMaterialObj::EditMaterialObj(
    IAllocator* allocator,
    int index,
    nw::g3d::ModelObj* ownerModelObj,
    nw::g3d::MaterialObj* materialObj)
    : m_pAllocator(allocator)
    , m_Index(index)
    , m_pOwnerModel(ownerModelObj)
    , m_pMaterial(materialObj)
    , m_WorkBuffer(allocator, MaterialObj::BUFFER_ALIGNMENT)
    , m_WorkBlockBuffer(allocator, MaterialObj::BLOCK_BUFFER_ALIGNMENT)
    , m_RenderInfo(allocator, materialObj)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(materialObj, "%s\n", NW_G3D_RES_GET_NAME(m_pOwnerModel->GetResource(), GetName()));
    nw::g3d::res::ResMaterial* resMaterial = materialObj->GetResource();
    NW_G3D_ASSERT_NOT_NULL_DETAIL(resMaterial, "%s\n", NW_G3D_RES_GET_NAME(m_pOwnerModel->GetResource(), GetName()));
    m_pOriginalMaterial = resMaterial;

    m_pOriginalBuffer = materialObj->GetBufferPtr();
    MaterialObj::InitArg arg(resMaterial);
    arg.BufferingCount(materialObj->GetBufferingCount());
    m_OriginalBufferSize = MaterialObj::CalcBufferSize(arg);
    m_OriginalBufferingCount = materialObj->GetBufferingCount();

    m_pOriginalBlockBuffer = materialObj->GetBlockBufferPtr();
    m_OriginalBlockBufferSize = materialObj->CalcBlockBufferSize();
}

bool EditMaterialObj::Init()
{
    int samplerCount = m_pMaterial->GetSamplerCount();
    if (samplerCount > 0)
    {
        size_t bufferSize = FixedSizeArray<EditSampler>::CalcBufferSize(samplerCount);
        void* buffer = m_pAllocator->Alloc(bufferSize, DEFAULT_ALIGNMENT);
        if (buffer == NULL)
        {
            return false;
        }

        m_SamplerArray.SetArrayBuffer(buffer, samplerCount);
        for(int i = 0, end = m_SamplerArray.Size(); i < end; ++i)
        {
            nw::g3d::res::ResSampler* resSampler = m_pMaterial->GetResSampler(i);
            EditSampler* samplerBuffer = m_SamplerArray.UnsafeAt(i);
            new(samplerBuffer) EditSampler(i, m_pMaterial, resSampler);
        }
    }
    return true;
}

void EditMaterialObj::Destroy()
{
    // オリジナルのマテリアルに戻す前に、RenderInfoDicも元に戻す
    m_RenderInfo.Destroy();

    DestroySamplers();
    m_WorkBlockBuffer.Clear();
    m_WorkBuffer.Clear();
}

void
EditMaterialObj::DestroySamplers()
{
    void* buffer = m_SamplerArray.GetBufferPtr();
    if (buffer == NULL)
    {
        return;
    }

    m_pAllocator->Free(buffer);
    m_SamplerArray.Clear();
}

bool EditMaterialObj::ReinitBuffer(nw::g3d::res::ResMaterial* resMaterial)
{
    nw::g3d::MaterialObj::InitArg arg(resMaterial);
    arg.BufferingCount(m_pMaterial->GetBufferingCount());

    u32 size = m_pMaterial->CalcBufferSize(arg);
    return m_WorkBuffer.Resize(size);
}

bool EditMaterialObj::ReinitBlockBuffer()
{
    u32 size = m_pMaterial->CalcBlockBufferSize();
    return m_WorkBlockBuffer.Resize(size);
}

void EditMaterialObj::SetupMaterialInstance(nw::g3d::res::ResMaterial* resMaterial)
{
    m_RenderInfo.UpdateMaterial(resMaterial);

    nw::g3d::MaterialObj::InitArg arg(resMaterial);
    arg.BufferingCount(m_pMaterial->GetBufferingCount());

    void* userPtr = m_pMaterial->GetUserPtr();// UserPtr を保持するために一時退避
    resMaterial->Cleanup();
    resMaterial->Setup();
    m_pMaterial->CleanupBlockBuffer();
    m_pMaterial->Init(arg, m_WorkBuffer.GetWorkBufferPtr(), m_WorkBuffer.Size());
    m_pMaterial->SetUserPtr(userPtr);
    bool result = ReinitBlockBuffer();
    NW_G3D_ASSERTMSG(result, "%s\n", NW_G3D_RES_GET_NAME(m_pOwnerModel->GetResource(), GetName())); // 今は止める
    m_pMaterial->SetupBlockBuffer(m_WorkBlockBuffer.GetWorkBufferPtr(), m_WorkBlockBuffer.Size());
}

void EditMaterialObj::CleanupSamplers()
{
    for (int i = 0, end = m_SamplerArray.Size(); i < end; ++i)
    {
        EditSampler* sampler = m_SamplerArray.UnsafeAt(i);
        sampler->Cleanup();
    }
}

void EditMaterialObj::Show()
{
    m_pOwnerModel->SetMatVisibility(m_Index, true);
}

void EditMaterialObj::Hide()
{
    m_pOwnerModel->SetMatVisibility(m_Index, false);
}

bool EditMaterialObj::IsVisibleByDefault() const
{
    return m_MaterialVisibilityEnable;
}

void EditSampler::Setup()
{
    // 初期値を保存
    m_WrapU = m_pSampler->GetGfxSampler()->GetClampX();
    m_WrapV = m_pSampler->GetGfxSampler()->GetClampY();
    m_WrapW = m_pSampler->GetGfxSampler()->GetClampZ();

    m_MagFilter = m_pSampler->GetGfxSampler()->GetMagFilter();
    m_MinFilter = m_pSampler->GetGfxSampler()->GetMinFilter();
    m_MipFilter = m_pSampler->GetGfxSampler()->GetMipFilter();
    m_MaxAniso = m_pSampler->GetGfxSampler()->GetMaxAniso();
}

void EditSampler::Cleanup()
{
    // 初期値に戻します。
    m_pSampler->GetGfxSampler()->SetClampX(m_WrapU);
    m_pSampler->GetGfxSampler()->SetClampY(m_WrapV);
    m_pSampler->GetGfxSampler()->SetClampZ(m_WrapW);

    m_pSampler->GetGfxSampler()->SetMagFilter(m_MagFilter);
    m_pSampler->GetGfxSampler()->SetMinFilter(m_MinFilter);
    m_pSampler->GetGfxSampler()->SetMipFilter(m_MipFilter);
    m_pSampler->GetGfxSampler()->SetMaxAniso(m_MaxAniso);
}

}}}} // namespace nw::g3d::edit::detail

#endif // NW_G3D_CONFIG_USE_HOSTIO
