﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <nn/os.h>
#include <nn/util/util_LockGuard.h>
#include <nn/ndd/detail/ndd_ReceiveSubmodule.h>
#include <nn/ndd/detail/ndd_Utility.h>

namespace nn { namespace ndd {

void RingBuffer::Initialize()
{
    NN_UTIL_LOCK_GUARD(m_Mutex);
    //[todo]fsからの復旧にする
    m_Counter = 0;
}

void RingBuffer::Add(const ReceiveDataDescription& dataDescription)
{
    NN_UTIL_LOCK_GUARD(m_Mutex);
    uint16_t index;
    index = GetIndex(m_Counter);
    NN_NDD_LOG("RingBuffer Add: counter = %d, index = %d, dataSize = %d\n", m_Counter, index, dataDescription.dataSize);
    m_Storage.Write(dataDescription);
    ++m_Counter;
}

bool RingBuffer::Get(ReceiveDataDescription* pBuffer, uint32_t* pNextCounter, uint32_t counter)
{
    NN_UTIL_LOCK_GUARD(m_Mutex);
    NN_ABORT_UNLESS_NOT_NULL(pBuffer);
    NN_ABORT_UNLESS_NOT_NULL(pNextCounter);
    if(IsValidCounter(counter))
    {
        auto normalizedCounter = NormalizeCounter(counter);
        m_Storage.Read(pBuffer, GetIndex(normalizedCounter));
        *pNextCounter = normalizedCounter + 1;
        return true;
    }
    else
    {
        *pNextCounter = GetNextCounter();//Nextを返す
        return false;
    }
}

uint32_t RingBuffer::GetCurrentCounter() const
{
    NN_UTIL_LOCK_GUARD(m_Mutex);
    if(m_Counter != 0)
    {
        return m_Counter - 1;
    }
    else
    {
        return 0;
    }
}

uint32_t RingBuffer::GetOldestCounter() const
{
    NN_UTIL_LOCK_GUARD(m_Mutex);
    //counterが指すのは次のAdd対象なので、BufferCountMaxの次を指すまでは最古は0
    if(m_Counter <= BufferCountMax)
    {
        return 0;
    }
    else
    {
        return m_Counter - BufferCountMax;
    }
}

uint32_t RingBuffer::GetNextCounter() const
{
    NN_UTIL_LOCK_GUARD(m_Mutex);
    return m_Counter;
}

uint32_t RingBuffer::GetRecentCounter(uint32_t counter, int count) const
{
    NN_UTIL_LOCK_GUARD(m_Mutex);
    //指定カウンタの決定
    uint32_t designatedCounter;
    if(IsValidCounter(counter))
    {
        //正常なカウンタ値のため、Normalizeする
        designatedCounter = NormalizeCounter(counter);
    }
    else
    {
        //異常なカウンタ値のため、Nextを使用
        designatedCounter = GetNextCounter();
    }

    //候補カウンタの決定
    auto candidateCounter = GetNextCounter() - count;
    if(IsValidCounter(candidateCounter))
    {
        //正常なカウンタ値のため、Normalizeする
        candidateCounter = NormalizeCounter(candidateCounter);
    }
    else
    {
        if(count == 0)
        {
            //回り込まずに異常なカウンタ値のため、Nextを候補にする
            candidateCounter = GetNextCounter();
        }
        else
        {
            //回り込んだ上での異常なカウンタ値のため、最古を候補にする
            candidateCounter = GetOldestCounter();
        }
    }
    return std::max<uint32_t>(candidateCounter, designatedCounter);
}

uint16_t RingBuffer::GetAvailableCount(uint64_t counter) const
{
    NN_UTIL_LOCK_GUARD(m_Mutex);
    if(IsValidCounter(counter))
    {
        return GetNextCounter() - NormalizeCounter(counter);
    }
    else
    {
        return 0;
    }
}

void RingBuffer::Clear()
{
    NN_UTIL_LOCK_GUARD(m_Mutex);
    m_Counter = 0;
    m_Storage.Clear();
}

void RingBuffer::Dump()
{
    NN_UTIL_LOCK_GUARD(m_Mutex);
    NN_DETAIL_NDD_INFO("Dump RingBuffer\n");
    NN_DETAIL_NDD_INFO("Oldest = %d, Current = %d, Next = %d, AvailableCount = %d\n",
            GetOldestCounter(), GetCurrentCounter(), GetNextCounter(), GetAvailableCount(GetOldestCounter()));
    NN_DETAIL_NDD_INFO("InternalCounter = %d\n", m_Counter);
    //カウンタによらず、バッファを全て出力
    for(int i=0;i<BufferCountMax;++i)
    {
        ReceiveDataDescription rdd;
        m_Storage.Read(&rdd, i);
        NN_DETAIL_NDD_INFO("Index = %d, DataSize = %d, receiveTimePoint = %lld\n", i, rdd.dataSize, rdd.receiveTimePoint.value);
        PrintByteDataAsHex(&rdd.data[0], rdd.dataSize);
    }
}

uint16_t RingBuffer::GetIndex(uint32_t counter) const
{
    return counter % BufferCountMax;
}

bool RingBuffer::IsValidCounter(uint32_t counter) const
{
    //最新含む未来のカウンタを指定した場合はエラー
    if(m_Counter <= counter)
    {
        return false;
    }
    else
    {
        return true;
    }
}

uint32_t RingBuffer::NormalizeCounter(uint32_t counter) const
{
    auto oldestCounter = GetOldestCounter();
    if(counter < oldestCounter)
    {
        //最古よりも古いカウンタの場合は、最古のカウンタを返す
        return oldestCounter;
    }
    else
    {
        //正常範囲内のため、何もしない
        return counter;
    }
}

RejectDataIdContainer::RejectDataIdContainer()
{
    Clear();
}

void RejectDataIdContainer::Add(const DataId& id)
{
    if(m_Count < IdCountMax)
    {
        //増加
        ++m_Count;
    }
    else
    {
        //上書き
    }

    if(m_Index < IdCountMax)
    {
        //終端ではない
        m_Id[m_Index] = id;
        ++m_Index;
    }
    else
    {
        //終端なので回りこむ
        m_Index = 0;
        m_Id[m_Index] = id;
        ++m_Index;
    }
}

void RejectDataIdContainer::Clear()
{
    m_Count = 0;
    m_Index = 0;
}

bool RejectDataIdContainer::IsReject(const DataId& id) const
{
    for(int i=0;i<m_Count;++i)
    {
        if(IsSameId(m_Id[i], id))
        {
            return true;
        }
    }
    return false;
}

bool RejectDataIdContainer::IsSameId(const DataId& idA, const DataId& idB) const
{
    for(int i=0;i<DataIdSize;++i)
    {
        if(idA.raw[i] != idB.raw[i])
        {
            return false;
        }
    }
    return true;
}

ScanBuffer::ScanBuffer()
{
    Clear();
}

void ScanBuffer::Add(const ReceiveDataDescription& dataDescription)
{
    NN_UTIL_LOCK_GUARD(m_Mutex);
    if(m_Count >= BufferCountMax)
    {
        return;
    }

    if(IsSameData(dataDescription))
    {
        return;
    }

    m_Rdd[m_Count] = dataDescription;
    ++m_Count;
}

void ScanBuffer::Clear()
{
    NN_UTIL_LOCK_GUARD(m_Mutex);
    m_Count = 0;
}

int ScanBuffer::GetCount() const
{
    NN_UTIL_LOCK_GUARD(m_Mutex);
    return m_Count;
}

ReceiveDataDescription ScanBuffer::Get(int index) const
{
    NN_UTIL_LOCK_GUARD(m_Mutex);
    NN_ABORT_UNLESS(index >= 0);
    NN_ABORT_UNLESS(index < m_Count);
    return m_Rdd[index];
}

bool ScanBuffer::IsSameData(const ReceiveDataDescription& dataDescription) const
{
    //[todo]DataIdでの比較を検討
    for(int i=0;i<m_Count;++i)
    {
        if(m_Rdd[i].dataSize == dataDescription.dataSize)
        {
            int cmpResult;
            cmpResult = memcmp(&m_Rdd[i].data[0], &dataDescription.data[0], dataDescription.dataSize);
            if(cmpResult == 0)
            {
                return true;
            }
        }
    }
    return false;
}

}}
