﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/os/os_SdkConditionVariable.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Abort.h>
#include <nn/util/util_LockGuard.h>

#include <nn/ndd/detail/ndd_InternalTypes.h>
#include <nn/ndd/detail/ndd_Queue.h>
#include <nn/ndd/detail/ndd_Utility.h>
#include <nn/ndd/ndd_PrivateResult.h>

namespace nn { namespace ndd { namespace queue {

namespace {
//キュー
nn::os::MessageQueueType g_Queue;
nn::os::SdkMutex g_Mutex;
nn::os::SdkConditionVariable g_ConditionVariableNoEmpty;
nn::os::SdkConditionVariable g_ConditionVariableNoFull;
const int MessageCountMax = 20;
uintptr_t g_QueueBuffer[MessageCountMax];
nn::os::MultiWaitHolderType g_MultiWaitHolder;

//キューが扱うメッセージ
struct MessageList{
    Message message[MessageCountMax];
};
MessageList g_MessageList;
}

void Initialize()
{
    nn::os::InitializeMessageQueue( &g_Queue, g_QueueBuffer, MessageCountMax );
    for(int i=0;i<MessageCountMax;++i)
    {
        g_MessageList.message[i].m_IsUsing = false;
    }
    nn::os::InitializeMultiWaitHolder(&g_MultiWaitHolder, &g_Queue, nn::os::MessageQueueWaitType_WaitForNotEmpty);
    nn::os::SetMultiWaitHolderUserData(&g_MultiWaitHolder, MultiWaitId_Queue);
}

nn::os::MultiWaitHolderType* GetMultiWaitHolderPtr()
{
    return &g_MultiWaitHolder;
}

bool SearchEmptyIndex(uint8_t* pIndex)
{
    for(int i=0;i<MessageCountMax;++i)
    {
        if(g_MessageList.message[i].m_IsUsing == false)
        {
            *pIndex = i;
            return true;
        }
    }

    //キューが一杯
    //キュー順を無視してメッセージを全て出力し、エラーを返す
    NN_DETAIL_NDD_ERROR("Queue full\n");
    for(int i=0;i<MessageCountMax;++i)
    {
        NN_NDD_LOG("%d IsUsing = %d, ApiId = %d\n", i , g_MessageList.message[i].m_IsUsing, g_MessageList.message[i].GetId());
    }
    return false;
}

//入力のisUsingは無視
//キューが一杯の場合、ブロックする
//ブロックの影響範囲は、サーバスレッド数、1クライアントあたりに許容するセッション数に応じて、変わる
//現状は、サーバースレッドが1つのため、全クライアントのIPCがブロックする
void EnQueue(const Message& message)
{
    NN_UTIL_LOCK_GUARD(g_Mutex);

    NN_NDD_LOG("Enqueue: id = %d\n", message.GetId());
    uint8_t index;
    while(!SearchEmptyIndex(&index))
    {
        g_ConditionVariableNoFull.Wait(g_Mutex);
    }

    g_MessageList.message[index] = message;
    g_MessageList.message[index].m_IsUsing = true;
    nn::os::SendMessageQueue( &g_Queue, reinterpret_cast<uintptr_t>(&g_MessageList.message[index]));
    g_ConditionVariableNoEmpty.Signal();
}

//出力のisUsingは要無視
void DeQueue(Message* pMessage)
{
    NN_UTIL_LOCK_GUARD(g_Mutex);

    uintptr_t pointerToData;
    while(!nn::os::TryReceiveMessageQueue( &pointerToData, &g_Queue ))
    {
        g_ConditionVariableNoEmpty.Wait(g_Mutex);
    }

    Message* pReceivedMessage;
    pReceivedMessage = reinterpret_cast<Message*>(pointerToData);
    *pMessage = *pReceivedMessage;
    pReceivedMessage->m_IsUsing = false;//フラグ落とした時点で送信側から再利用されうるので、バッファコピー後に落とす
    g_ConditionVariableNoFull.Signal();
}

}}}
