﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/ndd/detail/ndd_HandlerSubmodule.h>
#include <nn/ndd/ndd_PrivateResult.h>
#include <nn/ndd/detail/ndd_Utility.h>

#include <nn/nifm/nifm_ApiForSystem.h>
#include <nn/nifm/nifm_ApiRequestPrivate.h>
#include <nn/nifm/nifm_ResultPrivate.h>
#include <nn/nifm/nifm_TypesRequirement.h>


namespace nn { namespace ndd {

ContextContainer::ContextContainer()
{
    m_ContextCount = 0;
}

ContextContainer::~ContextContainer()
{

}

void ContextContainer::Add(Context* pContext, Handle handle)
{
    NN_ABORT_UNLESS_NOT_NULL(pContext);
    NN_ABORT_UNLESS(m_ContextCount <= ContextCountMax);

    pContext->SetHandle(handle);
    m_pContext[m_ContextCount] = pContext;
    ++m_ContextCount;
}

uint8_t ContextContainer::Search(Handle handle)
{
    for(int i=0;i<m_ContextCount;++i)
    {
        if(m_pContext[i]->GetHandle() == handle)
        {
            return i;
        }
    }
    NN_DETAIL_NDD_ERROR("target handle had been removed.\n");
    return 0xFF;
}

void ContextContainer::Remove(Handle handle)
{
    uint8_t index = Search(handle);
    NN_ABORT_UNLESS(index != 0xFF);
    m_pContext[index]->SignalReply();//Searchで参照する都合上、同期用のシグナルはここで行う
    //NN_NDD_LOG("index = %d\n", index);
    for(int i=index;i<ContextCountMax - 1;++i)
    {
        m_pContext[i] = m_pContext[i + 1];
    }
    --m_ContextCount;
}

void ContextContainer::Signal(Handle handle, Context::EventId eventId)
{
    uint8_t index = Search(handle);
    if(index != 0xFF)
    {
        m_pContext[index]->Signal(eventId);
    }
}

void ContextContainer::Signal(Context::EventId eventId)
{
    for(int i=0;i<m_ContextCount;++i)
    {
        m_pContext[i]->Signal(eventId);
    }
}

void ContextContainer::Reply(Handle handle)
{
    uint8_t index = Search(handle);
    NN_ABORT_UNLESS(index != 0xFF);
    m_pContext[index]->SignalReply();
}

void ContextContainer::Reply(Handle handle, const void* pReply)
{
    NN_ABORT_UNLESS_NOT_NULL(pReply);
    uint8_t index = Search(handle);
    NN_ABORT_UNLESS(index != 0xFF);
    //[todo]データコピー
    m_pContext[index]->SignalReply();
}

HandleProvider::HandleProvider()
{
    m_HandleCount = 0;
}

Handle HandleProvider::Acquire()
{
    if(m_HandleCount == HandleCountMax)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ndd::ResultFaultyDesign());
    }

    m_Handle[m_HandleCount] = m_Resource;
    ++m_HandleCount;
    Handle result = m_Resource;
    m_Resource.Change();

    return result;
}

uint8_t HandleProvider::Search(Handle handle)
{
    for(int i=0;i<m_HandleCount;++i)
    {
        if(m_Handle[i] == handle)
        {
            return i;
        }
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ndd::ResultFaultyDesign());
    return 0xFF;
}

void HandleProvider::Delete(Handle handle)
{
    uint8_t index;
    index = Search(handle);
    for(int i=index;i<HandleCountMax - 1;++i)
    {
        m_Handle[i] = m_Handle[i + 1];
    }
    --m_HandleCount;
}

SendDataDescriptionContainer::SendDataDescriptionContainer()
{
    Clear();
}

void SendDataDescriptionContainer::Add(const SendDataDescription& sendDataDescription)
{
    NN_ABORT_UNLESS(sendDataDescription.dataSize <= SendDataSizeMax);
    m_SendDataDescription = sendDataDescription;
    m_IsExist = true;
}

void SendDataDescriptionContainer::Clear()
{
    m_SendDataDescription.dataSize = 0;
    m_IsExist = false;
}

bool SendDataDescriptionContainer::IsExist() const
{
    return m_IsExist;
}

const SendDataDescription* SendDataDescriptionContainer::GetPtr() const
{
    return &m_SendDataDescription;
}

StateConfig::StateConfig()
{
    //[todo]設定値の読み出し
    Awake();
    DisableNetwork();
    EnableAutoCommunication();
}

void StateConfig::Awake()
{
    m_IsAwake = true;
}

void StateConfig::Sleep()
{
    m_IsAwake = false;
}

bool StateConfig::IsAwake() const
{
    return m_IsAwake;
}

void StateConfig::EnableNetwork()
{
    m_IsNetworkEnabled = true;
}

void StateConfig::DisableNetwork()
{
    m_IsNetworkEnabled = false;
}

bool StateConfig::IsNetworkEnabled() const
{
    return m_IsNetworkEnabled;
}

void StateConfig::EnableAutoCommunication()
{
    m_IsAutoCommunication = true;
}

void StateConfig::DisableAutoCommunication()
{
    m_IsAutoCommunication = false;
}

bool StateConfig::IsAutoCommunicationEnabled() const
{
    return m_IsAutoCommunication;
}

void StateConfig::AddSendData(const SendDataDescription& sendDataDescription)
{
    m_SendDataDescriptionContainer.Add(sendDataDescription);
}

void StateConfig::ClearSendData()
{
    m_SendDataDescriptionContainer.Clear();
}

bool StateConfig::IsSendDataExist() const
{
    return m_SendDataDescriptionContainer.IsExist();
}

const SendDataDescription* StateConfig::GetSendDataPtr() const
{
    return m_SendDataDescriptionContainer.GetPtr();
}

State StateConfig::Get() const
{
    if(IsAwake())
    {
        if(IsNetworkEnabled())
        {
            if(IsAutoCommunicationEnabled())
            {
                if(IsSendDataExist())
                {
                    return State_ReceiveSend;
                }
                else
                {
                    return State_Receive;
                }
            }
            else
            {
                return State_Idle;
            }
        }
        else
        {
            return State_Stop;
        }
    }
    else
    {
        if(IsNetworkEnabled())
        {
            if(IsAutoCommunicationEnabled())
            {
                if(IsSendDataExist())
                {
                    return State_SleepReceiveSend;
                }
                else
                {
                    return State_SleepReceive;
                }
            }
            else
            {
                return State_SleepIdle;
            }
        }
        else
        {
            return State_SleepStop;
        }
    }
}

State StateConfig::GetNext(State currentState, State targetState)
{
    switch(currentState)
    {
        case State_Stop:
        {
            return GetNextFromStop(targetState);
            break;
        }
        case State_Idle:
        {
            return GetNextFromIdle(targetState);
            break;
        }
        case State_Receive:
        {
            return GetNextFromReceive(targetState);
            break;
        }
        case State_ReceiveSend:
        {
            return GetNextFromReceiveSend(targetState);
            break;
        }
        case State_SleepStop:
        {
            return GetNextFromSleepStop(targetState);
            break;
        }
        case State_SleepIdle:
        {
            return GetNextFromSleepIdle(targetState);
            break;
        }
        case State_SleepReceive:
        {
            return GetNextFromSleepReceive(targetState);
            break;
        }
        case State_SleepReceiveSend:
        {
            return GetNextFromSleepReceiveSend(targetState);
            break;
        }
        default:
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ndd::ResultFaultyDesign());
            NN_ABORT("FaultyDesign\n");
        }
    }
}

State StateConfig::GetNextFromStop(State targetState)
{
    switch(targetState)
    {
        case State_Stop:
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ndd::ResultFaultyDesign());
            NN_ABORT("FaultyDesign\n");
            break;
        case State_SleepStop:
            return State_SleepStop;
            break;
        case State_Idle:
        case State_Receive:
        case State_ReceiveSend:
        case State_SleepIdle:
        case State_SleepReceive:
        case State_SleepReceiveSend:
            return State_Idle;
            break;
        default:
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ndd::ResultFaultyDesign());
            NN_ABORT("FaultyDesign\n");
    }
}

State StateConfig::GetNextFromIdle(State targetState)
{
    switch(targetState)
    {
        case State_Idle:
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ndd::ResultFaultyDesign());
            NN_ABORT("FaultyDesign\n");
            break;
        case State_Stop:
        case State_SleepStop:
            return State_Stop;
            break;
        case State_SleepIdle:
            return State_SleepIdle;
            break;
        case State_Receive:
        case State_ReceiveSend:
        case State_SleepReceive:
        case State_SleepReceiveSend:
            return State_Receive;
            break;
        default:
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ndd::ResultFaultyDesign());
            NN_ABORT("FaultyDesign\n");
    }
}

State StateConfig::GetNextFromReceive(State targetState)
{
    switch(targetState)
    {
        case State_Receive:
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ndd::ResultFaultyDesign());
            NN_ABORT("FaultyDesign\n");
            break;
        case State_Stop:
        case State_Idle:
        case State_SleepStop:
        case State_SleepIdle:
            return State_Idle;
            break;
        case State_SleepReceive:
            return State_SleepReceive;
            break;
        case State_ReceiveSend:
        case State_SleepReceiveSend:
            return State_ReceiveSend;
            break;
        default:
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ndd::ResultFaultyDesign());
            NN_ABORT("FaultyDesign\n");
    }
}

State StateConfig::GetNextFromReceiveSend(State targetState)
{
    switch(targetState)
    {
        case State_ReceiveSend:
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ndd::ResultFaultyDesign());
            NN_ABORT("FaultyDesign\n");
            break;
        case State_Stop:
        case State_Idle:
        case State_Receive:
        case State_SleepStop:
        case State_SleepIdle:
        case State_SleepReceive:
            return State_Receive;
            break;
        case State_SleepReceiveSend:
            return State_SleepReceiveSend;
            break;
        default:
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ndd::ResultFaultyDesign());
            NN_ABORT("FaultyDesign\n");
    }
}

State StateConfig::GetNextFromSleepStop(State targetState)
{
    switch(targetState)
    {
        case State_SleepStop:
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ndd::ResultFaultyDesign());
            NN_ABORT("FaultyDesign\n");
            break;
        case State_Stop:
        case State_Idle:
        case State_Receive:
        case State_ReceiveSend:
            return State_Stop;
            break;
        case State_SleepIdle:
        case State_SleepReceive:
        case State_SleepReceiveSend:
            //ステート視点では越権だが、仕様上発生し得ない遷移のため、fatalにしておく
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ndd::ResultFaultyDesign());
            NN_ABORT("FaultyDesign\n");
            break;
        default:
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ndd::ResultFaultyDesign());
            NN_ABORT("FaultyDesign\n");
    }
}

State StateConfig::GetNextFromSleepIdle(State targetState)
{
    switch(targetState)
    {
        case State_SleepIdle:
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ndd::ResultFaultyDesign());
            NN_ABORT("FaultyDesign\n");
            break;
        case State_Stop:
        case State_Idle:
        case State_Receive:
        case State_ReceiveSend:
            return State_Idle;
            break;
        case State_SleepStop:
        case State_SleepReceive:
        case State_SleepReceiveSend:
            //ステート視点では越権だが、仕様上発生し得ない遷移のため、fatalにしておく
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ndd::ResultFaultyDesign());
            NN_ABORT("FaultyDesign\n");
            break;
        default:
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ndd::ResultFaultyDesign());
            NN_ABORT("FaultyDesign\n");
    }
}

State StateConfig::GetNextFromSleepReceive(State targetState)
{
    switch(targetState)
    {
        case State_SleepReceive:
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ndd::ResultFaultyDesign());
            NN_ABORT("FaultyDesign\n");
            break;
        case State_Stop:
        case State_Idle:
        case State_Receive:
        case State_ReceiveSend:
            return State_Receive;
            break;
        case State_SleepStop:
        case State_SleepIdle:
        case State_SleepReceiveSend:
            //ステート視点では越権だが、仕様上発生し得ない遷移のため、fatalにしておく
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ndd::ResultFaultyDesign());
            NN_ABORT("FaultyDesign\n");
            break;
        default:
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ndd::ResultFaultyDesign());
            NN_ABORT("FaultyDesign\n");
    }
}

State StateConfig::GetNextFromSleepReceiveSend(State targetState)
{
    switch(targetState)
    {
        case State_SleepReceiveSend:
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ndd::ResultFaultyDesign());
            NN_ABORT("FaultyDesign\n");
            break;
        case State_Stop:
        case State_Idle:
        case State_Receive:
        case State_ReceiveSend:
            return State_ReceiveSend;
            break;
        case State_SleepStop:
        case State_SleepIdle:
        case State_SleepReceive:
            //ステート視点では越権だが、仕様上発生し得ない遷移のため、fatalにしておく
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ndd::ResultFaultyDesign());
            NN_ABORT("FaultyDesign\n");
            break;
        default:
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ndd::ResultFaultyDesign());
            NN_ABORT("FaultyDesign\n");
    }
}

ScanConfig::ScanConfig()
{
    Disable();
}

bool ScanConfig::Enable(Handle handle)
{
    auto index = Search(handle);

    if(index == 0xFF)
    {
        //Requesterとして未登録のため、登録する
        NN_ABORT_UNLESS(m_RequesterCount < RequesterCountMax - 1);
        if(m_RequesterCount == 0)
        {
            //Scanの新規開始
            m_Requester[m_RequesterCount] = handle;
            ++m_RequesterCount;
            return true;
        }
        else
        {
            //実行中のScanに参加
            m_Requester[m_RequesterCount] = handle;
            ++m_RequesterCount;
            return false;
        }
    }
    else
    {
        //既にRequesterとして登録されているため、何もしない
        return false;
    }
}

void ScanConfig::Disable()
{
    m_RequesterCount = 0;
}

bool ScanConfig::IsEnabled(Handle handle) const
{
    auto index = Search(handle);
    if(index == 0xFF)
    {
        return false;
    }
    else
    {
        return true;
    }
}

uint8_t ScanConfig::GetRequesterCount() const
{
    return m_RequesterCount;
}

Handle ScanConfig::GetRequester(uint8_t index) const
{
    NN_ABORT_UNLESS(index < RequesterCountMax - 1);
    return m_Requester[index];
}

uint8_t ScanConfig::Search(Handle handle) const
{
    for(int i=0;i<m_RequesterCount;++i)
    {
        if(m_Requester[i] == handle)
        {
            return i;
        }
    }
    return 0xFF;
}

NetworkRequest::NetworkRequest()
{
    m_pNetworkRequest = nullptr;
}

void NetworkRequest::Initialize()
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::InitializeSystem());
    nn::nifm::RequestParameters requestParams;
    requestParams.requirementPreset = nn::nifm::RequirementPreset_NeighborDetectionForSystemProcess;
    m_pNetworkRequest = new nn::nifm::Request(requestParams);
    Submit();

    nn::os::InitializeMultiWaitHolder(&m_MultiWaitHolder, m_pNetworkRequest->GetSystemEvent().GetBase());
    nn::os::SetMultiWaitHolderUserData(&m_MultiWaitHolder, MultiWaitId_NetworkRequest);
}

NetworkRequest::~NetworkRequest()
{
    if(m_pNetworkRequest != nullptr)
    {
        delete m_pNetworkRequest;
    }
}

nn::os::MultiWaitHolderType* NetworkRequest::GetMultiWaitHolderPtr()
{
    return &m_MultiWaitHolder;
}

void NetworkRequest::ClearEvent()
{
    nn::os::ClearSystemEvent(m_pNetworkRequest->GetSystemEvent().GetBase());
}

nn::nifm::RequestState NetworkRequest::GetState() const
{
    return m_pNetworkRequest->GetRequestState();
}

void NetworkRequest::Submit()
{
    m_pNetworkRequest->Submit();
}

void NetworkRequest::Cancel()
{
    m_pNetworkRequest->Cancel();
}

void PowerRequest::Initialize()
{
    const nn::psc::PmModuleId dependencies[] = {nn::psc::PmModuleId_Pcie, nn::psc::PmModuleId_Fs};
    auto result = m_PmModule.Initialize(
            nn::psc::PmModuleId_Ndd,
            dependencies,
            sizeof(dependencies) / sizeof(dependencies[0]),
            nn::os::EventClearMode_ManualClear);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    nn::os::InitializeMultiWaitHolder(&m_MultiWaitHolder, m_PmModule.GetEventPointer()->GetBase());
    //static nn::os::EventType tempEvent;
    //nn::os::InitializeEvent(&tempEvent, false, nn::os::EventClearMode_AutoClear);
    //nn::os::InitializeMultiWaitHolder(&m_MultiWaitHolder, &tempEvent);
    nn::os::SetMultiWaitHolderUserData(&m_MultiWaitHolder, MultiWaitId_PowerRequest);
}

nn::os::MultiWaitHolderType* PowerRequest::GetMultiWaitHolderPtr()
{
    return &m_MultiWaitHolder;
}

void PowerRequest::ClearEvent()
{
    nn::os::ClearSystemEvent(m_PmModule.GetEventPointer()->GetBase());
}

nn::psc::PmState PowerRequest::GetState()
{
    nn::psc::PmState    pmState;
    nn::psc::PmFlagSet  pmFlags;
    auto result = m_PmModule.GetRequest(&pmState, &pmFlags);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    return pmState;
}

void PowerRequest::Acknowledge(nn::psc::PmState pmState)
{
    m_PmModule.Acknowledge(pmState, nn::ResultSuccess());
}

MultiWait::MultiWait()
{
    nn::os::InitializeMultiWait(&m_MultiWait);
}

void MultiWait::Link(nn::os::MultiWaitHolderType* pMultiWaitHolder)
{
    nn::os::LinkMultiWaitHolder(&m_MultiWait, pMultiWaitHolder);
}

void MultiWait::Unlink(nn::os::MultiWaitHolderType* pMultiWaitHolder)
{
    nn::os::UnlinkMultiWaitHolder(pMultiWaitHolder);
}

MultiWaitId MultiWait::Wait()
{
    nn::os::MultiWaitHolderType* pHolder;
    MultiWaitId multiWaitId;

    pHolder = nn::os::WaitAny(&m_MultiWait);
    multiWaitId = static_cast<MultiWaitId>(nn::os::GetMultiWaitHolderUserData(pHolder));

    return multiWaitId;
}

}}
