﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <type_traits>
#include <nn/nn_Common.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/account.h>
#include <nn/account/account_ApiForSystemServices.h>
#include <nn/nd/detail/nd_FirmwareDebugSettings.h>
#include <nn/nd/detail/nd_InitializationManager.h>
#include <nn/nd/detail/nd_Log.h>
#include <nn/nd/service/nd_EventHandler.h>
#include <nn/nd/service/nd_ReceiveDataManager.h>
#include <nn/nd/service/nd_ScanManager.h>
#include <nn/nd/service/nd_SendDataManager.h>
#include <nn/ndd.h>
#include <nn/os.h>
#include <nn/os/os_Event.h>
#include <nn/pdm/pdm_QueryApiForSystem.h>
#include <nn/util/util_ScopeExit.h>

namespace nn { namespace nd { namespace service {

namespace
{

NN_OS_ALIGNAS_THREAD_STACK char g_EventHandlerThreadStack[16 * 1024];
nn::os::ThreadType g_EventHandlerThread;
detail::InitializationManager g_InitializationManager = NN_ND_DETAIL_INITIALIZATION_MANAGER_INITIALIZER;
os::Event g_StopEvent(os::EventClearMode_ManualClear);

struct EventHandlerResource
{
    UserIdManager* pUserIdManager;
    ReceiveDataManager* pReceiveDataManager;
    SendDataManager* pSendDataManager;
    ScanManager* pScanManager;
} g_EventHandlerResource = { nullptr, nullptr, nullptr, nullptr };

void ThreadFunc(void*) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_EventHandlerResource.pUserIdManager);
    NN_SDK_ASSERT(g_EventHandlerResource.pReceiveDataManager);
    NN_SDK_ASSERT(g_EventHandlerResource.pSendDataManager);
    NN_SDK_ASSERT(g_EventHandlerResource.pScanManager);

    os::SystemEvent dataReceiveEvent;
    ndd::AcquireReceiveDataEvent(dataReceiveEvent.GetBase());

    os::SystemEvent scanEvent;
    ndd::AcquireDeviceScanEvent(scanEvent.GetBase());

    account::Notifier userRegistrationNotifier;
    NN_ABORT_UNLESS_RESULT_SUCCESS(account::GetUserRegistrationNotifier(&userRegistrationNotifier));
    os::SystemEvent userRegistrationEvent;
    NN_ABORT_UNLESS_RESULT_SUCCESS(userRegistrationNotifier.GetSystemEvent(&userRegistrationEvent));

    account::Notifier profileUpdateNotifier;
    NN_ABORT_UNLESS_RESULT_SUCCESS(account::GetProfileUpdateNotifier(&profileUpdateNotifier));
    os::SystemEvent profileUpdateEvent;
    NN_ABORT_UNLESS_RESULT_SUCCESS(profileUpdateNotifier.GetSystemEvent(&profileUpdateEvent));

    account::Notifier networkServiceAccountAvailabilityChangeNotifier;
    NN_ABORT_UNLESS_RESULT_SUCCESS(account::GetNetworkServiceAccountAvailabilityChangeNotifier(&networkServiceAccountAvailabilityChangeNotifier));
    os::SystemEvent networkServiceAccountAvailabilityChangeEvent;
    NN_ABORT_UNLESS_RESULT_SUCCESS(networkServiceAccountAvailabilityChangeNotifier.GetSystemEvent(&networkServiceAccountAvailabilityChangeEvent));

    auto pRecentlyPlayedApplicationUpdateEvent = pdm::GetRecentlyPlayedApplicationUpdateEvent();

    auto& networkUserIdUpdateEvent = g_EventHandlerResource.pUserIdManager->GetNetworkUserIdUpdateEventRef();

    while( !g_StopEvent.TryWait() )
    {
        auto e = os::WaitAny(
            dataReceiveEvent.GetBase(),
            userRegistrationEvent.GetBase(),
            profileUpdateEvent.GetBase(),
            networkServiceAccountAvailabilityChangeEvent.GetBase(),
            pRecentlyPlayedApplicationUpdateEvent->GetBase(),
            scanEvent.GetBase(),
            g_EventHandlerResource.pSendDataManager->GetSendDataIdUpdateScheduleEvent().GetBase(),
            networkUserIdUpdateEvent.GetBase(),
            g_StopEvent.GetBase());
        switch( e )
        {
        case 0: // ndd でのデータ受信
            {
                NN_DETAIL_ND_TRACE("[nd] EventHandler : ndd received data.\n");
                dataReceiveEvent.Clear();
                g_EventHandlerResource.pReceiveDataManager->SignalNeighborInfoUpdateEventForSystem();
            }
            break;
        case 1: // ユーザー数の変化の通知
            {
                NN_DETAIL_ND_TRACE("[nd] EventHandler : User registration is notified.\n");
                userRegistrationEvent.Clear();
                int accountCount;
                account::Uid uids[account::UserCountMax];
                NN_ABORT_UNLESS_RESULT_SUCCESS(account::ListAllUsers(&accountCount, uids, NN_ARRAY_SIZE(uids)));
                g_EventHandlerResource.pUserIdManager->Update({ uids, accountCount });
                g_EventHandlerResource.pReceiveDataManager->Update({ uids, accountCount });
                g_EventHandlerResource.pSendDataManager->Update({ uids, accountCount });
            }
            break;
        case 2: // プロフィールの更新の通知
            {
                NN_DETAIL_ND_TRACE("[nd] EventHandler : Profile update is notified.\n");
                profileUpdateEvent.Clear();
                g_EventHandlerResource.pSendDataManager->RefreshAccountProfile();
            }
            break;
        case 3: // ネットワークサービスアカウントの可用性の変化の通知
            {
                NN_DETAIL_ND_TRACE("[nd] EventHandler : NetworkServiceAccount availability change is notified.\n");
                networkServiceAccountAvailabilityChangeEvent.Clear();
                g_EventHandlerResource.pUserIdManager->RefreshNetworkServiceAccountId();
            }
            break;
        case 4: // 最近遊んだアプリケーションの更新
            {
                NN_DETAIL_ND_TRACE("[nd] EventHandler : Recently played application is updated.\n");
                pRecentlyPlayedApplicationUpdateEvent->Clear();
                g_EventHandlerResource.pSendDataManager->RefreshRecentlyPlayedApplication();
            }
            break;
        case 5: // デバイススキャン完了
            {
                NN_DETAIL_ND_TRACE("[nd] EventHandler : Device scan done.\n");
                scanEvent.Clear();
                g_EventHandlerResource.pScanManager->HandleScanDoneEvent();
            }
            break;
        case 6: // 送信データID更新スケジュールイベントの通知
            {
                NN_DETAIL_ND_TRACE("[nd] EventHandler : SendDataId update schedule event is triggered.\n");
                g_EventHandlerResource.pSendDataManager->GetSendDataIdUpdateScheduleEvent().Clear();
                g_EventHandlerResource.pSendDataManager->RefreshSendDataId();
            }
            break;
        case 7: // NetworkUserId の更新
            {
                NN_DETAIL_ND_TRACE("[nd] EventHandler : NetworkUserId update event is triggered.\n");
                networkUserIdUpdateEvent.Clear();
                g_EventHandlerResource.pSendDataManager->CheckNetworkUserIdUpdate();
            }
            break;
        case 8: // 終了
            {
                // nop. while の判定で抜ける。
            }
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
} // NOLINT(impl/function_size)

} // ~nn::nd::service::<anonymous>

void EventHandler::Start(UserIdManager& userIdManager, ReceiveDataManager& receiveDataManager, SendDataManager& sendDataManager, ScanManager& scanManager) NN_NOEXCEPT
{
    g_InitializationManager.Initialize([&]() {
        g_EventHandlerResource = { &userIdManager, &receiveDataManager, &sendDataManager, &scanManager };
        os::CreateThread(&g_EventHandlerThread, ThreadFunc, nullptr,
            g_EventHandlerThreadStack, sizeof(g_EventHandlerThreadStack), NN_SYSTEM_THREAD_PRIORITY(nd, EventHandler));
        os::SetThreadName(&g_EventHandlerThread, NN_SYSTEM_THREAD_NAME(nd, EventHandler));
        os::StartThread(&g_EventHandlerThread);
    });
}

void EventHandler::Stop() NN_NOEXCEPT
{
    g_InitializationManager.Finalize([]() {
        g_StopEvent.Signal();
        os::DestroyThread(&g_EventHandlerThread);
    });
}

}}} // ~nn::nd::service
