﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/fs.h>
#include <nn/fs/fs_File.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_Mount.h>
#include <nn/nd/detail/nd_Log.h>
#include <nn/nd/detail/nd_SystemSaveData.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_IntUtil.h>
#include <nn/util/util_LockGuard.h>
#include <nn/util/util_ScopeExit.h>

namespace nn { namespace nd { namespace detail {

SystemSaveData::SystemSaveData() NN_NOEXCEPT
{
    Mount();
}

SystemSaveData::~SystemSaveData() NN_NOEXCEPT
{
    Unmount();
}

Result SystemSaveData::Create(const char* path, size_t fileSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(path);
    NN_SDK_REQUIRES(util::IsIntValueRepresentable<int64_t>(fileSize));

    char fullPath[FullPathLengthMax];
    GetFullPath(fullPath, path);

    return fs::CreateFile(fullPath, static_cast<int64_t>(fileSize));
}

Result SystemSaveData::Write(const char* path, const void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(path);
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES(util::IsIntValueRepresentable<int64_t>(bufferSize));

    char fullPath[FullPathLengthMax];
    GetFullPath(fullPath, path);

    NN_UTIL_LOCK_GUARD(m_Mutex);
    fs::FileHandle file;
    NN_RESULT_DO(fs::OpenFile(&file, fullPath, fs::OpenMode_Write | fs::OpenMode_AllowAppend));
    NN_UTIL_SCOPE_EXIT{ fs::CloseFile(file); };
    NN_RESULT_DO(fs::SetFileSize(file, static_cast<int64_t>(bufferSize)));
    NN_RESULT_DO(fs::WriteFile(file, 0, buffer, static_cast<int64_t>(bufferSize), fs::WriteOption::MakeValue(fs::WriteOptionFlag_Flush)));
    NN_RESULT_SUCCESS;
}

Result SystemSaveData::Read(size_t* pOutActualSize, void* outBuffer, size_t outBufferSize, const char* path) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutActualSize);
    NN_SDK_REQUIRES_NOT_NULL(outBuffer);
    NN_SDK_REQUIRES(util::IsIntValueRepresentable<int64_t>(outBufferSize));
    NN_SDK_REQUIRES_NOT_NULL(path);

    char fullPath[FullPathLengthMax];
    GetFullPath(fullPath, path);

    NN_UTIL_LOCK_GUARD(m_Mutex);
    fs::FileHandle file;
    NN_RESULT_DO(fs::OpenFile(&file, fullPath, fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT{ fs::CloseFile(file); };
    size_t readSize;
    NN_RESULT_DO(fs::ReadFile(&readSize, file, 0, outBuffer, static_cast<int64_t>(outBufferSize), fs::ReadOption::MakeValue(0)));
    *pOutActualSize = readSize;
    NN_RESULT_SUCCESS;
}

Result SystemSaveData::Commit() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);
    return fs::CommitSaveData(MountName);
}

void SystemSaveData::Mount() NN_NOEXCEPT
{
    auto r = fs::CreateSystemSaveData(SystemSaveDataId, SystemSaveDataSize, SystemSaveDataJournalSize, SaveDataFlags);
    if( !(r.IsSuccess() || fs::ResultPathAlreadyExists::Includes(r)) )
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(r);
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(fs::MountSystemSaveData(MountName, SystemSaveDataId));
}

void SystemSaveData::Unmount() NN_NOEXCEPT
{
    fs::Unmount(MountName);
}

void SystemSaveData::GetFullPath(char(&outFullPath)[FullPathLengthMax], const char* path) const NN_NOEXCEPT
{
    auto l = util::SNPrintf(outFullPath, sizeof(outFullPath), "%s:/%s", MountName, path);
    NN_ABORT_UNLESS(l < FullPathLengthMax);
}

}}} // ~namespace nn::nd::detail
