﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/migration/user/migration_UserMigrationProgressMonitor.h>

#include <algorithm>
#include <mutex>

#include <nn/migration/detail/migration_Diagnosis.h>
#include <nn/os/os_Tick.h>

namespace nn { namespace migration { namespace user {

namespace {

inline float GetBps(size_t dataSize, const TimeSpan& duration) NN_NOEXCEPT
{
    auto seconds = std::max(duration.GetSeconds(), int64_t(1));
    return static_cast<float>(static_cast<double>(dataSize) / static_cast<double>(seconds) * 8.0f);;
}

bool IsPrintableIntervalElapsed() NN_NOEXCEPT
{
    NN_FUNCTION_LOCAL_STATIC(util::optional<TimeSpan>, s_LastPrinted);

    auto current = os::GetSystemTick().ToTimeSpan();

    bool isPrintable = false;
    if (!(s_LastPrinted && (current - *s_LastPrinted).GetSeconds() <= 1))
    {
        isPrintable = true;
        s_LastPrinted = current;
    }
    return isPrintable;
}

} // ~namespace nn::migration::user::<anonymous>

// ---------------------------------------------------------------------------------------------
// PerformanceMonitor

PerformanceMonitor::PerformanceMonitor() NN_NOEXCEPT
    : m_Traffic(0)
    , m_IsMeasuring(false)
{
}

float PerformanceMonitor::GetThroughput() const NN_NOEXCEPT
{
    auto currentTraffic = m_Traffic;
    auto currentDuration = m_Duration;
    if (m_IsMeasuring)
    {
        currentTraffic += m_TrafficDelta;
        currentDuration += (os::GetSystemTick().ToTimeSpan() - m_StartedAt);
    }
    return GetBps(currentTraffic, currentDuration);
}
TimeSpan PerformanceMonitor::GetElapsedTime() const NN_NOEXCEPT
{
    auto currentDuration = m_Duration;
    if (m_IsMeasuring)
    {
        currentDuration += (os::GetSystemTick().ToTimeSpan() - m_StartedAt);
    }
    return currentDuration;
}

void PerformanceMonitor::Start() NN_NOEXCEPT
{
    NN_SDK_ASSERT(!m_IsMeasuring);

    m_StartedAt = os::GetSystemTick().ToTimeSpan();
    m_TrafficDelta = 0u;
    m_IsMeasuring = true;
}
void PerformanceMonitor::Update(size_t dataSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_IsMeasuring);

    m_TrafficDelta += dataSize;
}
void PerformanceMonitor::End() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_IsMeasuring);

    m_Traffic += m_TrafficDelta;
    m_Duration += (os::GetSystemTick().ToTimeSpan() - m_StartedAt);
    m_IsMeasuring = false;
}

// ---------------------------------------------------------------------------------------------
// ProgressMonitor

ProgressMonitor::ProgressMonitor(ClientContext& clientContext) NN_NOEXCEPT
    : m_ClientContext(clientContext)
    , m_Lock(true)
    , m_IsReceiving(false)
{
}

TransferInfo ProgressMonitor::GetTransferInfo() const NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(m_Lock);
    auto current = m_ClientContext.GetCurrentTransferInfo();
    if (m_IsReceiving)
    {
        current.sizeInBytes += m_Progress;
    }
    return current;
}

void ProgressMonitor::Start(const detail::DataInfo& dataInfo) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(m_Lock);
    NN_SDK_ASSERT(!m_IsReceiving);

    m_DataInfo = dataInfo;
    m_Progress = 0u;
    m_PerformanceMonitor.Start();
    m_IsReceiving = true;
}
void ProgressMonitor::Update(size_t dataSize) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(m_Lock);
    NN_SDK_ASSERT(m_IsReceiving);

    m_Progress += dataSize;
    m_PerformanceMonitor.Update(dataSize);

    if (IsPrintableIntervalElapsed())
    {
        auto current = GetTransferInfo();
        auto total = m_ClientContext.GetTotalTransferInfo();
        NN_MIGRATION_DETAIL_TRACE(
            "[ProgressMonitor] [%u of %u] %llu of %llu bytes\n",
            current.count, total.count,
            current.sizeInBytes, total.sizeInBytes);
        NN_MIGRATION_DETAIL_TRACE(
            "[ProgressMonitor]   - Current   : %llu of %llu bytes\n",
            m_Progress, m_DataInfo.codedSize);
        NN_MIGRATION_DETAIL_TRACE(
            "[ProgressMonitor]   - Throughput: %.3f bps\n",
            m_PerformanceMonitor.GetThroughput());
        NN_MIGRATION_DETAIL_TRACE(
            "[ProgressMonitor]   - Elapsed   : %lld sec\n",
            m_PerformanceMonitor.GetElapsedTime().GetSeconds());
    }
}
void ProgressMonitor::End() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(m_Lock);
    NN_SDK_ASSERT(m_IsReceiving);
    NN_SDK_ASSERT_EQUAL(m_Progress, static_cast<size_t>(m_DataInfo.codedSize));

    m_ClientContext.UpdateProgress(m_DataInfo);
    m_PerformanceMonitor.End();
    m_IsReceiving = false;

    auto current = GetTransferInfo();
    auto total = m_ClientContext.GetTotalTransferInfo();
    NN_MIGRATION_DETAIL_TRACE(
        "[ProgressMonitor] [%u of %u] %llu of %llu bytes\n",
        current.count, total.count,
        current.sizeInBytes, total.sizeInBytes);
    NN_MIGRATION_DETAIL_TRACE(
        "[ProgressMonitor]   - Throughput: %.3f bps\n",
        m_PerformanceMonitor.GetThroughput());
    NN_MIGRATION_DETAIL_TRACE(
        "[ProgressMonitor]   - Elapsed   : %lld sec\n",
        m_PerformanceMonitor.GetElapsedTime().GetSeconds());
}

}}} // ~namespace nn::migration::user
