﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/crypto.h>
#include <nn/crypto/crypto_Compare.h>
#include <nn/crypto/crypto_PasswordBasedKeyGenerator.h>
#include <nn/migration/idc/migration_KeyExchangeCommandEncryptor.h>
#include <nn/migration/detail/migration_Log.h>

#if defined (NN_BUILD_CONFIG_OS_HORIZON)
#include <mutex>
#include <nn/spl/spl_Api.h>
#include <nn/util/util_ScopeExit.h>
#endif

namespace nn { namespace migration { namespace idc {

#if defined (NN_BUILD_CONFIG_OS_HORIZON)

namespace
{
    os::Mutex g_Mutex(false);

    int LoadAesKey() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(g_Mutex.IsLockedByCurrentThread());
        int slotIndex;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::spl::AllocateAesKeySlot(&slotIndex));

        spl::AccessKey accessKey;
        const uint8_t KekSource[] = { 0xa3,0x8b,0xc9,0x9b,0xf1,0xe4,0x80,0x1c,0x2e,0xbc,0x3a,0x22,0xc0,0xb2,0x9b,0x03 };
        NN_ABORT_UNLESS_RESULT_SUCCESS(spl::GenerateAesKek(&accessKey, KekSource, sizeof(KekSource), spl::LatestKeyGeneration, 0));
        const uint8_t KeySource[] = { 0xaf,0xed,0xd1,0x7c,0x2a,0xbe,0x99,0x88,0x4a,0x25,0x87,0xae,0x04,0x99,0x19,0x82 };
        NN_ABORT_UNLESS_RESULT_SUCCESS(spl::LoadAesKey(slotIndex, accessKey, KeySource, sizeof(KeySource)));

        return slotIndex;
    }

    void UnloadAesKey(int slot) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(g_Mutex.IsLockedByCurrentThread());
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::spl::DeallocateAesKeySlot(slot));
    }

    void ProcessMessageImpl(void* out, size_t outSize, const void* in, size_t inSize, const void* iv, size_t ivSize) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(out);
        NN_SDK_REQUIRES_GREATER_EQUAL(outSize, inSize);
        NN_SDK_REQUIRES_NOT_NULL(in);
        NN_SDK_REQUIRES_GREATER(inSize, 0u);
        NN_SDK_REQUIRES_NOT_NULL(iv);
        NN_SDK_REQUIRES_EQUAL(ivSize, nn::crypto::CtrEncryptor<crypto::AesEncryptor128>::BlockSize);

        std::lock_guard<decltype(g_Mutex)> lock(g_Mutex);
        auto slotIndex = LoadAesKey();
        NN_UTIL_SCOPE_EXIT{ UnloadAesKey(slotIndex); };
        NN_ABORT_UNLESS_RESULT_SUCCESS(spl::ComputeCtr(out, outSize, slotIndex, in, inSize, iv, ivSize));
    }
}

KeyExchangeCommandEncryptor::KeyExchangeCommandEncryptor() NN_NOEXCEPT
{
    std::memset(m_Salt, 0, sizeof(m_Salt));
}

KeyExchangeCommandEncryptor::KeyExchangeCommandEncryptor(const KeyExchangeCommandConfig::Salt& salt) NN_NOEXCEPT
{
    std::memcpy(m_Salt, salt, sizeof(m_Salt));
}

void KeyExchangeCommandEncryptor::Encrypt(
    void* outEncrypted, size_t outEncryptedSize,
    const void* plain, size_t plainSize,
    const void* iv, size_t ivSize) const NN_NOEXCEPT
{
    ProcessMessageImpl(outEncrypted, outEncryptedSize, plain, plainSize, iv, ivSize);
}

void KeyExchangeCommandEncryptor::Decrypt(
    void* outPlain, size_t outPlainSize,
    const void* encrypted, size_t encryptedSize,
    const void* iv, size_t ivSize) const NN_NOEXCEPT
{
    ProcessMessageImpl(outPlain, outPlainSize, encrypted, encryptedSize, iv, ivSize);
}

void KeyExchangeCommandEncryptor::GenerateMac(void* outMac, size_t macSize, const void* data, size_t dataSize) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outMac);
    NN_SDK_REQUIRES_EQUAL(macSize, nn::crypto::CmacGenerator<nn::crypto::AesEncryptor128>::MacSize);
    NN_SDK_REQUIRES_NOT_NULL(data);
    NN_SDK_REQUIRES_GREATER(dataSize, 0u);

    std::lock_guard<decltype(g_Mutex)> lock(g_Mutex);
    auto slotIndex = LoadAesKey();
    NN_UTIL_SCOPE_EXIT{ UnloadAesKey(slotIndex); };
    NN_ABORT_UNLESS_RESULT_SUCCESS(spl::ComputeCmac(outMac, macSize, slotIndex, data, dataSize));
}

void KeyExchangeCommandEncryptor::GenerateMacWithSalt(void* outMac, size_t macSize, const KeyExchangeCommandConfig::Challenge& challenge) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outMac);
    NN_SDK_REQUIRES_EQUAL(macSize, nn::crypto::CmacGenerator<nn::crypto::AesEncryptor128>::MacSize);
    NN_SDK_REQUIRES_NOT_NULL(challenge);

    std::lock_guard<decltype(g_Mutex)> lock(g_Mutex);
    auto slotIndex = LoadAesKey();
    NN_UTIL_SCOPE_EXIT{ UnloadAesKey(slotIndex); };
    Bit8 data[KeyExchangeCommandConfig::ChallengeSize + KeyExchangeCommandConfig::SaltSize];
    std::memcpy(data, challenge, KeyExchangeCommandConfig::ChallengeSize);
    std::memcpy(data + KeyExchangeCommandConfig::ChallengeSize, m_Salt, sizeof(m_Salt));
    NN_ABORT_UNLESS_RESULT_SUCCESS(spl::ComputeCmac(outMac, macSize, slotIndex, data, sizeof(data)));
}

#endif

// DebugKeyExchangeCommandEncryptor

DebugKeyExchangeCommandEncryptor::DebugKeyExchangeCommandEncryptor() NN_NOEXCEPT
{
    std::memset(m_Salt, 0, sizeof(m_Salt));
}

DebugKeyExchangeCommandEncryptor::DebugKeyExchangeCommandEncryptor(const KeyExchangeCommandConfig::Salt& salt) NN_NOEXCEPT
{
    std::memcpy(m_Salt, &salt, sizeof(m_Salt));
}

void DebugKeyExchangeCommandEncryptor::Encrypt(
    void* outEncrypted, size_t outEncryptedSize,
    const void* plain, size_t plainSize,
    const void* iv, size_t ivSize) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outEncrypted);
    NN_SDK_REQUIRES_GREATER_EQUAL(outEncryptedSize, plainSize);
    NN_SDK_REQUIRES_NOT_NULL(plain);
    NN_SDK_REQUIRES_GREATER(plainSize, 0u);
    NN_SDK_REQUIRES_NOT_NULL(iv);

    NN_UNUSED(outEncryptedSize);
    NN_UNUSED(iv);
    NN_UNUSED(ivSize);

    std::memcpy(outEncrypted, plain, plainSize);
}

void DebugKeyExchangeCommandEncryptor::Decrypt(
    void* outPlain, size_t outPlainSize,
    const void* encrypted, size_t encryptedSize,
    const void* iv, size_t ivSize) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outPlain);
    NN_SDK_REQUIRES_GREATER_EQUAL(outPlainSize, encryptedSize);
    NN_SDK_REQUIRES_NOT_NULL(encrypted);
    NN_SDK_REQUIRES_GREATER(encryptedSize, 0u);
    NN_SDK_REQUIRES_NOT_NULL(iv);

    NN_UNUSED(outPlainSize);
    NN_UNUSED(iv);
    NN_UNUSED(ivSize);

    std::memcpy(outPlain, encrypted, encryptedSize);
}

void DebugKeyExchangeCommandEncryptor::GenerateMac(void* outMac, size_t macSize, const void* data, size_t dataSize) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outMac);
    NN_SDK_REQUIRES_EQUAL(macSize, nn::crypto::CmacGenerator<nn::crypto::AesEncryptor128>::MacSize);
    NN_UNUSED(data);
    NN_UNUSED(dataSize);
    std::memset(outMac, 0, macSize);
}

void DebugKeyExchangeCommandEncryptor::GenerateMacWithSalt(void* outMac, size_t macSize, const KeyExchangeCommandConfig::Challenge& challenge) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outMac);
    NN_SDK_REQUIRES_EQUAL(macSize, nn::crypto::CmacGenerator<nn::crypto::AesEncryptor128>::MacSize);
    NN_UNUSED(challenge);
    std::memset(outMac, 0, macSize);
}

}}};
