﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "migration_MigrationAuthenticator.h"

#include <cctype>

#include <nn/http/stream/http_ClientCertificate.h>
#include <nn/http/stream/http_CurlInputStream.h>
#include <nn/http/json/http_JsonErrorMap.h>
#include <nn/http/json/http_JsonPath.h>
#include <nn/http/json/http_RapidJsonApi.h>
#include <nn/http/http_Result.h>

#include <nn/nn_Version.h>
#include <nn/migration/detail/migration_Diagnosis.h>
#include <nn/migration/detail/migration_Result.h>
#include <nn/migration/detail/migration_Authenticator.h>
#include <nn/util/util_Base64.h>
#include <nn/util/util_FormatString.h>

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
#include <nn/settings/system/settings_FirmwareVersion.h>
#endif

namespace nn { namespace migration { namespace detail {
namespace {
const char* GetUserAgent() NN_NOEXCEPT
{
    NN_FUNCTION_LOCAL_STATIC(os::SdkMutex, s_Lock);
    std::lock_guard<decltype(s_Lock)> lock(s_Lock);

#define NN_HTTP_UA_AGENT     "libcurl"
#define NN_HTTP_UA_MODULE    "nnMigration"

#if defined(NN_BUILD_CONFIG_SPEC_GENERIC)
#define NN_HTTP_UA_PLATFORM  "5797bc97-d32c-48ad-b47e-3847bda0db7c" // Generic Windows
#elif defined(NN_BUILD_CONFIG_SPEC_NX)
#if defined(NN_BUILD_CONFIG_OS_WIN)
#define NN_HTTP_UA_PLATFORM  "159b8d57-d7af-4ce6-823e-c38d1d661918" // NX Windows
#elif defined(NN_BUILD_CONFIG_OS_HORIZON)
#define NN_HTTP_UA_PLATFORM  "789f928b-138e-4b2f-afeb-1acae821d897" // NX Horizon
#endif // OS
#endif // Spec

    typedef char UserAgentStringType[256];
    NN_FUNCTION_LOCAL_STATIC(UserAgentStringType, s_UserAgent, = {});

    NN_FUNCTION_LOCAL_STATIC(bool, s_Initialized, = false);
    if (!s_Initialized)
    {
        auto l = util::SNPrintf(
            s_UserAgent, sizeof(s_UserAgent),
            "%s (%s; %s; SDK %d.%d.%d.%d)",
            NN_HTTP_UA_AGENT, NN_HTTP_UA_MODULE, NN_HTTP_UA_PLATFORM,
            NN_SDK_VERSION_MAJOR, NN_SDK_VERSION_MINOR, NN_SDK_VERSION_MICRO, NN_SDK_VERSION_RELSTEP);
        NN_ABORT_UNLESS(l < sizeof(s_UserAgent), "[nn::migration] Error: Insufficient buffer for user agent: required %d bytes\n", l + 1);

        NN_MIGRATION_DETAIL_TRACE("[nn::migration] User agent is initialized as \"%s\"\n", s_UserAgent);
        s_Initialized = true;
    }
    return s_UserAgent;
}

const char KeyGenerationUrl[] = "https://migration.%.scsi.srv.nintendo.net/api/nx/v1/account_migrations/gen_key";
const char KeyExchangeUrl[] = "https://migration.%.scsi.srv.nintendo.net/api/nx/v1/account_migrations/get_key";
const Result MigrationAuthResultTable[] = {
    ResultMigrationAuthenticationStatusNo0000(),
    ResultMigrationAuthenticationStatusNo0001(),
    ResultMigrationAuthenticationStatusNo0002(),
    ResultMigrationAuthenticationStatusNo0003(),
    ResultMigrationAuthenticationStatusNo0004(),
    ResultMigrationAuthenticationStatusNo0005(),
    ResultMigrationAuthenticationStatusNo0006(),
    ResultMigrationAuthenticationStatusNo0007(),
    ResultMigrationAuthenticationStatusNo0008(),
    ResultMigrationAuthenticationStatusNo0009(),
};
NN_STATIC_ASSERT(std::extent<decltype(MigrationAuthResultTable)>::value == 10);

inline bool ExtractErrorCodeFromString(uint16_t* pOut, const char* value) NN_NOEXCEPT
{
    static const int MigrationAuthenticationErrorCodeLength = 4;

    if (value[MigrationAuthenticationErrorCodeLength] != '\0')
    {
        return false;
    }
    int16_t code = 0;
    for (auto i = 0; i < MigrationAuthenticationErrorCodeLength; ++ i)
    {
        auto c = value[i];
        if (!std::isdigit(c))
        {
            return false;
        }
        code = (code * 10) + c - '0';
    }
    *pOut = code;
    return true;
}

struct LookupEntry
{
    const char* path;
    bool found {false};

    explicit LookupEntry(const char* p) NN_NOEXCEPT
        : path(p)
    {
        NN_SDK_ASSERT(p);
    }
    NN_EXPLICIT_OPERATOR bool() const NN_NOEXCEPT
    {
        return found;
    }
    template <typename JsonPathType>
    bool CanAccept(const JsonPathType& jsonPath) const NN_NOEXCEPT
    {
        return !(*this) && jsonPath.Match(path);
    }
    void MarkAccepted() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(!found);
        found = true;
    }
};

class MigrationAuthenticationAdaptorBase
{
    NN_DISALLOW_COPY(MigrationAuthenticationAdaptorBase);

private:
    uint16_t m_ErrorCode;

    LookupEntry m_CodeEntry;
    LookupEntry m_MessageEntry;
    int m_ErrorIndex;
    char m_ErrorLabel[32];
    LookupEntry m_ErrorsEntry;

public:
    typedef http::json::JsonPath<8, 32> JsonPathType;

    void UpdateErrorLabel() NN_NOEXCEPT
    {
        auto l = util::SNPrintf(m_ErrorLabel, sizeof(m_ErrorLabel), "$.errors[%d]", m_ErrorIndex);
        if (!(l < sizeof(m_ErrorLabel)))
        {
            // インデックスがラベルに収まらないので空文字列にする
            std::strncpy(m_ErrorLabel, "", sizeof(m_ErrorLabel) - 1);
        }
        ++ m_ErrorIndex;
        m_ErrorsEntry = LookupEntry(m_ErrorLabel);
    }

public:
    MigrationAuthenticationAdaptorBase() NN_NOEXCEPT
        : m_ErrorCode(0u)
        // JSON から抽出するフィールド値
        , m_CodeEntry("$.code")
        , m_MessageEntry("$.message")
        , m_ErrorIndex(0)
        , m_ErrorsEntry("")
    {
        UpdateErrorLabel(); // エラーを "$.errors[0]" で初期化
    }

    Result Adapt() const NN_NOEXCEPT
    {
        if (!m_CodeEntry)
        {
            NN_RESULT_SUCCESS;
        }
        NN_RESULT_THROW_UNLESS(true
            && m_CodeEntry
            && static_cast<size_t>(m_ErrorCode) < std::extent<decltype(MigrationAuthResultTable)>::value,
            ResultMigrationAuthenticationUnacceptableResponse());
        NN_RESULT_THROW(MigrationAuthResultTable[m_ErrorCode]);
    }

    bool Update(const JsonPathType& jsonPath, const char* value, int valueLength) NN_NOEXCEPT
    {
        if (m_CodeEntry.CanAccept(jsonPath))
        {
            uint16_t code;
            if (!ExtractErrorCodeFromString(&code, value))
            {
                return true;
            }

            NN_MIGRATION_DETAIL_ERROR("Migration autentication Error: \"%s\"\n", value);
            m_ErrorCode = code;
            m_CodeEntry.MarkAccepted();
            return true;
        }
        else if (m_MessageEntry.CanAccept(jsonPath))
        {
            NN_MIGRATION_DETAIL_ERROR("Migration autentication Error: message: \"%s\"\n", value);
            m_MessageEntry.MarkAccepted();
            return true;
        }
        else if (m_ErrorsEntry.CanAccept(jsonPath))
        {
            NN_MIGRATION_DETAIL_ERROR("Migration autentication Error: [%d] \"%s\"\n", m_ErrorIndex - 1, value);
            UpdateErrorLabel();
            return true;
        }
        return false;
    }
};

class MigrationAuthenticationAdaptor
    : MigrationAuthenticationAdaptorBase
{
    NN_DISALLOW_COPY(MigrationAuthenticationAdaptor);

private:
    MigrationAuthenticationData* m_pAuthenticationData;
    OnetimeToken* m_pOtt;

    LookupEntry m_KeySeedEntry;
    LookupEntry m_DigestEntry;
    LookupEntry m_OnetimeTokenEntry;

public:
    typedef http::json::JsonPath<8, 32> JsonPathType;

public:
    MigrationAuthenticationAdaptor(MigrationAuthenticationData* pOut, OnetimeToken* pOutOtt) NN_NOEXCEPT
        : m_pAuthenticationData(pOut)
        , m_pOtt(pOutOtt)
        // JSON から抽出するフィールド値
        , m_KeySeedEntry("$.key_seed")
        , m_DigestEntry("$.digest")
        , m_OnetimeTokenEntry("$.onetime_token")
    {
    }
    explicit MigrationAuthenticationAdaptor(MigrationAuthenticationData* pOut) NN_NOEXCEPT
        : m_pAuthenticationData(pOut)
        , m_pOtt(nullptr)
        // JSON から抽出するフィールド値
        , m_KeySeedEntry("$.key_seed")
        , m_DigestEntry("$.digest")
        , m_OnetimeTokenEntry("$.onetime_token")
    {
    }

    Result Adapt() const NN_NOEXCEPT
    {
        NN_RESULT_DO(MigrationAuthenticationAdaptorBase::Adapt());

        if (m_KeySeedEntry && m_DigestEntry && !(m_pOtt && !m_OnetimeTokenEntry))
        {
            NN_RESULT_SUCCESS;
        }

#define PRINT_ENTRY_STATE(e) \
    NN_SDK_LOG("    %s : %s\n", (e)? "+": "-", (e).path)

        NN_MIGRATION_DETAIL_ERROR("Migration authentication failed\n");
        PRINT_ENTRY_STATE(m_KeySeedEntry);
        PRINT_ENTRY_STATE(m_DigestEntry);
        PRINT_ENTRY_STATE(m_OnetimeTokenEntry);
#undef PRINT_ENTRY_STATE
        NN_RESULT_THROW(ResultMigrationAuthenticationUnacceptableResponse());
    }

    void Update(const JsonPathType& jsonPath, const char* value, int valueLength) NN_NOEXCEPT
    {
        if (MigrationAuthenticationAdaptorBase::Update(jsonPath, value, valueLength))
        {
            return;
        }

        if (m_KeySeedEntry.CanAccept(jsonPath))
        {
            size_t rawSize;
            FsKeySeed seed;
            auto status = util::Base64::FromBase64String(&rawSize, seed.data, sizeof(seed.data), value, util::Base64::Mode_UrlSafe);
            if (!(status == util::Base64::Status_Success && rawSize == sizeof(seed.data)))
            {
                NN_MIGRATION_DETAIL_ERROR(
                    "Failed to decode base64 for FS key seed: %d, %zu bytes\n",
                    status,
                    status == util::Base64::Status_Success
                        ? rawSize
                        : 0);
                return;
            }

            m_pAuthenticationData->fsKeySeed = seed;
            m_KeySeedEntry.MarkAccepted();
        }
        else if (m_DigestEntry.CanAccept(jsonPath))
        {
            size_t rawSize;
            FsKeySeedDigest digest;
            auto status = util::Base64::FromBase64String(&rawSize, digest.data, sizeof(digest.data), value, util::Base64::Mode_UrlSafe);
            if (!(status == util::Base64::Status_Success && rawSize == sizeof(digest.data)))
            {
                NN_MIGRATION_DETAIL_ERROR(
                    "Failed to decode base64 for digest for FS key seed: %d, %zu bytes\n",
                    status,
                    status == util::Base64::Status_Success
                        ? rawSize
                        : 0);
                return;
            }

            m_pAuthenticationData->fsKeySeedDigest = digest;
            m_DigestEntry.MarkAccepted();
        }
        else if (m_pOtt && m_OnetimeTokenEntry.CanAccept(jsonPath))
        {
            if (!(valueLength < sizeof(m_pOtt->data)))
            {
                return;
            }
            std::strncpy(m_pOtt->data, value, sizeof(m_pOtt->data));
            m_OnetimeTokenEntry.MarkAccepted();
        }
    }

    // 以下対象外
    void Update(const JsonPathType& jsonPath, int64_t value) const NN_NOEXCEPT {}
    void Update(const JsonPathType&, std::nullptr_t) const NN_NOEXCEPT {}
    void Update(const JsonPathType&, bool) const NN_NOEXCEPT {}
    void Update(const JsonPathType&, uint64_t) const NN_NOEXCEPT {}
    void Update(const JsonPathType&, double) const NN_NOEXCEPT {}
    void NotifyObjectBegin(const JsonPathType&) const NN_NOEXCEPT {}
    void NotifyObjectEnd(const JsonPathType&) const NN_NOEXCEPT {}
};

Result ConvertHttpError(const Result& result) NN_NOEXCEPT
{
    if (http::ResultHttpStatusGone::Includes(result))
    {
        NN_RESULT_THROW(ResultMigrationAuthenticationHttp410());
    }
    else if (http::ResultHttpStatusServiceUnavailable::Includes(result))
    {
        NN_RESULT_THROW(ResultMigrationAuthenticationHttp503());
    }
    NN_RESULT_THROW(result);
}

// バッファ定義
struct MigrationAuthenticationBuffer
{
    char postDataBuffer[sizeof(DeviceAuthenticationToken) + sizeof(NsaIdToken) + 256];
    union
    {
        struct
        {
            char clientCertBuffer[4096]; // クライアント証明書程度
            char stringBuffer[sizeof(FsKeySeed) * 2 + 8];
            char inputBuffer[2048];
        } io;
    }u ;
};
NN_STATIC_ASSERT(sizeof(MigrationAuthenticationBuffer) <= RequiredBufferSizeForMigrationAuthentication);
NN_STATIC_ASSERT(std::alignment_of<MigrationAuthenticationBuffer>::value <= std::alignment_of<std::max_align_t>::value);
} // ~namespace nn::migration::detail::<anonymous>

Result AuthenticateMigrationSource(
    MigrationAuthenticationData* pOutAuth, OnetimeToken* pOutOtt,
    const NsaIdToken& nsaIdToken, const FsChallenge& fsChallenge,
    void* rawBuffer, size_t rawBufferSize, const Cancellable* pCancellable) NN_NOEXCEPT
{
    // 事前条件
    NN_SDK_REQUIRES(rawBuffer != nullptr);
    NN_SDK_REQUIRES(reinterpret_cast<uintptr_t>(rawBuffer) % std::alignment_of<MigrationAuthenticationBuffer>::value == 0);
    NN_SDK_REQUIRES(rawBufferSize >= sizeof(MigrationAuthenticationBuffer));
    NN_UNUSED(rawBufferSize);
    auto* buffer = reinterpret_cast<MigrationAuthenticationBuffer*>(rawBuffer);

    // curl_easy_init
    auto curlHandle = curl_easy_init();
    NN_ABORT_UNLESS(curlHandle);
    NN_UTIL_SCOPE_EXIT
    {
        curl_easy_cleanup(curlHandle);
    };

    // リクエストの作成
    http::stream::CurlInputStream stream(curlHandle, pCancellable);
    NN_RESULT_DO(stream.Initialize());
    stream.SetStringBuffer(buffer->u.io.stringBuffer, sizeof(buffer->u.io.stringBuffer));
    stream.SetInputBuffer(buffer->u.io.inputBuffer, sizeof(buffer->u.io.inputBuffer));

    char fsChallengeEncoded[32];
    auto status = util::Base64::ToBase64String(fsChallengeEncoded, sizeof(fsChallengeEncoded), fsChallenge.data, sizeof(fsChallenge.data), util::Base64::Mode_UrlSafe);
    NN_RESULT_THROW_UNLESS(status == util::Base64::Status_Success, ResultDataUnexpectedEncoding());

    auto l = nn::util::SNPrintf(
        buffer->postDataBuffer, sizeof(buffer->postDataBuffer),
        "nsa_id_token=%s&challenge=%s",
        nsaIdToken.data, fsChallengeEncoded);
    NN_SDK_ASSERT(static_cast<uint32_t>(l)  < sizeof(buffer->postDataBuffer));
    NN_UNUSED(l);
    NN_RESULT_DO(stream.SetHttpPost(buffer->postDataBuffer, false));

    http::stream::CertIoBuffer certBuffer = {buffer->u.io.clientCertBuffer, sizeof(buffer->u.io.clientCertBuffer)};
    NN_RESULT_DO(stream.SetSslContextHandler(http::stream::SslCtxHandlerWithClientCert, &certBuffer));
    NN_RESULT_DO(stream.SetUrl(KeyGenerationUrl));
    NN_RESULT_DO(stream.SetUserAgent(GetUserAgent()));

    // 通信の実行とデータの適用
    MigrationAuthenticationAdaptor adaptor(pOutAuth, pOutOtt);
    NN_RESULT_DO(stream.Open());
    NN_RESULT_DO(ConvertHttpError(http::json::ImportJsonByRapidJson<http::json::DefaultJsonErrorMap>(adaptor, stream, pCancellable)));
    stream.Close();
    return adaptor.Adapt();
}

Result AuthenticateMigrationDestination(
    MigrationAuthenticationData* pOutAuth,
    const NsaIdToken& nsaIdToken, const FsChallenge& fsChallenge, const OnetimeToken& ott,
    void* rawBuffer, size_t rawBufferSize, const Cancellable* pCancellable) NN_NOEXCEPT
{
    // 事前条件
    NN_SDK_REQUIRES(rawBuffer != nullptr);
    NN_SDK_REQUIRES(reinterpret_cast<uintptr_t>(rawBuffer) % std::alignment_of<MigrationAuthenticationBuffer>::value == 0);
    NN_SDK_REQUIRES(rawBufferSize >= sizeof(MigrationAuthenticationBuffer));
    NN_UNUSED(rawBufferSize);
    auto* buffer = reinterpret_cast<MigrationAuthenticationBuffer*>(rawBuffer);

    // curl_easy_init
    auto curlHandle = curl_easy_init();
    NN_ABORT_UNLESS(curlHandle);
    NN_UTIL_SCOPE_EXIT
    {
        curl_easy_cleanup(curlHandle);
    };

    // リクエストの作成
    http::stream::CurlInputStream stream(curlHandle, pCancellable);
    NN_RESULT_DO(stream.Initialize());
    stream.SetStringBuffer(buffer->u.io.stringBuffer, sizeof(buffer->u.io.stringBuffer));
    stream.SetInputBuffer(buffer->u.io.inputBuffer, sizeof(buffer->u.io.inputBuffer));

    char fsChallengeEncoded[32];
    auto status = util::Base64::ToBase64String(fsChallengeEncoded, sizeof(fsChallengeEncoded), fsChallenge.data, sizeof(fsChallenge.data), util::Base64::Mode_UrlSafe);
    NN_RESULT_THROW_UNLESS(status == util::Base64::Status_Success, ResultDataUnexpectedEncoding());

    auto l = nn::util::SNPrintf(
        buffer->postDataBuffer, sizeof(buffer->postDataBuffer),
        "nsa_id_token=%s&onetime_token=%s&challenge=%s",
        nsaIdToken.data, ott.data, fsChallengeEncoded);
    NN_SDK_ASSERT(static_cast<uint32_t>(l)  < sizeof(buffer->postDataBuffer));
    NN_UNUSED(l);
    NN_RESULT_DO(stream.SetHttpPost(buffer->postDataBuffer, false));

    http::stream::CertIoBuffer certBuffer = {buffer->u.io.clientCertBuffer, sizeof(buffer->u.io.clientCertBuffer)};
    NN_RESULT_DO(stream.SetSslContextHandler(http::stream::SslCtxHandlerWithClientCert, &certBuffer));
    NN_RESULT_DO(stream.SetUrl(KeyExchangeUrl));
    NN_RESULT_DO(stream.SetUserAgent(GetUserAgent()));

    // 通信の実行とデータの適用
    MigrationAuthenticationAdaptor adaptor(pOutAuth);
    NN_RESULT_DO(stream.Open());
    NN_RESULT_DO(ConvertHttpError(http::json::ImportJsonByRapidJson<http::json::DefaultJsonErrorMap>(adaptor, stream, pCancellable)));
    stream.Close();
    return adaptor.Adapt();
}

}}} // ~namespace nn::migration::detail

