﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/migration/detail/migration_ConnectionCreator.h>

#include <mutex>

#include <nn/crypto/crypto_Compare.h>
#include <nn/ldn/ldn_Api.h>
#include <nn/ldn/ldn_PrivateApi.h>
#include <nn/ldn/ldn_Result.h>
#include <nn/ldn/ldn_Settings.h>
#include <nn/ldn/ldn_SystemApi.h>
#include <nn/migration/migration_Result.h>
#include <nn/migration/idc/detail/migration_HandleSocketError.h>
#include <nn/migration/idc/migration_Result.h>
#include <nn/migration/detail/migration_Authenticator.h>
#include <nn/migration/detail/migration_Cancellable.h>
#include <nn/migration/detail/migration_ConnectionUtil.h>
#include <nn/migration/detail/migration_Diagnosis.h>
#include <nn/migration/detail/migration_InternalConfig.h>
#include <nn/migration/detail/migration_NetworkConnection.h>
#include <nn/migration/detail/migration_Result.h>
#include <nn/migration/detail/migration_Settings.h>
#include <nn/nifm/nifm_ApiRequest.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/socket/socket_Api.h>

namespace nn { namespace migration { namespace detail {

namespace {
const Bit8 LdnAdvertisementSalt[] = {
    0x37, 0x4A, 0x5E, 0xFE, 0xAF, 0x3A, 0xC4, 0x83, 0x0B, 0x61, 0xA9, 0x68, 0x28, 0x16, 0xD3, 0x78,
    0x88, 0xF8, 0xD4, 0x66, 0xA7, 0xCB, 0xE8, 0x9A, 0xBE, 0x41, 0x21, 0x6B, 0x60, 0x4E, 0x66, 0x96,
    0x49, 0x2B, 0xBE, 0x29, 0x85, 0x61, 0x45, 0x62, 0x72, 0xF3, 0x2C, 0x46, 0x31, 0x19, 0xAC, 0x68,
    0x70, 0x69, 0xF1, 0x17, 0xEC, 0x57, 0x10, 0xA2, 0xD4, 0xB6, 0x22, 0x75, 0xCD, 0x44, 0x96, 0xDF,
    0x45, 0x7A, 0x1F, 0xDE, 0xD4, 0x62, 0x1B, 0x72, 0x37, 0x9F, 0x65, 0x12, 0xE7, 0x73, 0xAE, 0x04,
    0xCC, 0xFB, 0x8F, 0x5F, 0x03, 0x8E, 0xE1, 0x82, 0xCB, 0x76, 0xD8, 0x2F, 0x2D, 0x9A, 0xF4, 0x10,
    0xDC, 0x3B, 0x7C, 0xB9, 0xE1, 0x39, 0x85, 0xC9, 0x73, 0xDD, 0x03, 0xBD, 0xA7, 0x5E, 0x2D, 0xBD,
    0xC3, 0x8A, 0xE4, 0xE2, 0xBF, 0x1C, 0xEE, 0x6E, 0x46, 0x1E, 0xEF, 0x7B, 0x20, 0x3C, 0x43, 0x90,
    0x98, 0x76, 0x4F, 0x6C, 0xAD, 0x14, 0x45, 0xA1, 0xAD, 0xCC, 0x37, 0xE6, 0x68, 0x49, 0xD0, 0x6C,
    0x6A, 0x7D, 0x27, 0xEE, 0x3D, 0x45, 0x16, 0x22, 0x42, 0x8D, 0xEE, 0xEA, 0x5D, 0xEA, 0x72, 0xFA,
    0x8C, 0x53, 0x5C, 0xDC, 0x80, 0x05, 0x63, 0xCA, 0x81, 0x85, 0xE9, 0x00, 0x2C, 0x6C, 0x3F, 0xC4,
    0xFC, 0x88, 0x35, 0x1D, 0xAF, 0x8F, 0x8F, 0x96, 0xAD, 0x84, 0x94, 0x01, 0x24, 0x02, 0x74, 0xBF,
    0xA7, 0xC4, 0x66, 0x96, 0x4C, 0x87, 0x9E, 0xD7, 0xA5, 0x91, 0x34, 0xD9, 0x4E, 0xF5, 0xBB, 0xD5,
    0xB6, 0x24, 0x81, 0x0A, 0x85, 0x49, 0x4C, 0x6D, 0x0C, 0x4D, 0x55, 0x00, 0x3C, 0x69, 0x81, 0x21,
    0xE6, 0x32, 0xC8, 0x8C, 0x77, 0x7E, 0xF0, 0xE4, 0x13, 0x28, 0xE8, 0xE2, 0x8C, 0x41, 0x6F, 0x18,
    0x5C, 0x40, 0x06, 0x88, 0xDA, 0xEB, 0x21, 0x87, 0x24, 0xE8, 0x7C, 0x07, 0xF2, 0xF1, 0xB4, 0x4B,
};

struct Advertise
{
    struct
    {
        account::NetworkServiceAccountId nsaId;
        char _reserved[88];
    } data; // sizeof(data) = 96 = 128 - 32
    char userData[256];
    char mac[crypto::Sha256Generator::HashSize]; // {data|userData|salt} の適当な Sha256

    void GetHash(void* hash, size_t hashSize) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(hashSize == sizeof(mac));

        crypto::Sha256Generator sha256gen;
        sha256gen.Initialize();
        sha256gen.Update(&data.nsaId, sizeof(data.nsaId));
        sha256gen.Update(&userData, sizeof(userData));
        sha256gen.Update(LdnAdvertisementSalt, sizeof(LdnAdvertisementSalt));
        sha256gen.GetHash(hash, hashSize);
    }
    bool Verify() const NN_NOEXCEPT
    {
        Bit8 hash[crypto::Sha256Generator::HashSize];
        NN_STATIC_ASSERT(sizeof(hash) == sizeof(mac));
        GetHash(hash, sizeof(hash));
        return crypto::IsSameBytes(hash, mac, sizeof(hash));
    }

    static Advertise Make(const account::NetworkServiceAccountId& nsaId, const UserMigrationServerProfile& profile) NN_NOEXCEPT
    {
        Advertise ad;
        ad.data.nsaId = nsaId;
        std::memcpy(ad.userData, profile.data, sizeof(ad.userData));
        ad.GetHash(ad.mac, sizeof(ad.mac));
        return ad;
    }
};
NN_STATIC_ASSERT(sizeof(Advertise) == ldn::AdvertiseDataSizeMax);

const int16_t LocalCommunicationVersion = 0;
const Bit8 LocalCommunicationPassphrase[] = {
    0x55, 0xee, 0x25, 0x18, 0x1d, 0xcb, 0x7d, 0x08, 0x07, 0xcb, 0xab, 0x73, 0xfd, 0xa1, 0xf5, 0x5a,
    0x5a, 0x79, 0x85, 0xb7, 0x69, 0x4b, 0xcf, 0x6b, 0x8f, 0x32, 0x88, 0x4d, 0xad, 0x94, 0x0b, 0xba,
    0x2d, 0x4a, 0xc0, 0x2e, 0xf9, 0x83, 0x64, 0xfe, 0xda, 0x04, 0x58, 0xbf, 0x98, 0x6e, 0xf9, 0x6f,
    0xa5, 0x08, 0xc6, 0xec, 0x97, 0xee, 0x93, 0x46, 0xa1, 0x2d, 0xc2, 0xc5, 0xfb, 0x59, 0xfd, 0x10,
};
NN_STATIC_ASSERT(sizeof(LocalCommunicationPassphrase) == 64);

void PrintLdnNetworkInfo(const ldn::NetworkInfo& info) NN_NOEXCEPT
{
    NN_MIGRATION_DETAIL_INFO("[ldn::NetworkInfo] %016llx, %04d, %02x%02x%02x%02x-%02x%02x%02x%02x-%02x%02x%02x%02x-%02x%02x%02x%02x\n",
        info.networkId.intentId.localCommunicationId, info.networkId.intentId.sceneId,
        info.networkId.sessionId.random[0], info.networkId.sessionId.random[1], info.networkId.sessionId.random[2], info.networkId.sessionId.random[3],
        info.networkId.sessionId.random[4], info.networkId.sessionId.random[5], info.networkId.sessionId.random[6], info.networkId.sessionId.random[7],
        info.networkId.sessionId.random[8], info.networkId.sessionId.random[9], info.networkId.sessionId.random[10], info.networkId.sessionId.random[11],
        info.networkId.sessionId.random[12], info.networkId.sessionId.random[13], info.networkId.sessionId.random[14], info.networkId.sessionId.random[15]);
    NN_MIGRATION_DETAIL_INFO("[ldn::NetworkInfo]   - BSSID: %02x:%02x:%02x:%02x:%02x:%02x\n",
        info.common.bssid.raw[0], info.common.bssid.raw[1], info.common.bssid.raw[2],
        info.common.bssid.raw[3], info.common.bssid.raw[4], info.common.bssid.raw[5]);
    NN_MIGRATION_DETAIL_INFO("[ldn::NetworkInfo]   - Type: %d\n", info.common.networkType);
    NN_MIGRATION_DETAIL_INFO("[ldn::NetworkInfo]   - Node: %d of %d\n", info.ldn.nodeCount, info.ldn.nodeCountMax);
    for (int i = 0; i < info.ldn.nodeCount; ++i)
    {
        const auto& node = info.ldn.nodes[i];
        NN_MIGRATION_DETAIL_INFO("[ldn::NetworkInfo]     - [%d] %s\n", i, node.isConnected ? "Connected" : "---");
        if (!node.isConnected)
        {
            continue;
        }
        NN_MIGRATION_DETAIL_INFO("[ldn::NetworkInfo]       - ID: %d\n", node.nodeId);
        NN_MIGRATION_DETAIL_INFO("[ldn::NetworkInfo]       - Version: %d\n", node.localCommunicationVersion);
        NN_MIGRATION_DETAIL_INFO(
            "[ldn::NetworkInfo]       - MAC address: %02x:%02x:%02x:%02x:%02x:%02x\n",
            node.macAddress.raw[0], node.macAddress.raw[1], node.macAddress.raw[2], node.macAddress.raw[3], node.macAddress.raw[4], node.macAddress.raw[5]);
        NN_MIGRATION_DETAIL_TRACE(
            "[ldn::NetworkInfo]       - IP address: %u.%u.%u.%u\n",
            reinterpret_cast<const Bit8*>(&node.ipv4Address.raw)[3], reinterpret_cast<const Bit8*>(&node.ipv4Address.raw)[2],
            reinterpret_cast<const Bit8*>(&node.ipv4Address.raw)[1], reinterpret_cast<const Bit8*>(&node.ipv4Address.raw)[0]);
    }
}

ldn::NetworkConfig MakeNetworkConfig() NN_NOEXCEPT
{
    ldn::NetworkConfig netConf = {};
    netConf.intentId = ldn::MakeIntentId(LocalCommunicationId, LocalCommunicationScene_User);
    netConf.channel = ldn::AutoChannel;
    netConf.nodeCountMax = 2;
    netConf.localCommunicationVersion = LocalCommunicationVersion;
    return netConf;
}

inline ldn::SecurityMode GetSecurityMode() NN_NOEXCEPT
{
#if defined(NN_SDK_BUILD_RELEASE)
    return ldn::SecurityMode_Product;
#else
    return ldn::SecurityMode_Debug;
#endif
}
ldn::SecurityConfig MakeSecurityConfig() NN_NOEXCEPT
{
    ldn::SecurityConfig secConf = {};
    secConf.securityMode = static_cast<uint16_t>(GetSecurityMode());
    std::memcpy(secConf.passphrase, LocalCommunicationPassphrase, sizeof(secConf.passphrase));
    NN_STATIC_ASSERT(sizeof(secConf.passphrase) <= sizeof(LocalCommunicationPassphrase));
    secConf.passphraseSize = static_cast<uint16_t>(sizeof(secConf.passphrase));
    return secConf;
}

ldn::ScanFilter MakeScanFilter() NN_NOEXCEPT
{
    ldn::ScanFilter filter = {};
    filter.networkId.intentId = ldn::MakeIntentId(LocalCommunicationId, LocalCommunicationScene_User);
    filter.networkType = ldn::NetworkType_Ldn;
    filter.flag = ldn::ScanFilterFlag_IntentId | ldn::ScanFilterFlag_NetworkType;
    return filter;
}

Result CreateLdnNetwork(const ldn::NetworkConfig netConf, const ldn::SecurityConfig secConf) NN_NOEXCEPT
{
    NN_SDK_ASSERT_EQUAL(ldn::GetState(), ldn::State_AccessPoint);

    NN_RESULT_DO(ldn::SetAdvertiseData(nullptr, 0));
    NN_RESULT_DO(ldn::ClearAcceptFilter());
    NN_RESULT_DO(ldn::SetStationAcceptPolicy(ldn::AcceptPolicy_AlwaysReject));

    ldn::UserConfig usrConf = {};
    NN_RESULT_DO(ldn::CreateNetwork(netConf, secConf, usrConf));
    NN_SDK_ASSERT_EQUAL(ldn::GetState(), ldn::State_AccessPointCreated);
    NN_RESULT_SUCCESS;
}

Result WaitLdnConnected(const Advertise& ad, int timeoutSeconds, ldn::NetworkInfo* buffer, const Cancellable* pCancellable) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!IsCanceled(pCancellable), ResultCanceled());

    NN_RESULT_DO(ldn::SetStationAcceptPolicy(ldn::AcceptPolicy_AlwaysAccept));
    NN_RESULT_DO(ldn::SetAdvertiseData(&ad, sizeof(ad)));

    os::TimerEvent timer(os::EventClearMode_ManualClear);
    timer.StartOneShot(TimeSpan::FromSeconds(timeoutSeconds));
    os::SystemEvent e;
    ldn::AttachStateChangeEvent(e.GetBase());

    bool connected = false;
    ldn::MacAddress clientMacAddr = {};
    while (!connected && !timer.TryWait())
    {
        if (!e.TimedWait(TimeSpan::FromMilliSeconds(500)))
        {
            NN_RESULT_THROW_UNLESS(!IsCanceled(pCancellable), ResultCanceled());
            continue;
        }

        NN_RESULT_DO(ldn::GetNetworkInfo(buffer));
        PrintLdnNetworkInfo(*buffer);
        if (buffer->ldn.nodeCount < 2)
        {
            continue;
        }
        NN_SDK_ASSERT(buffer->ldn.nodeCount == 2);

        for (auto i = 0; i < buffer->ldn.nodeCount; ++ i)
        {
            auto& node = buffer->ldn.nodes[i];
            if (node.isConnected && node.nodeId != 0)
            {
                clientMacAddr = node.macAddress;
                connected = true;
                break;
            }
        }
    }
    NN_RESULT_THROW_UNLESS(connected, ResultConnectionWaitingTimeout());

    NN_RESULT_DO(ldn::SetStationAcceptPolicy(ldn::AcceptPolicy_WhiteList));
    NN_RESULT_DO(ldn::AddAcceptFilterEntry(clientMacAddr));
    NN_RESULT_SUCCESS;
}

Result ConnectToLdnAccessPoint(
    ldn::Ipv4Address* pOutServer,
    const ldn::NetworkInfo& server, const ldn::SecurityConfig secConf, int16_t version, int timeoutSeconds,
    ldn::NetworkInfo* buffer, const Cancellable* pCancellable) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!IsCanceled(pCancellable), ResultCanceled());

    os::TimerEvent timer(os::EventClearMode_ManualClear);
    timer.StartOneShot(TimeSpan::FromSeconds(timeoutSeconds));

    bool connected = false;
    while (!connected && !timer.TryWait())
    {
        ldn::UserConfig usrConf = {};
        auto r = ldn::Connect(server, secConf, usrConf, version, ldn::ConnectOption_None);
        NN_RESULT_THROW_UNLESS(false
            || r.IsSuccess()
            || ldn::ResultConnectionTimeout::Includes(r)
            || ldn::ResultNetworkNotFound::Includes(r),
            r);
        if (!r.IsSuccess())
        {
            NN_RESULT_THROW_UNLESS(!IsCanceled(pCancellable), ResultCanceled());
            NN_MIGRATION_DETAIL_TRACE(
                "[ConnectToLdnAccessPoint] Retry caused by %03d-%04d(0x%08lx)\n",
                r.GetModule(), r.GetDescription(), r.GetInnerValueForDebug());
            os::SleepThread(TimeSpan::FromMilliSeconds(500));
            continue;
        }

        connected = true;
    }
    NN_RESULT_THROW_UNLESS(connected, ResultConnectingTimeout());

    bool success = false;
    NN_UTIL_SCOPE_EXIT
    {
        if (!success)
        {
            NN_MIGRATION_DETAIL_DO_IGNORING_FAILURE(ldn::Disconnect());
        }
    };

    NN_RESULT_DO(ldn::GetNetworkInfo(buffer));
    PrintLdnNetworkInfo(*buffer);
    NN_RESULT_THROW_UNLESS(buffer->ldn.nodeCount >= 2, ResultConnectionLostInProcessing());
    NN_SDK_ASSERT(buffer->ldn.nodeCount == 2);

    for (auto i = 0; i < buffer->ldn.nodeCount; ++ i)
    {
        auto& node = buffer->ldn.nodes[i];
        if (node.isConnected && node.nodeId == 0)
        {
            *pOutServer = node.ipv4Address;
            success = true;
            NN_RESULT_SUCCESS;
        }
    }
    NN_RESULT_THROW(ResultConnectionLostInProcessing());
}

} // ~namesapce nn::migration::detail::<anonymous>

// ---------------------------------------------------------------------------------------------
// LdnServerConnectionCreator
LdnServerConnectionCreator::LdnServerConnectionCreator() NN_NOEXCEPT
    : m_Lock(false)
    , m_IsLdnInitialized(false)
{
}
LdnServerConnectionCreator::~LdnServerConnectionCreator() NN_NOEXCEPT
{
    if (m_IsLdnInitialized)
    {
        Reset();
    }
}
Result LdnServerConnectionCreator::EnableLdn() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(m_Lock);
    NN_ABORT_UNLESS(!m_IsLdnInitialized);

    // 初期化
    NN_RESULT_DO(ldn::InitializeSystem());
    NN_UTIL_SCOPE_EXIT
    {
        if (!m_IsLdnInitialized)
        {
            ldn::FinalizeSystem();
        }
    };

    // HighSpeed Mode
    NN_RESULT_DO(ldn::SetOperationMode(ldn::OperationMode_HighSpeed));

    // AP の作成
    NN_RESULT_DO(ldn::OpenAccessPoint());
    NN_UTIL_SCOPE_EXIT
    {
        if (!m_IsLdnInitialized)
        {
            NN_MIGRATION_DETAIL_DO_IGNORING_FAILURE(ldn::CloseAccessPoint());
        }
    };

    // ネットワークの作成
    NN_RESULT_DO(CreateLdnNetwork(MakeNetworkConfig(), MakeSecurityConfig()));
    m_IsLdnInitialized = true;
    NN_RESULT_SUCCESS;
}
void LdnServerConnectionCreator::Reset() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(m_Lock);
    NN_ABORT_UNLESS(m_IsLdnInitialized);
    NN_MIGRATION_DETAIL_DO_IGNORING_FAILURE(ldn::DestroyNetwork());
    NN_MIGRATION_DETAIL_DO_IGNORING_FAILURE(ldn::CloseAccessPoint());
    ldn::FinalizeSystem();
    m_IsLdnInitialized = false;
}

Result LdnServerConnectionCreator::CreateNetwork(
    int* pOutConnection,
    const account::NetworkServiceAccountId& nsaId, const UserMigrationServerProfile& profile,
    const detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    ldn::Ipv4Address ipAddr;
    ldn::SubnetMask mask;
    NN_RESULT_DO(ldn::GetIpv4Address(&ipAddr, &mask));
    NN_MIGRATION_DETAIL_TRACE(
        "[LdnServerConnectionCreator] Server IP Address: %u.%u.%u.%u\n",
        reinterpret_cast<Bit8*>(&ipAddr.raw)[3], reinterpret_cast<Bit8*>(&ipAddr.raw)[2],
        reinterpret_cast<Bit8*>(&ipAddr.raw)[1], reinterpret_cast<Bit8*>(&ipAddr.raw)[0]);

    auto acceptSocket = socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp);
    NN_RESULT_THROW_UNLESS(acceptSocket >= 0, idc::detail::HandleSocketError(socket::GetLastError()));
    bool success = false;
    NN_UTIL_SCOPE_EXIT
    {
        if (!success)
        {
            NN_ABORT_UNLESS_EQUAL(socket::Close(acceptSocket), 0);
        }
    };

    NN_MIGRATION_DETAIL_TRACE("[LdnServerConnectionCreator] Creating port to listen\n");
    NN_RESULT_DO(SetupConnectionPort(acceptSocket, ipAddr, ServerPort_User));

    NN_MIGRATION_DETAIL_TRACE("[LdnServerConnectionCreator] Waiting for LDN connection\n");
    Advertise ad = Advertise::Make(nsaId, profile);
    NN_RESULT_DO(WaitLdnConnected(ad, GetLdnConnectionTimeoutSeconds(), &m_NetworkInfoBuffer, pCancellable));

    *pOutConnection = acceptSocket;
    success = true;
    NN_RESULT_SUCCESS;
}


// ---------------------------------------------------------------------------------------------
// LdnClientConnectionCreator

const size_t LdnClientConnectionCreator::ScanCountMax;
const size_t LdnClientConnectionCreator::ListCountMax;

LdnClientConnectionCreator::LdnClientConnectionCreator() NN_NOEXCEPT
    : m_Lock(false)
    , m_IsLdnInitialized(false)
    , m_ServerCount(0u)
{
}
LdnClientConnectionCreator::~LdnClientConnectionCreator() NN_NOEXCEPT
{
    if (m_IsLdnInitialized)
    {
        Reset();
    }
}
Result LdnClientConnectionCreator::EnableLdnIfNot() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(m_Lock);
    if (!m_IsLdnInitialized)
    {
        // 初期化
        NN_RESULT_DO(ldn::InitializeSystem());
        NN_UTIL_SCOPE_EXIT
        {
            if (!m_IsLdnInitialized)
            {
                ldn::FinalizeSystem();
            }
        };

        // HighSpeed Mode
        NN_RESULT_DO(ldn::SetOperationMode(ldn::OperationMode_HighSpeed));

        // Station の作成
        NN_RESULT_DO(ldn::OpenStation());
        m_IsLdnInitialized = true;
    }
    NN_RESULT_SUCCESS;
}
void LdnClientConnectionCreator::Reset() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(m_Lock);
    NN_ABORT_UNLESS(m_IsLdnInitialized);
    NN_MIGRATION_DETAIL_DO_IGNORING_FAILURE(ldn::CloseStation());
    ldn::FinalizeSystem();
    m_IsLdnInitialized = false;
}
Result LdnClientConnectionCreator::ScanServers(
    size_t* pOutActual, UserMigrationServerInfo* pOut, size_t count,
    const account::NetworkServiceAccountId& nsaId,
    const Cancellable* pCancellable) NN_NOEXCEPT
{
    NN_UNUSED(pCancellable);

    NN_RESULT_DO(EnableLdnIfNot());
    bool success = false;
    NN_UTIL_SCOPE_EXIT
    {
        if (!success)
        {
            Reset();
        }
    };

    ldn::ScanFilter filter = MakeScanFilter();
    int actualCount;
    NN_RESULT_DO(ldn::Scan(m_ScannedInfos, &actualCount, std::extent<decltype(m_ScannedInfos)>::value, filter, ldn::AutoChannel));
    NN_MIGRATION_DETAIL_TRACE("[LdnClientConnectionCreator]   -> count=%d\n", actualCount);

    size_t filled = 0;
    for (auto i = 0; i < actualCount && filled < count && filled < std::extent<decltype(m_Servers)>::value; ++ i)
    {
        const auto& info = m_ScannedInfos[i];
        if (!(true
            && info.ldn.securityMode == GetSecurityMode()
            && info.ldn.nodeCountMax == 2 && info.ldn.nodeCount < 2
            && info.ldn.advertiseDataSize == sizeof(Advertise)))
        {
            continue;
        }

        Advertise ad;
        std::memcpy(&ad, info.ldn.advertiseData, sizeof(ad));
        NN_MIGRATION_DETAIL_TRACE("[LdnClientConnectionCreator]     [%d] %s, %016llx\n", i, ad.Verify() ? "+" : "-", ad.data.nsaId);

        if (ad.Verify() && ad.data.nsaId == nsaId)
        {
            m_Servers[filled].id = util::GenerateUuid();
            m_Servers[filled].info = info;

            pOut[filled].id = m_Servers[filled].id;
            std::memcpy(&pOut[filled].profile, ad.userData, sizeof(pOut[filled].profile));

            ++filled;
        }
    }
    *pOutActual = m_ServerCount = filled;
    success = true;
    NN_RESULT_SUCCESS;
}
Result LdnClientConnectionCreator::ConnectImpl(
    idc::SocketConnection* pOut,
    const ldn::NetworkInfo& server, const account::NetworkServiceAccountId& nsaId,
    const Cancellable* pCancellable) NN_NOEXCEPT
{
    ldn::Ipv4Address ipAddr;
    NN_RESULT_DO(ConnectToLdnAccessPoint(
        &ipAddr,
        server, MakeSecurityConfig(), LocalCommunicationVersion, detail::GetTransferLowSpeedTimeoutSeconds(),
        &m_NetworkInfoBuffer, pCancellable));
    NN_MIGRATION_DETAIL_TRACE(
        "[LdnClientConnectionCreator] Server IP Address: %u.%u.%u.%u\n",
        reinterpret_cast<Bit8*>(&ipAddr.raw)[3], reinterpret_cast<Bit8*>(&ipAddr.raw)[2],
        reinterpret_cast<Bit8*>(&ipAddr.raw)[1], reinterpret_cast<Bit8*>(&ipAddr.raw)[0]);

    NN_MIGRATION_DETAIL_TRACE("[LdnClientConnectionCreator] Connecting to server port\n");
    auto clientSocket = socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp);
    NN_RESULT_THROW_UNLESS(clientSocket >= 0, idc::detail::HandleSocketError(socket::GetLastError()));
    NN_RESULT_DO(ConnectToPort(clientSocket, ipAddr, ServerPort_User));

    *pOut = idc::SocketConnection(clientSocket);
    NN_RESULT_SUCCESS;
}

}}} // ~namespace nn::migration::detail
