﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "manu_CreateUfioRequest.h"
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>
#include <nn/sf/sf_Types.h>
#include <nn/manu/manu_IManu.sfdl.h>

#include <string>

namespace nn { namespace manu {

namespace
{
// Shim ライブラリ実装用のサービスオブジェクトへの共有ポインタ
nn::sf::SharedPointer<IManu> g_UfioRequest;
}

namespace server
{
    void InitializeForUfio(int portNumber) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(!g_UfioRequest);
        g_UfioRequest = GetUfioRequestByDfc();
        g_UfioRequest->InitializeForUfio(portNumber);
        g_UfioRequest = nullptr;
    }

    void InitializeForUsbTransfer(int portNumber) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(!g_UfioRequest);
        g_UfioRequest = GetUfioRequestByDfc();
        g_UfioRequest->InitializeForUsbTransfer(portNumber);
        g_UfioRequest = nullptr;
    }
}

void InitializeUfio() NN_NOEXCEPT
{
    NN_SDK_ASSERT(!g_UfioRequest);
    g_UfioRequest = GetUfioRequestByHipc();
}

void FinalizeUfio() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_UfioRequest);
    // 共有ポインタへの nullptr の代入で解放できる。
    g_UfioRequest = nullptr;
}

nn::Result GetFileSize(size_t* pOutData, const char* path) NN_NOEXCEPT
{
    auto pathLength = std::strlen(path) + 1;

    nn::sf::InArray<char> inArray(path, pathLength);
    nn::sf::Out<uint64_t> outData(reinterpret_cast<uint64_t*>(pOutData));

    auto result = g_UfioRequest->GetFileSize(outData, inArray);
    *pOutData = *outData;
    return result;
}

nn::Result ReadFromHost(void* pOutBuffer, size_t bufferSize, const char* path, size_t offset, size_t readSize) NN_NOEXCEPT
{
    auto pathLength = std::strlen(path) + 1;
    nn::sf::InArray<char> inArray(path, pathLength);
    nn::sf::OutBuffer outBuffer(static_cast<char*>(pOutBuffer), readSize);

    return g_UfioRequest->ReadFromHost(outBuffer, bufferSize, inArray, offset, readSize);
}

nn::Result WriteToHost(const void* pBuffer, size_t bufferSize, const char* path, size_t offset, size_t writeSize) NN_NOEXCEPT
{
    auto pathLength = std::strlen(path) + 1;
    nn::sf::InArray<char> inArray(path, pathLength);
    nn::sf::InBuffer inBuffer(static_cast<const char*>(pBuffer), writeSize);

    return g_UfioRequest->WriteToHost(inBuffer, bufferSize, inArray, offset, writeSize);
}

nn::Result UsbRead(size_t* pOutSuccessReadSize, void* pOutBuffer, size_t bufferSize, size_t readSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutSuccessReadSize);
    NN_SDK_ASSERT_NOT_NULL(pOutBuffer);
    NN_SDK_ASSERT_LESS_EQUAL(readSize, bufferSize);

    nn::sf::Out<uint64_t> outData(reinterpret_cast<uint64_t*>(pOutSuccessReadSize));
    nn::sf::OutBuffer outBuffer(static_cast<char*>(pOutBuffer), readSize);

    auto result = g_UfioRequest->UsbRead(outData, outBuffer, bufferSize, readSize);
    *pOutSuccessReadSize = *outData;
    return result;
}

nn::Result UsbWrite(size_t* pOutSuccessWriteSize, const void* pBuffer, size_t bufferSize, size_t writeSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutSuccessWriteSize);
    NN_SDK_ASSERT_NOT_NULL(pBuffer);
    NN_SDK_ASSERT_LESS_EQUAL(writeSize, bufferSize);

    nn::sf::Out<uint64_t> outData(reinterpret_cast<uint64_t*>(pOutSuccessWriteSize));
    nn::sf::InBuffer inBuffer(static_cast<const char*>(pBuffer), writeSize);

    auto result = g_UfioRequest->UsbWrite(outData, inBuffer, bufferSize, writeSize);
    *pOutSuccessWriteSize = *outData;
    return result;
}

}}
