﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_TimeSpan.h>
#include <nn/dd/dd_Cache.h>
#include <nn/usb/usb_Device.h>
#include <nn/manu/manu_Result.public.h>
#include <nn/settings/system/settings_SerialNumber.h>

#include "manu_Usb.h"

namespace nn { namespace manu { namespace detail {

bool UsbTransferPipe::WaitUsbStateEqualTo(nn::usb::UsbState usbStatus, nn::TimeSpan timespan) NN_NOEXCEPT
{
    nn::usb::UsbState currentState;

    NN_ABORT_UNLESS_RESULT_SUCCESS(dsClient.GetState(&currentState));
    while(currentState != usbStatus)
    {
        auto usbStateChangeEvent = dsClient.GetStateChangeEvent();
        auto isStateChange = nn::os::TimedWaitSystemEvent(usbStateChangeEvent, timespan);

        if(isStateChange)
        {
            nn::os::ClearSystemEvent(usbStateChangeEvent);
            break;
        }
        else
        {
            return false;
        }
    }

    return true;
}

bool UsbTransferPipe::WaitXferCompletion(::nn::usb::DsEndpoint& endPoint, nn::TimeSpan timespan) NN_NOEXCEPT
{
    auto usbCompletionEvent = endPoint.GetCompletionEvent();
    auto isCompletion =  nn::os::TimedWaitSystemEvent(usbCompletionEvent, timespan);

    if(isCompletion)
    {
        nn::os::ClearSystemEvent(usbCompletionEvent);
    }

    return isCompletion;
}

bool UsbTransferPipe::CanTransfer() NN_NOEXCEPT
{
    nn::usb::UsbState usbStatus;
    NN_ABORT_UNLESS_RESULT_SUCCESS(dsClient.GetState(&usbStatus));

    return (usbStatus == nn::usb::UsbState_Configured);
}

bool UsbTransferPipe::IsUrbEmpty(::nn::usb::DsEndpoint& endPoint) NN_NOEXCEPT
{
    nn::usb::UrbReport urb;
    endPoint.GetUrbReport(&urb);

    return (urb.count == 0);
}

::nn::Result UsbTransferPipe::Initialize(uint8_t interfaceNumber) NN_NOEXCEPT
{
    // シリアルナンバーが取得できたらiSerialNumberに設定する
    {
        // Print serial number to sring table
        // Get the hardware serial number.
        nn::settings::system::SerialNumber SerialNumber;
        SerialNumber.string[0] = '\0';

        nn::settings::system::GetSerialNumber(&SerialNumber);

        if(SerialNumber.string[0] == '\0')
        {
            strcpy(SerialNumber.string, "SerialNumber");
        }

        uint16_t *pDst  = &serialNumberStringDescriptor.wData[0];
        uint8_t *pSrc   = reinterpret_cast<uint8_t*>(SerialNumber.string);
        uint8_t count = 0;

        while (*pSrc)
        {
            *pDst++ = static_cast<uint16_t>(*pSrc++);
            count++;
        }

        serialNumberStringDescriptor.bLength = 2 + (count * 2);
    }

    NN_RESULT_DO(dsClient.Initialize(nn::usb::ComplexId_Tegra21x));


    // Interfce nummber 3 seems to be the first to initialize, we only need to do the string and device descriptors once
    if (interfaceNumber == 3)
    {
        NN_RESULT_DO(dsClient.ClearDeviceData());

        uint8_t stringIndex;

        // String index will be in order starting from 0 after clearing all the device data
        NN_RESULT_DO(dsClient.AddUsbStringDescriptor(&stringIndex, &languageStringDescriptor));
        NN_RESULT_DO(dsClient.AddUsbStringDescriptor(&stringIndex, &manufacturerStringDescriptor));
        NN_RESULT_DO(dsClient.AddUsbStringDescriptor(&stringIndex, &productStringDescriptor));
        NN_RESULT_DO(dsClient.AddUsbStringDescriptor(&stringIndex, &serialNumberStringDescriptor));

        // Device descriptor for all speeds
        NN_RESULT_DO(dsClient.SetUsbDeviceDescriptor(&fullSpeedDeviceDescriptor,   ::nn::usb::UsbDeviceSpeed_Full));
        NN_RESULT_DO(dsClient.SetUsbDeviceDescriptor(&highSpeedDeviceDescriptor,   ::nn::usb::UsbDeviceSpeed_High));
        NN_RESULT_DO(dsClient.SetUsbDeviceDescriptor(&superSpeedDeviceDescriptor,  ::nn::usb::UsbDeviceSpeed_Super));
        NN_RESULT_DO(dsClient.SetBinaryObjectStore(binaryObjectStore, sizeof(binaryObjectStore)));
    }

    // Interface, configuration data per interface for all speeds
    usbInterfaceDescriptor.bInterfaceNumber = interfaceNumber;

    uint8_t endpointAddressIn   = 0;
    uint8_t endpointAddressOut  = 0;

    switch (interfaceNumber)
    {
    case 3:
        endpointAddressIn   = 0x81;
        endpointAddressOut  = 0x01;
        break;
    case 1:
        endpointAddressIn   = 0x82;
        endpointAddressOut  = 0x02;
        break;
    case 2:
        endpointAddressIn   = 0x83;
        endpointAddressOut  = 0x03;
        break;
    default:
        break;
    }


    usbEndpointDescriptorsFullSpeed[0].bEndpointAddress     = endpointAddressIn;
    usbEndpointDescriptorsFullSpeed[1].bEndpointAddress     = endpointAddressOut;
    usbEndpointDescriptorsHighSpeed[0].bEndpointAddress     = endpointAddressIn;
    usbEndpointDescriptorsHighSpeed[1].bEndpointAddress     = endpointAddressOut;
    usbEndpointDescriptorsSuperSpeed[0].bEndpointAddress    = endpointAddressIn;
    usbEndpointDescriptorsSuperSpeed[1].bEndpointAddress    = endpointAddressOut;

    NN_RESULT_DO(dsInterface.Initialize(&dsClient, usbInterfaceDescriptor.bInterfaceNumber));
    NN_RESULT_DO(dsInterface.AppendConfigurationData(::nn::usb::UsbDeviceSpeed_Full, &usbInterfaceDescriptor, sizeof(::nn::usb::UsbInterfaceDescriptor)));
    NN_RESULT_DO(dsInterface.AppendConfigurationData(::nn::usb::UsbDeviceSpeed_Full, &usbEndpointDescriptorsFullSpeed[0], sizeof(::nn::usb::UsbEndpointDescriptor)));
    NN_RESULT_DO(dsInterface.AppendConfigurationData(::nn::usb::UsbDeviceSpeed_Full, &usbEndpointDescriptorsFullSpeed[1], sizeof(::nn::usb::UsbEndpointDescriptor)));

    NN_RESULT_DO(dsInterface.AppendConfigurationData(::nn::usb::UsbDeviceSpeed_High, &usbInterfaceDescriptor, sizeof(::nn::usb::UsbInterfaceDescriptor)));
    NN_RESULT_DO(dsInterface.AppendConfigurationData(::nn::usb::UsbDeviceSpeed_High, &usbEndpointDescriptorsHighSpeed[0], sizeof(::nn::usb::UsbEndpointDescriptor)));
    NN_RESULT_DO(dsInterface.AppendConfigurationData(::nn::usb::UsbDeviceSpeed_High, &usbEndpointDescriptorsHighSpeed[1], sizeof(::nn::usb::UsbEndpointDescriptor)));

    NN_RESULT_DO(dsInterface.AppendConfigurationData(::nn::usb::UsbDeviceSpeed_Super, &usbInterfaceDescriptor, sizeof(::nn::usb::UsbInterfaceDescriptor)));
    NN_RESULT_DO(dsInterface.AppendConfigurationData(::nn::usb::UsbDeviceSpeed_Super, &usbEndpointDescriptorsSuperSpeed[0], sizeof(::nn::usb::UsbEndpointDescriptor)));
    NN_RESULT_DO(dsInterface.AppendConfigurationData(::nn::usb::UsbDeviceSpeed_Super, &usbSuperSpeedUsbEndpointCompanionDescriptor, sizeof(::nn::usb::UsbEndpointCompanionDescriptor)));
    NN_RESULT_DO(dsInterface.AppendConfigurationData(::nn::usb::UsbDeviceSpeed_Super, &usbEndpointDescriptorsSuperSpeed[1], sizeof(::nn::usb::UsbEndpointDescriptor)));
    NN_RESULT_DO(dsInterface.AppendConfigurationData(::nn::usb::UsbDeviceSpeed_Super, &usbSuperSpeedUsbEndpointCompanionDescriptor, sizeof(::nn::usb::UsbEndpointCompanionDescriptor)));

    // Endpoints
    NN_RESULT_DO(dsEndpoints[0].Initialize(&dsInterface, endpointAddressIn));
    NN_RESULT_DO(dsEndpoints[1].Initialize(&dsInterface, endpointAddressOut));

    NN_RESULT_DO(dsInterface.Enable());

    // This will enable the device, do this on the last interface to be initialized
    if (interfaceNumber == 2)
    {
        NN_RESULT_DO(dsClient.EnableDevice());
    }

    NN_RESULT_SUCCESS;
}

::nn::Result UsbTransferPipe::Finalize() NN_NOEXCEPT
{
    // It would be good if we can tell if this is the last interface to be torn down
    dsClient.DisableDevice();

    NN_RESULT_DO(dsInterface.Disable());
    for(int i=0; i<static_cast<int>(EndPointIndex::MaxEndPointNum); i++)
    {
        NN_RESULT_DO(dsEndpoints[i].Cancel());
        NN_RESULT_DO(dsEndpoints[i].Finalize());
    }
    NN_RESULT_DO(dsInterface.Finalize());
    NN_RESULT_DO(dsClient.Finalize());

    NN_RESULT_SUCCESS;
}

::nn::Result UsbTransferPipe::Cancel() NN_NOEXCEPT
{
    for(int i=0; i<static_cast<int>(EndPointIndex::MaxEndPointNum); i++)
    {
        NN_RESULT_DO(dsEndpoints[i].Cancel());
    }

    NN_RESULT_SUCCESS;
}

::nn::Result UsbTransferPipe::Xfer(::nn::usb::DsEndpoint& endPoint, uint32_t *pOutBytesTransferred, void *buffer, const uint32_t bufferLength) NN_NOEXCEPT
{
    while(!WaitUsbStateEqualTo(nn::usb::UsbState_Configured, nn::TimeSpan::FromMilliSeconds(1000)));
    while(!IsUrbEmpty(endPoint))
    {
        nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(10) );
    }
    NN_RESULT_DO(endPoint.PostBuffer(pOutBytesTransferred, buffer, bufferLength));
    NN_RESULT_SUCCESS;
}

::nn::Result UsbTransferPipe::Receive(uint32_t *pOutBytesTransferred, void *buffer, const uint32_t bufferLength) NN_NOEXCEPT
{
    auto& endPoint = dsEndpoints[static_cast<int>(EndPointIndex::BulkOut)];
    return Xfer(endPoint, pOutBytesTransferred, buffer, bufferLength);
}

::nn::Result UsbTransferPipe::Send(uint32_t *pOutBytesTransferred, const void *buffer, const uint32_t bufferLength) NN_NOEXCEPT
{
    auto& endPoint = dsEndpoints[static_cast<int>(EndPointIndex::BulkIn)];
    return Xfer(endPoint, pOutBytesTransferred,  const_cast<void *>(buffer), bufferLength);
}

}}}
