﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Result.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os.h>
#include <string>

#include <nn/manu/server/manu_Log.h>
#include <nn/manu/manu_Result.public.h>

#include "manu_Request.h"
#include "manu_Types.h"
#include "manu_Usb.h"

namespace nn { namespace manu { namespace detail {

namespace
{
    UsbTransferPipe usbTransferPipe;
    const size_t UsbBufferSize=1024 * 128;
    NN_USB_DMA_ALIGN nn::Bit8 usbWorkBuffer[UsbBufferSize]={0};

    nn::Result SendHeader(RequestId id, size_t bodySize)
    {
        RequestHeader* header = reinterpret_cast<RequestHeader*>(&usbWorkBuffer[0]);
        header->MagicNumber   = MagicNumber;
        header->Request       = id;
        header->BodySize      = bodySize;
        uint32_t sendSize = 0;
        NN_RESULT_DO(usbTransferPipe.Send(&sendSize, header, sizeof(RequestHeader)));
        NN_RESULT_SUCCESS;
    }

    nn::Result ReceiveResult(ResultValue *result)
    {
        uint32_t receiveSize = 0;
        NN_RESULT_DO(usbTransferPipe.Receive(&receiveSize, result, sizeof(ResultValue)));
        NN_RESULT_SUCCESS;
    }
}

void InitializeForUfio(int portNumber) NN_NOEXCEPT
{
    usbTransferPipe.Initialize(static_cast<uint8_t>(portNumber));
}

nn::Result GetFileSize(size_t* pOutData, const char* path) NN_NOEXCEPT
{
    uint32_t pathLength = static_cast<uint32_t>(::std::strlen(path));
    NN_RESULT_DO(SendHeader(RequestId::GetFileSize, pathLength));
    char* body = reinterpret_cast<char*>(usbWorkBuffer);
    ::std::memcpy(body, path, pathLength);
    uint32_t sendSize = 0;
    usbTransferPipe.Send(&sendSize, body, pathLength);
    ResultValue *result = reinterpret_cast<ResultValue*>(usbWorkBuffer);
    NN_RESULT_DO(ReceiveResult(result));

    if(result->Id != ResultId::Success)
    {
        NN_MANU_ERROR_LOG("GetFileSize() Error received from host. 0x%0x\n", result->Id);
        return ResultHostPcError();
    }

    *pOutData = result->ReturnValue;

    NN_RESULT_SUCCESS;
}

nn::Result ReadFromHost(void* pOutBuffer, size_t bufferSize, const char* path, size_t offset, size_t readSize) NN_NOEXCEPT
{
    uint32_t pathLength = static_cast<uint32_t>(::std::strlen(path));
    uint32_t bodyLength = sizeof(readSize) + sizeof(offset) + pathLength;

    NN_RESULT_DO(SendHeader(RequestId::ReadFromHost, bodyLength));

    ReadWriteBody& body = *reinterpret_cast<ReadWriteBody*>(usbWorkBuffer);
    body.Length = readSize;
    body.Offset = offset;
    ::std::memcpy(body.Path, path, pathLength);
    uint32_t sendSize = 0;
    usbTransferPipe.Send(&sendSize, &body, bodyLength);

    ResultValue *result = reinterpret_cast<ResultValue*>(usbWorkBuffer);
    NN_RESULT_DO(ReceiveResult(result));
    if(result->Id != ResultId::Success)
    {
        NN_MANU_ERROR_LOG("ReadFromHost() ReadWriteBody Error received from host. 0x%0x\n", result->Id);
        return ResultHostPcError();
    }

    if(readSize + offset > result->ReturnValue)
    {
        NN_MANU_ERROR_LOG("Too large readSize. File (%s) is %d  bytes. read area is %d to %d\n",path, result->ReturnValue, offset, offset + readSize);
        return ResultOverSizeRead();
    }

    for(size_t remainSize=readSize; remainSize>0;)
    {
        auto receivedSize = readSize - remainSize;
        auto xferSize = ::std::min(remainSize, UsbBufferSize);
        uint32_t actualXferSize = 0;

        usbTransferPipe.Receive(&actualXferSize, usbWorkBuffer, static_cast<uint32_t>(xferSize));
        ::std::memcpy(reinterpret_cast<nn::Bit8*>(pOutBuffer) + receivedSize, usbWorkBuffer, actualXferSize);

        remainSize -= actualXferSize;
    }
    NN_RESULT_DO(ReceiveResult(result));
    if(result->Id != ResultId::Success)
    {
        NN_MANU_ERROR_LOG("ReadFromHost() Error received from host. 0x%0x\n", result->Id);
        return ResultHostPcError();
    }

    NN_RESULT_SUCCESS;
}

nn::Result WriteToHost(const void* pBuffer, size_t bufferSize, const char* path, size_t offset, size_t writeSize) NN_NOEXCEPT
{
    uint32_t pathLength = static_cast<uint32_t>(::std::strlen(path));
    uint32_t bodyLength = sizeof(writeSize) + sizeof(offset) + pathLength;

    NN_RESULT_DO(SendHeader(RequestId::WriteToHost, bodyLength));

    ReadWriteBody& body = *reinterpret_cast<ReadWriteBody*>(usbWorkBuffer);
    body.Length = writeSize;
    body.Offset = offset;
    ::std::memcpy(body.Path, path, pathLength);
    uint32_t sendSize = 0;
    usbTransferPipe.Send(&sendSize, &body, bodyLength);

    ResultValue *result = reinterpret_cast<ResultValue*>(usbWorkBuffer);
    NN_RESULT_DO(ReceiveResult(result));
    if(result->Id != ResultId::Success)
    {
        NN_MANU_ERROR_LOG("WriteToHost() ReadWriteBody Host Error 0x%0x\n", result->Id);
        return ResultHostPcError();
    }

    for(size_t remainSize=writeSize; remainSize>0;)
    {
        auto sentSize = writeSize - remainSize;
        auto xferSize = ::std::min(remainSize, UsbBufferSize);
        uint32_t successXferSize = 0;

        ::std::memcpy(usbWorkBuffer, reinterpret_cast<const nn::Bit8*>(pBuffer) + sentSize, xferSize);
        NN_RESULT_DO(usbTransferPipe.Send(&successXferSize, usbWorkBuffer, static_cast<uint32_t>(xferSize)));

        remainSize -= successXferSize;
    }

    NN_RESULT_DO(ReceiveResult(result));
    if(result->Id != ResultId::Success)
    {
        NN_MANU_ERROR_LOG("WriteToHost() End Host Error 0x%0x\n", result->Id);
        return ResultHostPcError();
    }

    NN_RESULT_SUCCESS;
}


}}}
