﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/ldn/ldn_NetworkId.h>
#include <nn/ldn/detail/ldn_Config.h>
#include <nn/ldn/detail/Debug/ldn_Log.h>
#include <nn/ldn/detail/Utility/ldn_ReverseByteOrder.h>
#include <nn/ldn/detail/Utility/ldn_Stringize.h>
#include <nn/util/util_FormatString.h>

#if defined(NN_LDN_BUILD_CONFIG_LOCAL_ID_CHECK_APPLICATION_META)
#include <nn/arp/arp_Api.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#endif

namespace nn { namespace ldn { namespace detail { namespace
{
    #if defined(NN_LDN_BUILD_CONFIG_LOCAL_ID_CHECK_APPLICATION_META)
    nn::ns::ApplicationControlProperty g_Property;
    #endif

}}}} // namespace nn::ldn::detail::<unnamed>

namespace nn { namespace ldn { namespace detail
{
    size_t ConvertToString(const IntentId& intentId, char* buffer, size_t bufferSize) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(buffer != nullptr);
        NN_SDK_ASSERT(StringizedIntentIdLength < bufferSize);
        return static_cast<size_t>(nn::util::SNPrintf(
            buffer, bufferSize, "%016llx-%02x%02x-%04x-%02x%02x%02x%02x",
            intentId.localCommunicationId, intentId._reserved1[0], intentId._reserved1[1],
            intentId.sceneId, intentId._reserved2[0], intentId._reserved2[1],
            intentId._reserved2[2], intentId._reserved2[3]));
    }

    size_t ConvertToString(const SessionId& sessionId, char* buffer, size_t bufferSize) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(buffer != nullptr);
        NN_SDK_ASSERT(StringizedSessionIdLength < bufferSize);
        return ConvertToHexString(buffer, bufferSize, &sessionId, sizeof(sessionId));
    }

    size_t ConvertToString(const NetworkId& networkId, char* buffer, size_t bufferSize) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(buffer != nullptr);
        NN_SDK_ASSERT(StringizedNetworkIdLength <= bufferSize);
        size_t length = 0;

        // IntentID を文字列化します。
        length += ConvertToString(networkId.intentId, buffer, bufferSize);

        // IntentID と SessionID の間に半角スペースを挟みます。
        buffer[length] = ' ';
        ++length;

        // SessionID を文字列化します。
        length += ConvertToString(networkId.sessionId, buffer + length, bufferSize - length);
        return length;
    }

    IntentId ReverseByteOrderImpl(const IntentId& src) NN_NOEXCEPT
    {
        IntentId dst = { };
        dst.localCommunicationId = ReverseByteOrder(src.localCommunicationId);
        dst.sceneId = ReverseByteOrder(src.sceneId);
        return dst;
    }

    SessionId ReverseByteOrderImpl(const SessionId& src) NN_NOEXCEPT
    {
        return src;
    }

    NetworkId ReverseByteOrderImpl(const NetworkId& src) NN_NOEXCEPT
    {
        NetworkId dst;
        dst.intentId = ReverseByteOrder(src.intentId);
        dst.sessionId = ReverseByteOrder(src.sessionId);
        return dst;
    }

    void GetDefaultLocalCommunicationId(Bit64* pOutId, Bit64 pid) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutId);
        #if defined(NN_LDN_BUILD_CONFIG_LOCAL_ID_CHECK_APPLICATION_META)
        nn::os::ProcessId processId = { pid };
        if (nn::arp::GetApplicationControlProperty(&g_Property, processId).IsSuccess())
        {
            *pOutId = g_Property.localCommunicationId[0];
        }
        else
        {
            NN_LDN_LOG_WARN("Failed to get application meta data.\n");
            *pOutId = 0;
        }
        #elif defined(NN_LDN_BUILD_CONFIG_LOCAL_ID_CHECK_DISABLED)
        NN_UNUSED(pid);
        *pOutId = 0;
        #endif
    }

    bool HasPermission(Bit64 pid, Bit64 localCommunicationId) NN_NOEXCEPT
    {
        #if defined(NN_LDN_BUILD_CONFIG_LOCAL_ID_CHECK_APPLICATION_META)
        nn::os::ProcessId processId = { pid };
        if (nn::arp::GetApplicationControlProperty(&g_Property, processId).IsFailure())
        {
            NN_LDN_LOG_WARN("Failed to get application meta data.\n");
            return true;
        }
        for (int i = 0; i < 8; ++i)
        {
            if (g_Property.localCommunicationId[i] == localCommunicationId)
            {
                return true;
            }
        }
        return false;
        #elif defined(NN_LDN_BUILD_CONFIG_LOCAL_ID_CHECK_DISABLED)
        NN_UNUSED(pid);
        NN_UNUSED(localCommunicationId);
        return true;
        #endif
    }

}}} // namespace nn::ldn
