﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/diag/text/diag_SdkTextLdn.h>
#include <nn/ldn/ldn_Ipv4Address.h>
#include <nn/ldn/detail/Utility/ldn_ReverseByteOrder.h>
#include <nn/util/util_FormatString.h>

namespace nn { namespace ldn
{
    const Ipv4Address MakeIpv4Address(Bit8 byte1, Bit8 byte2, Bit8 byte3, Bit8 byte4) NN_NOEXCEPT
    {
        Ipv4Address address = {
            static_cast<Bit32>((byte1 << 24) | (byte2 << 16) | (byte3 << 8) | byte4)};
        return address;
    }

    const Ipv4Address MakeIpv4Address(const Bit8 (&rawAddress)[4]) NN_NOEXCEPT
    {
        Ipv4Address address = {
            static_cast<Bit32>((rawAddress[0] << 24) | (rawAddress[1] << 16) |
                               (rawAddress[2] << 8)  | (rawAddress[3]))};
        return address;
    }

    bool operator == (Ipv4Address lhs, Ipv4Address rhs) NN_NOEXCEPT
    {
        return lhs.raw == rhs.raw;
    }

    const SubnetMask MakeSubnetMask(Bit8 byte1, Bit8 byte2, Bit8 byte3, Bit8 byte4) NN_NOEXCEPT
    {
        SubnetMask mask = {
            static_cast<Bit32>((byte1 << 24) | (byte2 << 16) | (byte3 << 8) | byte4)};
        NN_SDK_REQUIRES(
            ((~mask.raw) & ((~mask.raw) + 1)) == 0,
            NN_TEXT_LDN("指定されたアドレスは正しいサブネットマスクを構成していません。")
        );
        return mask;
    }

    const SubnetMask MakeSubnetMask(const Bit8 (&raw)[4]) NN_NOEXCEPT
    {
        SubnetMask mask = {
            static_cast<Bit32>((raw[0] << 24) | (raw[1] << 16) | (raw[2] << 8) | raw[3])};
        NN_SDK_REQUIRES(
            ((~mask.raw) & ((~mask.raw) + 1)) == 0,
            NN_TEXT_LDN("指定されたアドレスは正しいサブネットマスクを構成していません。")
        );
        return mask;
    }

    const SubnetMask MakeSubnetMask(int cidr) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(0 <= cidr && cidr <= 32);
        SubnetMask mask = { UINT32_C(0xFFFFFFFF) << (32 - cidr) };
        return mask;
    }

    const Ipv4Address MakeNetworkAddress(Ipv4Address address, SubnetMask mask) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(((~mask.raw) & ((~mask.raw) + 1)) == 0,
            NN_TEXT_LDN("サブネットマスクが不正です。"));
        Ipv4Address network = { address.raw & mask.raw };
        return network;
    }

    const Ipv4Address MakeBroadcastAddress(Ipv4Address address, SubnetMask mask) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(((~mask.raw) & ((~mask.raw) + 1)) == 0,
            NN_TEXT_LDN("サブネットマスクが不正です。"));
        Ipv4Address broadcast = { address.raw | (~mask.raw) };
        return broadcast;
    }

}} // namespace nn::ldn

namespace nn { namespace ldn { namespace detail
{
    size_t ConvertToString(Ipv4Address ipv4Address, char* buffer, size_t bufferSize) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(buffer != nullptr);
        NN_SDK_ASSERT(StringizedIpv4AddressLengthMax < bufferSize);
        Bit32 address = ConvertToNetworkByteOrder(ipv4Address.raw);
        const auto* p = reinterpret_cast<uint8_t*>(&address);
        return static_cast<size_t>(nn::util::SNPrintf(
            buffer, bufferSize, "%u.%u.%u.%u", p[0], p[1], p[2], p[3]));
    }

    const Ipv4Address ReverseByteOrderImpl(Ipv4Address src) NN_NOEXCEPT
    {
        Ipv4Address ipv4Address = { ReverseByteOrder(src.raw) };
        return ipv4Address;
    }

}}} // namespace nn::ldn
