﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <mutex>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/ldn/ldn_Result.h>
#include <nn/ldn/detail/ldn_Core.h>
#include <nn/ldn/detail/Authentication/ldn_AuthenticationDataBuilder.h>
#include <nn/ldn/detail/Debug/ldn_Assert.h>
#include <nn/ldn/detail/Debug/ldn_Log.h>
#include <nn/ldn/detail/NetworkInterface/ldn_NintendoEthernet.h>
#include <nn/ldn/detail/Utility/ldn_SecureRandom.h>
#include <nn/ldn/detail/Utility/ldn_Stringize.h>
#include <nn/result/result_HandlingUtility.h>

NN_DISABLE_WARNING_DEPRECATED_DECLARATIONS

namespace nn { namespace ldn { namespace detail { namespace
{
    // 送信キューの容量です。
    const size_t SendQueueCapacity = 4;

    // 切断通知用の受信キューの容量です。
    const size_t RejectReceiveQueueCapacity = 4;

    // 切断通知が届くまで待機する時間です。
    const TimeSpan RejectRequestWaitTime = nn::TimeSpan::FromMilliSeconds(400);

    /**
     * @brief       切断通知に使用するフレームです。
     */
    struct RejectRequest
    {
        EthernetHeader              ethernetHeader;
        OuiExtendedEthernetHeader   ouiExtendedEthernetHeader;
        NintendoEthernetHeader      nintendoEthernetHeader;
        int8_t                      reason;
        Bit8                        reserved[31];
    };
    NN_STATIC_ASSERT(sizeof(RejectRequest) == 52);

    /**
     * @brief       ネットワーク識別子から SSID を生成します。
     * @param[in]   networkId       SSID の生成に使用するネットワーク識別子です。
     * @return      生成された SSID です。
     */
    inline Ssid CreateSsidFromNetworkId(const NetworkId& networkId) NN_NOEXCEPT
    {
        const auto& sessionId = networkId.sessionId;

        // 入力データを 16 進文字列に変換して SSID として使用します。
        Ssid   ssid;
        size_t size = ConvertToHexString(
            ssid.raw, sizeof(ssid.raw), &sessionId, sizeof(sessionId));

        // 1 バイトが 2 文字に変換されるため、SSID は入力データの 2 倍のサイズになるはずです。
        NN_SDK_ASSERT_EQUAL(size, sizeof(SessionId) * 2);
        ssid.length = static_cast<uint8_t>(size);
        return ssid;
    }

}}}} // namespace nn::ldn::detail::<unnamed>

namespace nn { namespace ldn { namespace detail
{
    Core::Core(void* buffer, size_t bufferSize) NN_NOEXCEPT
        : m_CloseEvent(nn::os::EventClearMode_ManualClear),
          m_RejectEvent(nn::os::EventClearMode_ManualClear),
          m_NetworkInterfaceMutex(false),
          m_SessionMutex(false),
          m_Buffer(static_cast<impl::CoreBuffer*>(buffer)),
          m_pNetworkInterface(nullptr),
          m_SessionManager(&m_Buffer->sessionManager, sizeof(m_Buffer->sessionManager)),
          m_AuthenticationServer(m_Buffer->authentication, sizeof(m_Buffer->authentication)),
          m_AuthenticationClient(m_Buffer->authentication, sizeof(m_Buffer->authentication)),
          m_State(State_None)
    {
        NN_SDK_ASSERT_NOT_NULL(buffer);
        NN_SDK_ASSERT_ALIGNED(buffer, nn::os::ThreadStackAlignment);
        NN_SDK_ASSERT(RequiredBufferSize <= bufferSize);
        NN_UNUSED(bufferSize);
    }

    Core::~Core() NN_NOEXCEPT
    {
    }

    Result Core::StartLocalCommunication(INetworkInterface* pNetworkInterface) NN_NOEXCEPT
    {
        NN_LDN_REQUIRES_EQUAL(m_State, State_None);

        // ネットワーク・インタフェースを保存しておきます。
        {
            std::lock_guard<nn::os::Mutex> lock(m_NetworkInterfaceMutex);
            m_pNetworkInterface = pNetworkInterface;
        }

        // スキャンは AP/STA に関係なく実施するため、ここで用意しておきます。
        m_AdvertiseScanner.Initialize(
            m_Buffer->advertiseScanner, sizeof(m_Buffer->advertiseScanner),
            ScanResultCountMax, pNetworkInterface);

        m_State = State_Initialized;
        NN_RESULT_SUCCESS;
    }

    Result Core::StopLocalCommunication(bool isTriggeredBySystem) NN_NOEXCEPT
    {
        NN_LDN_REQUIRES_NOT_EQUAL(m_State, State_None);

        // 切断処理を行います。
        if (m_State == State_AccessPoint || m_State == State_AccessPointCreated)
        {
            CloseAccessPoint(isTriggeredBySystem);
        }
        else if (m_State == State_Station || m_State == State_StationConnected)
        {
            CloseStation(isTriggeredBySystem);
        }

        // スキャン周りのバッファを解放します。
        m_AdvertiseScanner.Finalize();

        // ネットワーク・インタフェースを解放します。
        {
            std::lock_guard<nn::os::Mutex> lock(m_NetworkInterfaceMutex);
            m_pNetworkInterface = nullptr;
        }

        // 終了処理に成功しました。
        m_State = State_None;
        m_DisconnectReason = DisconnectReason_None;
        NN_RESULT_SUCCESS;
    }

    State Core::GetState() const NN_NOEXCEPT
    {
        return m_State;
    }

    Result Core::Scan(
        NetworkInfo* pOutScanResultArray,
        int* pOutCount,
        int bufferCount,
        const ScanFilter& filter,
        int channel) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_SessionMutex);
        NN_LDN_REQUIRES_NOT_EQUAL(m_State, State_None);
        NN_LDN_REQUIRES_NOT_NULL(pOutScanResultArray);
        NN_LDN_REQUIRES_NOT_NULL(pOutCount);
        NN_LDN_REQUIRES(0 < bufferCount);
        NN_LDN_REQUIRES((filter.flag & ScanFilterFlag_NetworkType) == 0 ||
                        (filter.networkType & ~(NetworkType_General | NetworkType_Ldn)) == 0);
        NN_LDN_REQUIRES((filter.flag & ScanFilterFlag_Ssid) == 0 || IsValidSsid(filter.ssid));
        NN_LDN_REQUIRES((filter.flag & ScanFilterFlag_Bssid) == 0 ||
                         filter.bssid != ZeroMacAddress);
        NN_LDN_REQUIRES((filter.flag & ScanFilterFlag_Bssid) == 0 ||
                         filter.bssid != BroadcastMacAddress);
        NN_LDN_REQUIRES((filter.flag & ~ScanFilterFlag_All) == 0);

        // 期待される状態でない場合には失敗します。
        if (m_State != State_AccessPointCreated &&
            m_State != State_Station && m_State != State_StationConnected)
        {
            return ResultInvalidState();
        }

        // スキャンを実行します。
        NN_RESULT_DO(m_AdvertiseScanner.Scan(
            pOutScanResultArray, pOutCount,
            std::min(bufferCount, ScanResultCountMax), filter, channel));
        NN_SDK_ASSERT_MINMAX(*pOutCount, 0, nn::ldn::ScanResultCountMax);

        // LDN の AP はステルス SSID になっていますが、ここで SessionID から復元します。
        for (int i = 0; i < *pOutCount; ++i)
        {
            NetworkInfo& network = pOutScanResultArray[i];
            if (network.common.networkType == NetworkType_Ldn)
            {
                network.common.ssid = CreateSsidFromNetworkId(network.networkId);
            }
        }
        NN_RESULT_SUCCESS;
    }

    Result Core::GetNetworkInfo(
        NetworkInfo* pOutNetwork, NodeLatestUpdate* outUpdates, int bufferCount) NN_NOEXCEPT
    {
        NN_LDN_REQUIRES_NOT_NULL(pOutNetwork);
        NN_LDN_REQUIRES_NOT_NULL(outUpdates);
        NN_LDN_REQUIRES(NodeCountMax <= bufferCount);
        NN_RESULT_DO(GetNetworkInfo(pOutNetwork));
        NN_RESULT_DO(m_SessionManager.GetNodeLatestUpdate(outUpdates, bufferCount));
        NN_RESULT_SUCCESS;
    }

    Result Core::GetNetworkInfo(NetworkInfo* pOutNetwork) const NN_NOEXCEPT
    {
        NN_LDN_REQUIRES_NOT_NULL(pOutNetwork);

        // 全体を 0 で初期化しておきます。
        std::memset(pOutNetwork, 0, sizeof(NetworkInfo));
        auto& common = pOutNetwork->common;

        // L2 のネットワーク情報を取得します。
        NetworkProfile network;
        int linkLevel;
        {
            std::lock_guard<nn::os::Mutex> lock(m_NetworkInterfaceMutex);
            if (m_pNetworkInterface == nullptr)
            {
                return ResultInvalidState();
            }
            NN_RESULT_DO(m_pNetworkInterface->GetNetworkProfile(&network));
            linkLevel = m_pNetworkInterface->GetLinkLevel();
        }
        common.bssid = network.bssid;
        common.ssid = network.ssid;
        common.channel = network.channel;
        common.linkLevel = static_cast<int8_t>(linkLevel);
        common.networkType = NetworkType_Ldn;

        // セッション情報を取得します。
        NN_RESULT_DO(m_SessionManager.GetNetworkInfo(&pOutNetwork->networkId, &pOutNetwork->ldn));
        NN_RESULT_SUCCESS;
    }

    Result Core::GetIpv4Address(
        Ipv4Address* pOutAddress, SubnetMask* pOutMask) const NN_NOEXCEPT
    {
        NN_LDN_REQUIRES_NOT_NULL(pOutAddress);
        NN_LDN_REQUIRES_NOT_NULL(pOutMask);
        NN_RESULT_DO(m_SessionManager.GetIpv4Address(pOutAddress, pOutMask));
        NN_RESULT_SUCCESS;
    }

    Result Core::GetSecurityParameter(SecurityParameter* pOutParam) const NN_NOEXCEPT
    {
        NN_LDN_REQUIRES_NOT_NULL(pOutParam);

        // 全体を 0 で初期化しておきます。
        auto& param = *pOutParam;
        std::memset(&param, 0, sizeof(SecurityParameter));

        // セッション情報を取得します。
        nn::ldn::NetworkId networkId;
        nn::ldn::LdnNetworkInfo ldn;
        NN_RESULT_DO(m_SessionManager.GetNetworkInfo(&networkId, &ldn));

        // セッション情報をセキュリティ・パラメータに変換します。
        std::memcpy(param._serverRandom, ldn.serverRandom, RandomSize);
        param._sessionId = networkId.sessionId;
        NN_RESULT_SUCCESS;
    }

    Result Core::GetNetworkConfig(NetworkConfig* pOutNetworkConfig) const NN_NOEXCEPT
    {
        NN_LDN_REQUIRES_NOT_NULL(pOutNetworkConfig);

        // 全体を 0 で初期化しておきます。
        auto& config = *pOutNetworkConfig;
        std::memset(&config, 0, sizeof(NetworkConfig));

        // ネットワーク情報を取得します。
        nn::ldn::NetworkInfo network;
        NN_RESULT_DO(GetNetworkInfo(&network));

        // NetworkConfig に変換します。
        config.intentId = network.networkId.intentId;
        config.channel = network.common.channel;
        config.nodeCountMax = network.ldn.nodeCountMax;
        config.localCommunicationVersion = network.ldn.nodes[0].localCommunicationVersion;
        NN_RESULT_SUCCESS;
    }

    Result Core::OpenAccessPoint(nn::os::SystemEventType* pStateChangeEvent) NN_NOEXCEPT
    {
        // この API を実行できる状態であることを検証します。
        NN_LDN_REQUIRES_NOT_EQUAL(m_State, State_None);
        if (m_State != State_Initialized)
        {
            return ResultInvalidState();
        }
        NN_LDN_REQUIRES_NOT_NULL(pStateChangeEvent);

        // L2 をステーションモードでオープンします。
        NN_RESULT_DO(m_pNetworkInterface->OpenAccessPoint());

        // アクセスポイント・ステーション共通のオープン処理です。
        OpenCommon(AccessPointMonitor, pStateChangeEvent);

        // アドバータイズを配信する準備です。
        m_AdvertiseDistributor.Initialize(
            m_Buffer->advertiseDistribution, sizeof(m_Buffer->advertiseDistribution),
            m_pNetworkInterface);

        m_State = State_AccessPoint;
        NN_RESULT_SUCCESS;
    }

    Result Core::CloseAccessPoint(bool isTriggeredBySystem) NN_NOEXCEPT
    {
        // この API を実行できる状態であることを検証します。
        NN_LDN_REQUIRES_NOT_EQUAL(m_State, State_None);
        if (m_State != State_AccessPoint && m_State != State_AccessPointCreated)
        {
            return ResultInvalidState();
        }

        // ネットワークが存在する場合には破棄します。
        if (m_State == State_AccessPointCreated)
        {
            DestroyNetwork(isTriggeredBySystem);
        }

        // アドバータイズの配信を停止します。
        m_AdvertiseDistributor.Finalize();

        // アクセスポイント・ステーション共通のクローズ処理です。
        CloseCommon();

        // L2 のステーションモードを終了します。
        Result result = m_pNetworkInterface->CloseAccessPoint();
        m_State = State_Initialized;
        return result;
    }

    Result Core::CreateNetwork(
        const NetworkConfig& network,
        const SecurityConfig& securityConfig,
        const SecurityParameter& securityParam,
        const UserConfig& user,
        int addressEntryCount,
        const AddressEntry* addressEntries) NN_NOEXCEPT
    {
        NN_LDN_REQUIRES_MINMAX(network.nodeCountMax, 1, NodeCountMax);
        NN_LDN_REQUIRES_MINMAX(network.localCommunicationVersion,
            LocalCommunicationVersionMin, LocalCommunicationVersionMax);
        NN_LDN_REQUIRES_INCLUDE(
            static_cast<SecurityMode>(securityConfig.securityMode),
            SecurityMode_Debug, SecurityMode_Product, SecurityMode_SystemDebug);
        NN_LDN_REQUIRES_MINMAX(
            securityConfig.passphraseSize,
            PassphraseSizeMin, PassphraseSizeMax);
        NN_LDN_REQUIRES_MINMAX(addressEntryCount, 0, AddressEntryCountMax);
        std::lock_guard<nn::os::Mutex> lock(m_SessionMutex);
        Result result;

        // この API を実行できる状態であることを検証します。
        NN_LDN_REQUIRES_NOT_EQUAL(m_State, State_None);
        if (m_State != State_AccessPoint)
        {
            return ResultInvalidState();
        }

        // ネットワーク識別子から SSID を生成します。
        NetworkId networkId = { network.intentId, securityParam._sessionId };
        Ssid ssid = CreateSsidFromNetworkId(networkId);

        // ステーション情報を栗化します。
        std::memset(m_Stations, 0, sizeof(m_Stations));

        // 暗号化に使用する鍵を生成します。
        Bit8 key[16];
        size_t keySize;
        m_SessionManager.SetSecurity(securityConfig, securityParam._serverRandom);
        m_SessionManager.CreateDataKey(key, &keySize, sizeof(key));

        // L2 ネットワークを生成します。
        NN_RESULT_DO(m_pNetworkInterface->CreateNetwork(
            ssid, network.channel, network.nodeCountMax, key, keySize));

        // セッションを構築します。
        m_SessionManager.SetUser(user);
        m_SessionManager.SetLocalCommunicationVersion(network.localCommunicationVersion);
        m_SessionManager.CreateSession(
            networkId, network.nodeCountMax, addressEntries, addressEntryCount);

        // フレームの送受信を準備します。
        m_Receiver.Initialize(
            m_pNetworkInterface, m_Buffer->receive, sizeof(m_Buffer->receive),
            sizeof(NintendoEthernetFrame), ProtocolCountMax);
        m_Sender.Initialize(
            m_pNetworkInterface, m_Buffer->send, sizeof(m_Buffer->send),
            sizeof(NintendoEthernetFrame), SendQueueCapacity);

        // Advertise の配信を開始します。
        LdnNetworkInfo ldn;
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_SessionManager.GetNetworkInfo(&networkId, &ldn));
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_AdvertiseDistributor.StartDistribution(
            CurrentVersion, networkId,
            static_cast<SecurityMode>(securityConfig.securityMode), ldn));

        // 認証サーバを起動します。
        m_AuthenticationServer.BeginServer(
            networkId, securityParam._serverRandom, &m_Receiver, &m_Sender);

        // 認証サーバが配信するデータを生成します。
        AuthenticationServerDataBuilder builder;
        builder.SetLdnNetworkInfo(ldn);
        m_AuthenticationServer.SetData(&builder.GetData(), sizeof(AuthenticationServerData));

        // 生成されたネットワークの情報を取得します。
        NetworkProfile profile;
        result = m_pNetworkInterface->GetNetworkProfile(&profile);
        if (result.IsSuccess())
        {
            NN_LDN_LOG_INFO("created a network on channel %d with SSID=%s\n",
                profile.channel, profile.ssid.raw);
        }
        else
        {
            NN_LDN_LOG_ERROR("created a network but failed to get channel\n");
        }

        // L2 ネットワークの構築に成功しました。
        m_State = State_AccessPointCreated;
        return result;
    }

    Result Core::DestroyNetwork(bool isTriggeredBySystem) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_SessionMutex);

        // この API を実行できる状態であることを検証します。
        NN_LDN_REQUIRES_NOT_EQUAL(m_State, State_None);
        if (m_State != State_AccessPointCreated)
        {
            return ResultInvalidState();
        }

        // 認証サーバを停止します。
        m_AuthenticationServer.EndServer();

        // Advertise の配信を停止します。
        m_AdvertiseDistributor.StopDistribution();

        // 切断理由を通知します。
        bool sentRequest = false;
        auto reason = isTriggeredBySystem ? DisconnectReason_DestroyedBySystem :
                                            DisconnectReason_DestroyedByUser;
        for (int aid = 1; aid < NodeCountMax; ++aid)
        {
            sentRequest |= SendRejectRequest(aid, reason).IsSuccess();
        }
        if (sentRequest)
        {
            nn::os::SleepThread(RejectRequestWaitTime);
        }

        // フレームの送受信を中止します。
        m_Sender.Finalize();
        m_Receiver.Finalize();

        // セッションを破棄します。
        m_SessionManager.DestroySession();

        // L2 ネットワークを破棄します。
        Result result = m_pNetworkInterface->DestroyNetwork();
        NN_LDN_LOG_INFO("destroyed a network\n");
        m_State = State_AccessPoint;
        return result;
    }

    Result Core::SetAdvertiseData(const void* data, size_t dataSize) NN_NOEXCEPT
    {
        NN_LDN_REQUIRES(data != nullptr || dataSize == 0);
        NN_LDN_REQUIRES_MINMAX(dataSize, 0U, AdvertiseDataSizeMax);

        // この API を実行できる状態であることを検証します。
        NN_LDN_REQUIRES_NOT_EQUAL(m_State, State_None);
        if (m_State != State_AccessPoint && m_State != State_AccessPointCreated)
        {
            return ResultInvalidState();
        }

        // Advertise で配信するデータを更新します。
        m_SessionManager.SetAdvertiseData(data, dataSize);
        UpdateAdvertise();
        NN_RESULT_SUCCESS;
    }

    Result Core::SetStationAcceptPolicy(AcceptPolicy policy) NN_NOEXCEPT
    {
        NN_LDN_REQUIRES_INCLUDE(policy,
            AcceptPolicy_AlwaysAccept,
            AcceptPolicy_AlwaysReject,
            AcceptPolicy_BlackList,
            AcceptPolicy_WhiteList);

        // この API を実行できる状態であることを検証します。
        NN_LDN_REQUIRES_NOT_EQUAL(m_State, State_None);
        if (m_State != State_AccessPoint && m_State != State_AccessPointCreated)
        {
            return ResultInvalidState();
        }

        // Advertise で配信するデータを更新します。
        m_SessionManager.SetStationAcceptPolicy(policy);
        UpdateAdvertise();
        NN_RESULT_SUCCESS;
    }

    Result Core::Reject(Ipv4Address ipv4Address) NN_NOEXCEPT
    {
        NN_LDN_REQUIRES_NOT_EQUAL(ipv4Address, BroadcastIpv4Address);
        NN_LDN_REQUIRES_NOT_EQUAL(ipv4Address, ZeroIpv4Address);
        std::lock_guard<nn::os::Mutex> lock(m_SessionMutex);

        // この API を実行できる状態であることを検証します。
        NN_LDN_REQUIRES_NOT_EQUAL(m_State, State_None);
        if (m_State != State_AccessPointCreated)
        {
            return ResultInvalidState();
        }

        // 対象のステーションを検索します。
        int aid = m_SessionManager.Find(ipv4Address);
        if (aid <= 0)
        {
            return ResultNodeNotFound();
        }
        NN_SDK_ASSERT_MINMAX(aid, 1, StationCountMax);

        // ネットワークから離脱させます。
        auto& station = m_Stations[aid - 1];
        m_AuthenticationServer.Unregister(station.authenticationId);
        m_SessionManager.RemoveUser(aid);

        // 切断理由を通知して切断します。
        Result result = SendRejectRequest(aid, DisconnectReason_Rejected);
        if (result.IsSuccess())
        {
            nn::os::SleepThread(RejectRequestWaitTime);
        }
        m_pNetworkInterface->Reject(station.macAddress);

        // アドバータイズは OnAccessPointStateChanged() で更新されます。
        // UpdateAdvertise();

        // 切断処理に成功しました。
        NN_LDN_STRINGIZE_HELPER(helper, StringizedMacAddressLength);
        NN_LDN_LOG_DEBUG("node %d disconnected (MAC=%s)\n",
            aid, helper.ToString(station.macAddress));
        NN_RESULT_SUCCESS;
    }

    Result Core::AddAcceptFilterEntry(MacAddress macAddress) NN_NOEXCEPT
    {
        // この API を実行できる状態であることを検証します。
        NN_LDN_REQUIRES_NOT_EQUAL(m_State, State_None);
        if (m_State != State_AccessPoint && m_State != State_AccessPointCreated)
        {
            return ResultInvalidState();
        }

        // フィルタにエントリを登録します。
        NN_RESULT_DO(m_SessionManager.AddAcceptFilterEntry(macAddress));
        NN_RESULT_SUCCESS;
    }

    Result Core::ClearAcceptFilter() NN_NOEXCEPT
    {
        // この API を実行できる状態であることを検証します。
        NN_LDN_REQUIRES_NOT_EQUAL(m_State, State_None);
        if (m_State != State_AccessPoint && m_State != State_AccessPointCreated)
        {
            return ResultInvalidState();
        }

        // フィルタからエントリを削除します。
        NN_RESULT_DO(m_SessionManager.ClearAcceptFilter());
        NN_RESULT_SUCCESS;
    }

    Result Core::OpenStation(nn::os::SystemEventType* pStateChangeEvent) NN_NOEXCEPT
    {
        // この API を実行できる状態であることを検証します。
        NN_LDN_REQUIRES_NOT_EQUAL(m_State, State_None);
        if (m_State != State_Initialized)
        {
            return ResultInvalidState();
        }
        NN_LDN_REQUIRES_NOT_NULL(pStateChangeEvent);

        // L2 をステーションモードでオープンします。
        NN_RESULT_DO(m_pNetworkInterface->OpenStation());

        // アドバータイズを監視する準備です。
        m_AdvertiseMonitor.Initialize(
            m_Buffer->advertiseMonitor, sizeof(m_Buffer->advertiseMonitor), m_pNetworkInterface);

        // アクセスポイント・ステーション共通のオープン処理です。
        OpenCommon(StationMonitor, pStateChangeEvent);

        m_State = State_Station;
        NN_RESULT_SUCCESS;
    }

    Result Core::CloseStation(bool isTriggeredBySystem) NN_NOEXCEPT
    {
        // この API を実行できる状態であることを検証します。
        NN_LDN_REQUIRES_NOT_EQUAL(m_State, State_None);
        if (m_State != State_Station && m_State != State_StationConnected)
        {
            return ResultInvalidState();
        }

        // ネットワークに接続している場合には切断します。
        if (m_State == State_StationConnected)
        {
            Disconnect(isTriggeredBySystem);
        }

        // アクセスポイント・ステーション共通のクローズ処理です。
        CloseCommon();

        // アドバータイズ関連のモジュールを解放します。
        m_AdvertiseMonitor.Finalize();

        // L2 のステーションモードを終了します。
        Result result = m_pNetworkInterface->CloseStation();
        m_State = State_Initialized;
        return result;
    }

    Result Core::Connect(
        const NetworkConfig& network,
        const SecurityConfig& securityConfig,
        const SecurityParameter& securityParam,
        const UserConfig& user,
        int version,
        ConnectOption option) NN_NOEXCEPT
    {
        Result result;
        auto securityMode = static_cast<SecurityMode>(securityConfig.securityMode);
        NN_LDN_REQUIRES_MINMAX(network.localCommunicationVersion,
            LocalCommunicationVersionMin, LocalCommunicationVersionMax);
        NN_LDN_REQUIRES_INCLUDE(securityMode,
            SecurityMode_Debug, SecurityMode_Product, SecurityMode_SystemDebug);
        NN_LDN_REQUIRES_MINMAX(securityConfig.passphraseSize,
            PassphraseSizeMin, PassphraseSizeMax);
        NN_LDN_REQUIRES_MINMAX(version,
            LocalCommunicationVersionMin, LocalCommunicationVersionMax);
        NN_LDN_REQUIRES_EQUAL(ConnectOption_None, option & ~ConnectOption_All);

        // この API を実行できる状態であることを検証します。
        std::lock_guard<nn::os::Mutex> lock(m_SessionMutex);
        NN_LDN_REQUIRES_NOT_EQUAL(m_State, State_None);
        if (m_State != State_Station)
        {
            return ResultInvalidState();
        }

        // ネットワーク識別子から SSID を生成します。
        NetworkId networkId = { network.intentId, securityParam._sessionId };
        Ssid ssid = CreateSsidFromNetworkId(networkId);

        // 暗号化に使用する鍵を生成します。
        Bit8 key[16];
        size_t keySize;
        m_SessionManager.SetSecurity(securityConfig, securityParam._serverRandom);
        m_SessionManager.CreateDataKey(key, &keySize, sizeof(key));

        // L2 ネットワークに接続します。
        INetworkInterface::ScopedConnection scopedConnection(
            m_pNetworkInterface, ssid, network.channel, network.nodeCountMax, key, keySize);
        NN_RESULT_DO(scopedConnection.GetResult());

        // 接続先のネットワークの情報を取得します。
        NetworkProfile profile;
        result = m_pNetworkInterface->GetNetworkProfile(&profile);
        if (result.IsFailure())
        {
            // 接続直後に切断されるのは認証を拒否されている場合です。
            return ResultConnectionRejected();
        }

        // フレームの送受信を準備します。
        FrameReceiver::ScopedInitializer scopedReceiveInitializer(
            &m_Receiver, m_pNetworkInterface, m_Buffer->receive, sizeof(m_Buffer->receive),
            sizeof(NintendoEthernetFrame), ProtocolCountMax);
        FrameSender::ScopedInitializer scopedSendInitializer(
            &m_Sender, m_pNetworkInterface, m_Buffer->send, sizeof(m_Buffer->send),
            sizeof(NintendoEthernetFrame), SendQueueCapacity);

        // 認証に必要な乱数を生成します。
        Bit8 clientRandom[RandomSize];
        FillSecureRandom(clientRandom, RandomSize);

        // 認証を実行します。
        IAuthenticationClient::ScopedClient scopedClient(
            &m_AuthenticationClient, clientRandom, &m_Receiver, &m_Sender);
        result = Authenticate(
            profile.bssid, networkId, securityParam._serverRandom, user, version);
        if (result.IsFailure())
        {
            return result;
        }

        // アドバータイズの監視を開始します。
        int aid;
        LdnNetworkInfo ldn;
        AdvertiseMonitor::ScopedMonitor scopedMonitor(
            &m_AdvertiseMonitor, &aid, &ldn, networkId, securityMode);
        NN_RESULT_DO(scopedMonitor.GetResult());

        // セッションに参加します。
        m_SessionManager.SetUser(user);
        m_SessionManager.SetLocalCommunicationVersion(version);
        result = m_SessionManager.JoinSession(networkId, ldn, aid);
        if (result.IsFailure())
        {
            return result;
        }

        // ネットワーク情報を保存します。
        result = m_SessionManager.UpdateNetworkInfo(ldn);
        if (result.IsFailure())
        {
            m_SessionManager.LeaveSession();
            return result;
        }

        // 切断通知の受信を開始します。
        m_DisconnectReason = DisconnectReason_None;
        m_RejectQueue.Initialize(
            m_Buffer->rejectReceiveBuffer, sizeof(m_Buffer->rejectReceiveBuffer),
            sizeof(RejectRequest), RejectReceiveQueueCapacity);
        m_Receiver.Register(RejectProtocolId, m_RejectEvent.GetBase(), &m_RejectQueue);

        // L2 ネットワークへの接続に成功しました。
        scopedConnection.Detach();
        scopedReceiveInitializer.Detach();
        scopedSendInitializer.Detach();
        scopedClient.Detach();
        scopedMonitor.Detach();
        NN_LDN_LOG_INFO("Connected to the network on channel %d with SSID=%s\n",
                        profile.channel, ssid.raw);
        m_State = State_StationConnected;
        NN_RESULT_SUCCESS;
    }

    Result Core::Disconnect(bool isTriggeredBySystem) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> stateLock(m_SessionMutex);
        auto reason = isTriggeredBySystem ? DisconnectReason_DisconnectedBySystem :
                                            DisconnectReason_DisconnectedByUser;
        return DisconnectImpl(reason);
    }

    Result Core::GetDisconnectReason(DisconnectReason* pOutReason) const NN_NOEXCEPT
    {
        NN_LDN_REQUIRES_NOT_NULL(pOutReason);
        *pOutReason = m_DisconnectReason;
        NN_RESULT_SUCCESS;
    }

    Result Core::SetOperationMode(OperationMode mode) NN_NOEXCEPT
    {
        NN_LDN_REQUIRES(mode == OperationMode_Stable || mode == OperationMode_HighSpeed);
        std::lock_guard<nn::os::Mutex> lock(m_SessionMutex);

        // この API を実行できる状態であることを検証します。
        NN_LDN_REQUIRES_NOT_EQUAL(m_State, State_None);
        if (m_State != State_Initialized)
        {
            return ResultInvalidState();
        }

        // L2 の動作モードを変更しておきます。
        m_pNetworkInterface->SetOperationMode(mode);
        NN_LDN_LOG_INFO("%s mode is enabled\n",
            mode == OperationMode_Stable ? "stable" : "high speed");
        NN_RESULT_SUCCESS;
    }

    Result Core::SetWirelessControllerRestriction(
        WirelessControllerRestriction restriction) NN_NOEXCEPT
    {
        NN_LDN_REQUIRES(restriction == WirelessControllerRestriction_Disabled ||
                        restriction == WirelessControllerRestriction_Enabled);
        std::lock_guard<nn::os::Mutex> lock(m_SessionMutex);

        // この API を実行できる状態であることを検証します。
        NN_LDN_REQUIRES_NOT_EQUAL(m_State, State_None);
        if (m_State != State_Initialized)
        {
            return ResultInvalidState();
        }

        // L2 の動作モードを変更しておきます。
        m_pNetworkInterface->SetWirelessControllerRestriction(restriction);
        NN_LDN_LOG_INFO("wireless controller restriction is %s\n",
            restriction == WirelessControllerRestriction_Enabled ? "enabled" : "disabled");
        NN_RESULT_SUCCESS;
    }

    void Core::AccessPointMonitor(void* pArg) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pArg);
        NN_LDN_LOG_DEBUG("AccessPointMonitorThread: started\n");
        auto& core = *static_cast<Core*>(pArg);
        auto& closeEvent = core.m_CloseEvent;
        auto& stateChangeEvent = core.m_pNetworkInterface->GetStateChangeEvent();
        auto& authEvent = core.m_AuthenticationServer.GetAuthenticationEvent();

        // このスレッドでは Close まで L2 の接続状態変化を監視します。
        nn::os::MultiWaitType       multiWait;
        nn::os::MultiWaitHolderType closeHolder;
        nn::os::MultiWaitHolderType stateChangeHolder;
        nn::os::MultiWaitHolderType authHolder;
        nn::os::InitializeMultiWait(&multiWait);
        nn::os::InitializeMultiWaitHolder(&closeHolder, closeEvent.GetBase());
        nn::os::InitializeMultiWaitHolder(&stateChangeHolder, stateChangeEvent.GetBase());
        nn::os::InitializeMultiWaitHolder(&authHolder, authEvent.GetBase());
        nn::os::LinkMultiWaitHolder(&multiWait, &closeHolder);
        nn::os::LinkMultiWaitHolder(&multiWait, &stateChangeHolder);
        nn::os::LinkMultiWaitHolder(&multiWait, &authHolder);

        // 接続状態の変化を監視します。
        nn::os::MultiWaitHolderType* pHolder;
        while ((pHolder = nn::os::WaitAny(&multiWait)) != &closeHolder)
        {
            if (pHolder == &stateChangeHolder)
            {
                stateChangeEvent.Clear();
                core.OnAccessPointStateChanged();
            }
            else if (pHolder == &authHolder)
            {
                authEvent.Clear();
                core.OnAuthenticationProcessed();
            }
        }

        // 終了処理です。
        nn::os::UnlinkAllMultiWaitHolder(&multiWait);
        nn::os::FinalizeMultiWaitHolder(&authHolder);
        nn::os::FinalizeMultiWaitHolder(&stateChangeHolder);
        nn::os::FinalizeMultiWaitHolder(&closeHolder);
        nn::os::FinalizeMultiWait(&multiWait);
        NN_LDN_LOG_DEBUG("AccessPointMonitorThread: finished\n");
    }

    void Core::StationMonitor(void* pArg) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pArg);
        NN_LDN_LOG_DEBUG("StationMonitorThread: started\n");
        auto& core = *static_cast<Core*>(pArg);
        auto& closeEvent = core.m_CloseEvent;
        auto& rejectEvent = core.m_RejectEvent;
        auto& advertiseEvent = core.m_AdvertiseMonitor.GetReceivedEvent();
        auto& stateChangeEvent = core.m_pNetworkInterface->GetStateChangeEvent();

        // このスレッドでは Close まで L2 の接続状態変化を監視します。
        nn::os::MultiWaitType       multiWait;
        nn::os::MultiWaitHolderType closeHolder;
        nn::os::MultiWaitHolderType rejectHolder;
        nn::os::MultiWaitHolderType advertiseHolder;
        nn::os::MultiWaitHolderType stateChangeHolder;
        nn::os::InitializeMultiWait(&multiWait);
        nn::os::InitializeMultiWaitHolder(&closeHolder, closeEvent.GetBase());
        nn::os::InitializeMultiWaitHolder(&rejectHolder, rejectEvent.GetBase());
        nn::os::InitializeMultiWaitHolder(&advertiseHolder, advertiseEvent.GetBase());
        nn::os::InitializeMultiWaitHolder(&stateChangeHolder, stateChangeEvent.GetBase());
        nn::os::LinkMultiWaitHolder(&multiWait, &closeHolder);
        nn::os::LinkMultiWaitHolder(&multiWait, &rejectHolder);
        nn::os::LinkMultiWaitHolder(&multiWait, &advertiseHolder);
        nn::os::LinkMultiWaitHolder(&multiWait, &stateChangeHolder);

        // 接続状態の変化を監視します。
        nn::os::MultiWaitHolderType* pHolder;
        while ((pHolder = nn::os::WaitAny(&multiWait)) != &closeHolder)
        {
            if (pHolder == &rejectHolder)
            {
                rejectEvent.Clear();
                core.OnRejectRequestReceived();
            }
            else if (pHolder == &advertiseHolder)
            {
                advertiseEvent.Clear();
                core.OnAdvertiseReceived();
            }
            else if (pHolder == &stateChangeHolder)
            {
                stateChangeEvent.Clear();
                core.OnStationStateChanged();
            }
        }

        // 終了処理です。
        nn::os::UnlinkAllMultiWaitHolder(&multiWait);
        nn::os::FinalizeMultiWaitHolder(&stateChangeHolder);
        nn::os::FinalizeMultiWaitHolder(&advertiseHolder);
        nn::os::FinalizeMultiWaitHolder(&rejectHolder);
        nn::os::FinalizeMultiWaitHolder(&closeHolder);
        nn::os::FinalizeMultiWait(&multiWait);
        NN_LDN_LOG_DEBUG("StationMonitorThread: finished\n");
    }

    void Core::StartNetworkMonitoring(nn::os::ThreadFunction func) NN_NOEXCEPT
    {
        m_CloseEvent.Clear();
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
            &m_NetworkMonitorThread, func, this,
            m_Buffer->networkMonitorThreadStack, sizeof(m_Buffer->networkMonitorThreadStack),
            NN_SYSTEM_THREAD_PRIORITY(ldn, NetworkMonitor)));
        nn::os::SetThreadNamePointer(
            &m_NetworkMonitorThread, NN_SYSTEM_THREAD_NAME(ldn, NetworkMonitor));
        nn::os::StartThread(&m_NetworkMonitorThread);
    }

    void Core::StopNetworkMonitoring() NN_NOEXCEPT
    {
        m_CloseEvent.Signal();
        nn::os::WaitThread(&m_NetworkMonitorThread);
        nn::os::DestroyThread(&m_NetworkMonitorThread);
    }

    void Core::OnAccessPointStateChanged() NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> sessionLock(m_SessionMutex);

        // 操作を実行できる状態であることを確認します。
        if (m_pNetworkInterface->GetState() != L2State_AccessPointCreated)
        {
            return;
        }

        // ステーションの情報を取得します。
        L2StationInfo stations[StationCountMax];
        int stationCount;
        Result result = m_pNetworkInterface->GetStations(stations, &stationCount, StationCountMax);
        if (result.IsFailure())
        {
            return;
        }

        // ステーションの接続状態変更を処理します。
        for (int i = 0; i < stationCount; ++i)
        {
            auto& ldn = m_Stations[i];
            const auto& l2 = stations[i];
            NN_SDK_ASSERT_EQUAL(l2.aid, i + 1);
            if (ldn.isConnected)
            {
                if (l2.status == L2StationState_Disconnected)
                {
                    // ステーションの切断です。
                    OnStationDisconnected(l2.aid);
                }
                else if (ldn.connectedAt != l2.connectedAt || ldn.macAddress != l2.macAddress)
                {
                    // ステーションの切断処理と新規接続処理が同時に発生しています。
                    OnStationDisconnected(l2.aid);
                    OnStationConnected(l2);
                }
            }
            else
            {
                if (l2.status == L2StationState_Connected)
                {
                    // ステーションの新規接続です。
                    OnStationConnected(l2);
                }
            }
        }

        // アドバータイズを最新の状態に更新します。
        UpdateAdvertise();
    }

    void Core::OnStationConnected(const L2StationInfo& l2) NN_NOEXCEPT
    {
        int aid = l2.aid;
        NN_SDK_ASSERT_MINMAX(aid, 1, StationCountMax);
        NN_SDK_ASSERT_EQUAL(l2.status, L2StationState_Connected);
        auto& ldn = m_Stations[aid - 1];
        NN_SDK_ASSERT(!ldn.isConnected);
        NN_LDN_STRINGIZE_HELPER(helper, StringizedMacAddressLength);
        NN_LDN_LOG_DEBUG("node %d connected (MAC=%s)\n", aid, helper.ToString(l2.macAddress));
        Result result = m_SessionManager.IsAcceptable(l2.macAddress);
        ldn.authenticationId = m_AuthenticationServer.Register(
            l2.macAddress, result.IsSuccess());
        ldn.isConnected = true;
        ldn.macAddress = l2.macAddress;
        ldn.connectedAt = l2.connectedAt;
    }

    void Core::OnStationDisconnected(int aid) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_MINMAX(aid, 1, StationCountMax);
        auto& ldn = m_Stations[aid - 1];
        NN_SDK_ASSERT(ldn.isConnected);
        m_AuthenticationServer.Unregister(ldn.authenticationId);
        m_SessionManager.RemoveUser(aid);
        m_pNetworkInterface->Reject(ldn.macAddress);
        NN_LDN_STRINGIZE_HELPER(helper, StringizedMacAddressLength);
        NN_LDN_LOG_DEBUG("node %d disconnected (MAC=%s)\n", aid, helper.ToString(ldn.macAddress));
        std::memset(&ldn, 0, sizeof(ldn));
    }

    void Core::OnAuthenticationProcessed() NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> sessionLock(m_SessionMutex);

        // 認証処理を実行できる状態であることを確認します。
        if (m_pNetworkInterface->GetState() != L2State_AccessPointCreated)
        {
            return;
        }

        // 最新の認証結果を取得します。
        AuthenticationHistroy history;
        if (!m_AuthenticationServer.GetAuthenticationHistory(&history))
        {
            return;
        }

        // 認証結果に含まれるノードを検索します。
        int i;
        for (i = 0; i < StationCountMax; ++i)
        {
            const auto& station = m_Stations[i];
            if (station.isConnected && station.authenticationId == history.id)
            {
                break;
            }
        }
        if (StationCountMax <= i)
        {
            // 切断済のノードに対する認証結果です。
            return;
        }

        // 認証の結果を確認します。
        const auto& station = m_Stations[i];
        const int aid = i + 1;
        if (history.authenticationResult == AuthenticationResult_Success)
        {
            AuthenticationClientDataParser parser(history.receivedData, history.receivedSize);
            if (parser.IsValid())
            {
                Result result = m_SessionManager.AddUser(
                    aid, station.macAddress, parser.GetLocalCommunicationVersion(),
                    history.clientRandom, parser.GetUserConfig());
                if (result.IsSuccess())
                {
                    NN_ABORT_UNLESS_RESULT_SUCCESS(UpdateAdvertise());
                }
                else
                {
                    NN_LDN_LOG_WARN("failed to add user\n");
                    OnStationDisconnected(aid);
                }
            }
            else
            {
                NN_LDN_LOG_WARN("failed to parse authentication data\n");
                OnStationDisconnected(aid);
            }
        }
        else if (history.authenticationResult == AuthenticationResult_Timeout)
        {
            OnStationDisconnected(aid);
        }
        else
        {
            // 再試行によって成功の可能性があるため、切断しません。
        }
    }

    void Core::OnStationStateChanged() NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> sessionLock(m_SessionMutex);

        // 操作を実行できる状態であることを確認します。
        if (m_pNetworkInterface->GetState() != L2State_StationConnected)
        {
            return;
        }

        // ステーションの接続状態を取得します。
        L2StationInfo station;
        if(m_pNetworkInterface->GetConnectionStatus(&station).IsFailure())
        {
            NN_LDN_LOG_WARN("failed to get connection status\n");
            return;
        }

        // L2 の状態に応じて LDN の状態も更新します。
        if (station.status == L2StationState_Connected)
        {
            // 接続完了は IPC スレッドでハンドリングするため特に必要な処理はありません。
        }
        else
        {
            auto reason = station.disconnectReason == L2DisconnectReason_BeaconLost ?
              DisconnectReason_SignalLost : DisconnectReason_Unknown;
            DisconnectImpl(reason);
        }
    }

    void Core::OnRejectRequestReceived() NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> sessionLock(m_SessionMutex);

        // 操作を実行できる状態であることを確認します。
        if (m_pNetworkInterface->GetState() != L2State_StationConnected)
        {
            return;
        }

        // 切断要求フレームを解析します。
        DisconnectReason reason = DisconnectReason_None;
        while (!m_RejectQueue.IsEmpty() && reason == DisconnectReason_None)
        {
            // 受信データを取得します。
            size_t receivedSize;
            const auto& frame = *static_cast<const RejectRequest*>(
                m_RejectQueue.GetFront(&receivedSize));

            // 切断要求の正当性を確認します。
            if (receivedSize == sizeof(RejectRequest))
            {
                reason = static_cast<DisconnectReason>(frame.reason);
            }
            else
            {
                NN_LDN_LOG_WARN("invalid reject request received (%u Bytes)\n", receivedSize);
            }
            m_RejectQueue.Dequeue();
        }

        // 切断理由の取得に成功した場合は切断します。
        if (reason != DisconnectReason_None)
        {
            DisconnectImpl(reason);
        }
    }

    void Core::OnAdvertiseReceived() NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> sessionLock(m_SessionMutex);

        // 操作を実行できる状態であることを確認します。
        if (m_pNetworkInterface->GetState() != L2State_StationConnected)
        {
            return;
        }

        // アドバータイズを取得します。
        LdnNetworkInfo ldn;
        if (!m_AdvertiseMonitor.GetAdvertise(&ldn))
        {
            // 新規のアドバータイズを受信していない場合には何もしません。
            return;
        }

        // アドバータイズで受信したセッションの情報を保存します。
        Result result = m_SessionManager.UpdateNetworkInfo(ldn);
        if (result.IsFailure())
        {
            // アドバータイズにこのステーションが含まれていません。
            // アクセスポイントから既に切断されたとみなします。
            DisconnectImpl(DisconnectReason_Unknown);
        }
        NN_LDN_TRACE;
    }

    Result Core::UpdateAdvertise() NN_NOEXCEPT
    {
        LdnNetworkInfo ldn;
        NetworkId networkId;
        NN_RESULT_DO(m_SessionManager.GetNetworkInfo(&networkId, &ldn));
        NN_RESULT_DO(m_AdvertiseDistributor.Update(ldn));
        NN_RESULT_SUCCESS;
    }

    void Core::OpenCommon(
        nn::os::ThreadFunction func, nn::os::SystemEventType* pStateChangeEvent) NN_NOEXCEPT
    {
        // 切断理由を初期化しておきます。
        m_DisconnectReason = DisconnectReason_None;

        // SessionManager を初期化します。
        NetworkInterfaceProfile profile;
        m_pNetworkInterface->GetNetworkInterfaceProfile(&profile);
        m_SessionManager.Initialize(profile, pStateChangeEvent);

        // 監視スレッドを起動します。
        StartNetworkMonitoring(func);
    }

    void Core::CloseCommon() NN_NOEXCEPT
    {
        // 監視スレッドを停止します。
        StopNetworkMonitoring();

        // SessionManager を破棄します。
        m_SessionManager.Finalize();
    }

    Result Core::DisconnectImpl(DisconnectReason reason) NN_NOEXCEPT
    {
        // この API を実行できる状態であることを検証します。
        NN_LDN_REQUIRES_NOT_EQUAL(m_State, State_None);
        if (m_State != State_StationConnected)
        {
            return ResultInvalidState();
        }

        // 先に状態を遷移させておきます。
        m_DisconnectReason = reason;
        m_State = State_Station;

        // 切断通知の受信を停止します。
        m_Receiver.Unregister(RejectProtocolId);
        m_RejectQueue.Finalize();

        // セッションから離脱します。
        m_SessionManager.LeaveSession();

        // アドバータイズの監視を停止します。
        m_AdvertiseMonitor.StopMonitoring();

        // 認証クライアントを停止します。
        m_AuthenticationClient.EndClient();

        // データフレームの送受信を停止します。
        m_Sender.Finalize();
        m_Receiver.Finalize();
        m_RejectQueue.Clear();

        // L2 ネットワークから離脱し、切断理由を記憶します。
        Result result = m_pNetworkInterface->Disconnect();
        NN_LDN_LOG_INFO("disconnected (reason: %d)\n", m_DisconnectReason);
        return result;
    }

    Result Core::Authenticate(
        MacAddress bssid, const NetworkId& networkId, const Bit8 (&serverRandom)[RandomSize],
        const UserConfig& user, int localCommunicationVersion) NN_NOEXCEPT
    {
        // 認証クライアントを起動し、認証クライアントが配信するデータを生成します。
        AuthenticationClientDataBuilder builder;
        builder.SetUserConfig(user);
        builder.SetLocalCommunicationVersion(localCommunicationVersion);
        m_AuthenticationClient.SetData(&builder.GetData(), sizeof(AuthenticationClientData));

        // アクセスポイントに認証を要求します。
        AuthenticationServerData serverData;
        size_t serverDataSize;
        auto authenticationResult = m_AuthenticationClient.Authenticate(
            &serverData, &serverDataSize, sizeof(serverData),
            bssid, networkId, serverRandom);
        if (authenticationResult == AuthenticationResult_Success)
        {
            return ResultSuccess();
        }
        else if (authenticationResult == AuthenticationResult_Timeout)
        {
            return ResultConnectionTimeout();
        }
        else
        {
            return ResultConnectionRejected();
        }
    }

    Result Core::SendRejectRequest(int aid, DisconnectReason reason) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_MINMAX(aid, 1, StationCountMax);
        auto& ldn = m_Stations[aid - 1];

        // ステーションが接続されていない場合には失敗します。
        if (!ldn.isConnected)
        {
            return ResultNodeNotFound();
        }

        // 切断要求フレームを生成します。
        RejectRequest request = { };
        request.reason = static_cast<int8_t>(reason);

        // ステーションに切断理由を送信します。
        size_t size = sizeof(RejectRequest) - NintendoEthernetFrameSizeMin;
        return m_Sender.Send(&request.reason, size, ldn.macAddress, RejectProtocolId);
    }

}}} // namespace nn::ldn::detail
