﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/ldn/detail/Debug/ldn_Assert.h>
#include <nn/ldn/detail/Utility/ldn_Digest.h>

#if defined(NN_LDN_BUILD_CONFIG_SPL_ENABLED)
#include <nn/spl/spl_Api.h>
#else
#include <nn/crypto/crypto_Aes128CtrEncryptor.h>
#endif

namespace nn { namespace ldn { namespace detail { namespace
{
    bool g_HasCache;
    bool g_IsProductionEnvironment;

    #if defined(NN_LDN_BUILD_CONFIG_SPL_ENABLED)
    const Bit8 KekSourceMask[] =
    {
        0xDA, 0xD8, 0xFC, 0x8E, 0x2D, 0x04, 0xAD, 0x06,
        0x72, 0xAF, 0x4B, 0x5B, 0x48, 0x53, 0x25, 0xA1
    };

    void InitializeImpl() NN_NOEXCEPT
    {
        nn::spl::InitializeForCrypto();
    }

    bool IsProductionEnvironmentImpl() NN_NOEXCEPT
    {
        return !nn::spl::IsDevelopment();
    }

    void GenerateAesKeyImpl(
        Bit8 (&outKey)[16], const void* keySource, size_t keySourceSize,
        const void* kekSource, size_t kekSourceSize) NN_NOEXCEPT
    {
        // Key Encryption Key を生成します。
        nn::spl::AccessKey accessKey;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::spl::GenerateAesKek(
            &accessKey, kekSource, kekSourceSize, 0, 0));

        // AES 鍵を生成します。
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::spl::GenerateAesKey(
            outKey, sizeof(outKey), accessKey, keySource, keySourceSize));
    }
    #else
    const Bit8 KekSourceMask[] =
    {
        0x1F, 0x27, 0xEF, 0x68, 0x99, 0x2C, 0xCB, 0x5C,
        0x3A, 0xBC, 0x35, 0xDE, 0x64, 0x9D, 0x1E, 0x0A
    };

    void InitializeImpl() NN_NOEXCEPT
    {
    }

    bool IsProductionEnvironmentImpl() NN_NOEXCEPT
    {
        return false;
    }

    void GenerateAesKeyImpl(
        Bit8 (&outKey)[16], const void* keySource, size_t keySourceSize,
        const void* kekSource, size_t kekSourceSize) NN_NOEXCEPT
    {
        Bit8 iv[16] = { };
        size_t keySize= nn::crypto::EncryptAes128Ctr(
            outKey, sizeof(outKey), kekSource, kekSourceSize, iv, sizeof(iv),
            keySource, keySourceSize);
        NN_SDK_ASSERT_EQUAL(keySize, sizeof(outKey));
        NN_UNUSED(keySize);
    }
    #endif

    bool g_IsInitialized;
    void Initialize() NN_NOEXCEPT
    {
        if (!g_IsInitialized)
        {
            InitializeImpl();
            g_IsInitialized = true;
        }
    }

}}}} // namespace nn::ldn::detail::<unnamed>

namespace nn { namespace ldn { namespace detail
{
    bool IsProductionEnvironment() NN_NOEXCEPT
    {
        if (!g_HasCache)
        {
            Initialize();
            g_IsProductionEnvironment = IsProductionEnvironmentImpl();
            g_HasCache = true;
        }
        return g_IsProductionEnvironment;
    }

    void GenerateAesKey(
        Bit8 (&outKey)[16], const void* keySource, size_t keySourceSize,
        const void* kekSource, size_t kekSourceSize) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(outKey);
        NN_SDK_ASSERT_NOT_NULL(kekSource);
        NN_SDK_ASSERT_EQUAL(kekSourceSize, 16U);

        // モジュールを初期化します。
        Initialize();

        // KEK ソースを変換します。
        Bit8 maskedKekSource[16];
        for (size_t i = 0; i < kekSourceSize; ++i)
        {
            maskedKekSource[i] = static_cast<const Bit8*>(kekSource)[i] ^ KekSourceMask[i];
        }

        // 鍵ソースの SHA-256 を生成します。
        Bit8 digest[32];
        size_t digestSize;
        CalculateSha256(digest, &digestSize, sizeof(digest), keySource, keySourceSize);
        NN_SDK_ASSERT_EQUAL(digestSize, sizeof(digest));
        NN_UNUSED(digestSize);

        // AES 鍵を生成します。
        GenerateAesKeyImpl(outKey, digest, 16, maskedKekSource, kekSourceSize);
    }

}}} // namespace nn::ldn::detail
