﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ldn/ldn_Result.h>
#include <nn/ldn/ldn_Types.h>
#include <nn/ldn/detail/Debug/ldn_Assert.h>
#include <nn/ldn/detail/NetworkInterface/ldn_NetworkInterfaceStub.h>
#include <nn/ldn/detail/NetworkInterface/ldn_NintendoEthernet.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace ldn { namespace detail { namespace
{
    // サポートする無線チャンネルです。
    const int SupportedChannels[] = { 1, 6, 11, 36, 40, 44, 48 };
    const int SupportedChannelCount = sizeof(SupportedChannels) / sizeof(int);
    const int AutoSelectableChannels[] = { 1, 6, 11 };
    const int AutoSelectableChannelCount = sizeof(AutoSelectableChannels) / sizeof(int);
    NN_STATIC_ASSERT(SupportedChannelCount <= SupportedChannelCountMax);
    NN_STATIC_ASSERT(AutoSelectableChannelCount <= AutoSelectableChannelCountMax);

    // 無線通信に関連するパラメータ (単位は Byte) です。
    const size_t Mtu = 1500;
    const size_t BeaconDataSizeMax = 1400;

    // スタブ用の MAC アドレスです。
    const MacAddress StubMacAddress = MakeMacAddress(0x12, 0x34, 0x56, 0x78, 0x9a, 0xbc);

}}}} // namespace nn::ldn::detail::<unnamed>

namespace nn { namespace ldn { namespace detail
{
    size_t NetworkInterfaceStub::GetRequiredBufferSize() NN_NOEXCEPT
    {
        return 4096;
    }

    NetworkInterfaceStub::NetworkInterfaceStub(void* buffer, size_t bufferSize) NN_NOEXCEPT
        : m_StateChangeEvent(nn::os::EventClearMode_AutoClear, false),
          m_BeaconReceivedEvent(nn::os::EventClearMode_AutoClear, false),
          m_CancelEvent(nn::os::EventClearMode_AutoClear, false),
          m_Status(L2State_None)
    {
        NN_SDK_ASSERT_NOT_NULL(buffer);
        NN_SDK_ASSERT_ALIGNED(buffer, nn::os::ThreadStackAlignment);
        NN_SDK_ASSERT(GetRequiredBufferSize() <= bufferSize);
        NN_UNUSED(buffer);
        NN_UNUSED(bufferSize);
    }

    NetworkInterfaceStub::~NetworkInterfaceStub() NN_NOEXCEPT
    {
        // ネットワークから離脱します。
        if (m_Status == L2State_AccessPoint || m_Status == L2State_AccessPointCreated)
        {
            CloseAccessPoint();
        }
        else if (m_Status == L2State_Station || m_Status == L2State_StationConnected)
        {
            CloseStation();
        }
    }

    nn::os::SystemEvent& NetworkInterfaceStub::GetStateChangeEvent() NN_NOEXCEPT
    {
        return m_StateChangeEvent;
    }

    nn::os::SystemEvent& NetworkInterfaceStub::GetBeaconReceivedEvent() NN_NOEXCEPT
    {
        return m_BeaconReceivedEvent;
    }

    Result NetworkInterfaceStub::GetNetworkProfile(NetworkProfile* pOutProfile) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutProfile);

        // ネットワークに接続した状態でなければ失敗させます。
        if (m_Status != L2State_AccessPointCreated && m_Status != L2State_StationConnected)
        {
            return ResultInvalidState();
        }

        // ネットワークに接続した状態であれば適当な値を返します。
        pOutProfile->bssid   = StubMacAddress;
        pOutProfile->ssid    = m_Ssid;
        pOutProfile->channel = static_cast<int16_t>(m_Channel);
        NN_RESULT_SUCCESS;
    }

    void NetworkInterfaceStub::GetNetworkInterfaceProfile(
        NetworkInterfaceProfile* pOutProfile) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutProfile);
        std::memset(pOutProfile, 0, sizeof(NetworkInterfaceProfile));
        pOutProfile->mtu = Mtu;
        pOutProfile->capability = 0;
        pOutProfile->nodeCountMax = NodeCountMax;
        pOutProfile->type = NetworkInterfaceType_Wireless80211;
        pOutProfile->physicalAddress = StubMacAddress;
        pOutProfile->beaconDataSizeMax = BeaconDataSizeMax;
        pOutProfile->supportedChannelCount = static_cast<int8_t>(SupportedChannelCount);
        pOutProfile->autoSelectableChannelCount = static_cast<int8_t>(AutoSelectableChannelCount);
        for (int i = 0; i < SupportedChannelCount; ++i)
        {
            pOutProfile->supportedChannels[i] =
                static_cast<int16_t>(SupportedChannels[i]);
        }
        for (int i = 0; i < AutoSelectableChannelCount; ++i)
        {
            pOutProfile->autoSelectableChannels[i] =
                static_cast<int16_t>(AutoSelectableChannels[i]);
        }
        std::strncpy(pOutProfile->name, "stub", NetworkInterfaceNameLengthMax);
    }

    Result NetworkInterfaceStub::Scan(
        L2ScanResult* outBuffer, int* pOutCount, int bufferCount, int channel) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(outBuffer);
        NN_SDK_ASSERT_NOT_NULL(pOutCount);
        NN_SDK_ASSERT(0 < bufferCount);
        NN_SDK_ASSERT(channel == AutoChannel || IsSupportedChannel(channel));
        NN_UNUSED(outBuffer);
        NN_UNUSED(bufferCount);

        // スキャンは STA、あるいはネットワークを構築した AP で実施できます。
        if (m_Status != L2State_Station && m_Status != L2State_StationConnected &&
            m_Status != L2State_AccessPointCreated)
        {
            return ResultInvalidState();
        }

        // スキャンの結果、何も見つからなかったと仮定して結果を返します。
        *pOutCount = 0;
        if (channel == AutoChannel)
        {
            NN_RESULT_SUCCESS;
        }
        else if (IsSupportedChannel(channel))
        {
            NN_RESULT_SUCCESS;
        }
        else
        {
            return ResultBadRequest();
        }
    }

    Result NetworkInterfaceStub::OpenAccessPoint() NN_NOEXCEPT
    {
        NN_SDK_ASSERT_EQUAL(m_Status, L2State_None);
        m_Status = L2State_AccessPoint;
        NN_RESULT_SUCCESS;
    }

    Result NetworkInterfaceStub::CreateNetwork(
        const Ssid& ssid, int channel, int nodeCountMax,
        const void* key, size_t keySize) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_EQUAL(m_Status, L2State_AccessPoint);
        NN_UNUSED(nodeCountMax);
        NN_UNUSED(key);
        NN_UNUSED(keySize);
        m_Ssid = ssid;
        m_Channel = channel == AutoChannel ? SupportedChannels[0] : channel;
        m_Status = L2State_AccessPointCreated;
        NN_RESULT_SUCCESS;
    }


    Result NetworkInterfaceStub::SetBeaconData(const void* data, size_t dataSize) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(dataSize <= BeaconDataSizeMax);
        NN_SDK_ASSERT_EQUAL(m_Status, L2State_AccessPointCreated);
        NN_UNUSED(data);
        NN_UNUSED(dataSize);
        NN_RESULT_SUCCESS;
    }

    Result NetworkInterfaceStub::Reject(MacAddress address) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_EQUAL(m_Status, L2State_AccessPointCreated);
        NN_UNUSED(address);
        NN_RESULT_SUCCESS;
    }

    Result NetworkInterfaceStub::GetStations(
        L2StationInfo* outStations, int* pOutCount, int bufferCount) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT_EQUAL(m_Status, L2State_AccessPointCreated);
        NN_SDK_ASSERT_NOT_NULL(outStations);
        NN_SDK_ASSERT_NOT_NULL(pOutCount);
        NN_SDK_ASSERT(0 < bufferCount);
        NN_UNUSED(outStations);
        NN_UNUSED(bufferCount);

        // ステーションが 1 台もいないと仮定して結果を返します。
        *pOutCount = 0;

        NN_RESULT_SUCCESS;
    }

    Result NetworkInterfaceStub::DestroyNetwork() NN_NOEXCEPT
    {
        NN_SDK_ASSERT_EQUAL(m_Status, L2State_AccessPointCreated);
        m_Status = L2State_AccessPoint;
        NN_RESULT_SUCCESS;
    }

    Result NetworkInterfaceStub::CloseAccessPoint() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_Status == L2State_AccessPoint || m_Status == L2State_AccessPointCreated);

        // ネットワークを破棄します。
        if (m_Status == L2State_AccessPointCreated)
        {
            DestroyNetwork();
        }
        NN_SDK_ASSERT_EQUAL(m_Status, L2State_AccessPoint);

        // アクセスポイントとしての動作を停止します。
        m_Status = L2State_None;
        NN_RESULT_SUCCESS;
    }

    Result NetworkInterfaceStub::OpenStation() NN_NOEXCEPT
    {
        NN_SDK_ASSERT_EQUAL(m_Status, L2State_None);
        m_Status = L2State_Station;
        NN_RESULT_SUCCESS;
    }

    Result NetworkInterfaceStub::Connect(
        const Ssid& ssid, int channel, int nodeCountMax,
        const void* key, size_t keySize) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_EQUAL(m_Status, L2State_Station);
        NN_SDK_ASSERT(IsSupportedChannel(channel));
        NN_SDK_ASSERT_NOT_NULL(key);
        NN_UNUSED(nodeCountMax);
        NN_UNUSED(key);
        NN_UNUSED(keySize);

        // 接続不可能な無線チャンネルを指定された場合には失敗します。
        if (!IsSupportedChannel(channel))
        {
            return ResultNetworkNotFound();
        }

        // ネットワークに接続したと仮定します。
        m_Ssid = ssid;
        m_Channel = channel;
        m_Status = L2State_StationConnected;
        NN_RESULT_SUCCESS;
    }

    bool NetworkInterfaceStub::GetBeaconData(L2ScanResult* pOutBeacon) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT_EQUAL(m_Status, L2State_StationConnected);
        NN_SDK_ASSERT_NOT_NULL(pOutBeacon);
        NN_UNUSED(pOutBeacon);
        return true;
    }

    Result NetworkInterfaceStub::Disconnect() NN_NOEXCEPT
    {
        NN_SDK_ASSERT_EQUAL(m_Status, L2State_StationConnected);
        m_Status = L2State_Station;
        NN_RESULT_SUCCESS;
    }

    Result NetworkInterfaceStub::GetConnectionStatus(L2StationInfo* pOut) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_Status == L2State_Station || m_Status == L2State_StationConnected);
        NN_SDK_ASSERT_NOT_NULL(pOut);

        // 0 埋めします。
        std::memset(pOut, 0, sizeof(L2StationInfo));

        // 適当な状態を返します。
        if (m_Status == L2State_StationConnected)
        {
            pOut->aid = 1;
            pOut->status = L2StationState_Connected;
            pOut->macAddress = StubMacAddress;
            pOut->rssi = -30;
        }
        else
        {
            pOut->status = L2StationState_Disconnected;
            pOut->macAddress = StubMacAddress;
            pOut->disconnectReason = DisconnectReason_Unknown;
            pOut->rssi = RssiMin;
        }
        NN_RESULT_SUCCESS;
    }

    Result NetworkInterfaceStub::CloseStation() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_Status == L2State_Station || m_Status == L2State_StationConnected);

        // ネットワークから切断します。
        if (m_Status == L2State_StationConnected)
        {
            Disconnect();
        }

        // ステーションとしての動作を停止します。
        m_Status = L2State_None;
        NN_RESULT_SUCCESS;
    }

    Result NetworkInterfaceStub::Send(const void* data, size_t dataSize) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_Status == L2State_AccessPointCreated ||
                      m_Status == L2State_StationConnected);
        NN_SDK_ASSERT_NOT_NULL(data);
        NN_SDK_ASSERT(dataSize <= sizeof(NintendoEthernetFrame));
        NN_UNUSED(data);
        NN_UNUSED(dataSize);
        NN_RESULT_SUCCESS;
    }

    Result NetworkInterfaceStub::Receive(
        void* buffer, size_t* pOutSize, size_t bufferSize) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_Status == L2State_AccessPointCreated ||
                      m_Status == L2State_StationConnected);
        NN_SDK_ASSERT_NOT_NULL(buffer);
        NN_SDK_ASSERT_NOT_NULL(pOutSize);
        NN_SDK_ASSERT(sizeof(EthernetHeader) <= bufferSize);
        NN_UNUSED(buffer);
        NN_UNUSED(bufferSize);

        // キャンセルされるまで待機します。
        m_CancelEvent.Wait();
        *pOutSize = 0;
        return ResultCancelled();
    }

    Result NetworkInterfaceStub::Cancel() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_Status == L2State_AccessPointCreated ||
                      m_Status == L2State_StationConnected);

        // 受信処理をキャンセルします。
        m_CancelEvent.Signal();

        NN_RESULT_SUCCESS;
    }

    L2State NetworkInterfaceStub::GetState() const NN_NOEXCEPT
    {
        return m_Status;
    }

    void NetworkInterfaceStub::SetOperationMode(OperationMode mode) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_Status == L2State_None);
        NN_UNUSED(mode);
    }

    void NetworkInterfaceStub::SetWirelessControllerRestriction(
        WirelessControllerRestriction restriction) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_Status == L2State_None);
        NN_UNUSED(restriction);
    }

    int NetworkInterfaceStub::ConvertRssiToLinkLevel(int rssi) const NN_NOEXCEPT
    {
        return rssi == RssiMin ? LinkLevelMin : LinkLevelMax;
    }

}}} // namespace nn::ldn::detail
