﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/ldn/ldn_Result.h>
#include <nn/ldn/detail/Debug/ldn_Log.h>
#include <nn/ldn/detail/NetworkInterface/ldn_NetworkInterfaceRequestImpl-os.win.h>
#include <nn/ldn/detail/NetworkInterface/ldn_NetworkInterfaceStub.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_Mutex.h>

namespace nn { namespace ldn { namespace detail { namespace impl { namespace
{
    typedef NetworkInterfaceStub NetworkInterface;
    NN_OS_ALIGNAS_THREAD_STACK char g_InterfaceBuffer[4 * 1024];
    std::aligned_storage<sizeof(NetworkInterface),
        NN_ALIGNOF(NetworkInterface)>::type g_InterfaceStorage;

    nn::os::Mutex g_Mutex(false);
    nn::os::Event g_ReleasedEvent(nn::os::EventClearMode_AutoClear);
    NetworkInterfaceRequestImpl* g_pActive;

}}}}} // namespace nn::ldn::detail::impl::<unnamed>

namespace nn { namespace ldn { namespace detail { namespace impl
{
    NetworkInterfaceRequestImpl::NetworkInterfaceRequestImpl() NN_NOEXCEPT
        : m_SystemEvent(nn::os::EventClearMode_AutoClear, false),
          m_pNetworkInterface(nullptr),
          m_LastError(ResultDeviceNotAvailable())
    {
    }

    NetworkInterfaceRequestImpl::NetworkInterfaceRequestImpl(
        NetworkInterfaceRequestImpl&& other) NN_NOEXCEPT
        : m_SystemEvent(nn::os::EventClearMode_AutoClear, false)
    {
        *this = std::move(other);
    }

    NetworkInterfaceRequestImpl::~NetworkInterfaceRequestImpl() NN_NOEXCEPT
    {
        if (g_pActive == this)
        {
            Release();
        }
    }

    NetworkInterfaceRequestImpl& NetworkInterfaceRequestImpl::operator = (
        NetworkInterfaceRequestImpl&& other) NN_NOEXCEPT
    {
        if (this != &other)
        {
            m_LastError = other.m_LastError;
            m_Param = other.m_Param;
            m_pNetworkInterface = other.m_pNetworkInterface;
            if (g_pActive == &other)
            {
                g_pActive = this;
            }
        }
        return *this;
    }

    Result NetworkInterfaceRequestImpl::Submit(
        const NetworkInterfaceRequestParameter& param) NN_NOEXCEPT
    {
        // インタフェースを確保できるまで待機します。
        for (;;)
        {
            {
                std::lock_guard<nn::os::Mutex> lock(g_Mutex);
                if (g_pActive == nullptr)
                {
                    // インタフェースが未使用なのでこのクライアントに占有させます。
                    g_pActive = this;
                    m_LastError = ResultSuccess();
                    m_Param = param;
                    m_pNetworkInterface = new (&g_InterfaceStorage) NetworkInterface(
                        g_InterfaceBuffer, sizeof(g_InterfaceBuffer));
                    break;
                }
                else if (g_pActive->m_Param.priority <= param.priority)
                {
                    // 優先度が等しい、あるいはより優先度が高い要求が存在するため諦めます。
                    m_LastError = ResultDeviceOccupied();
                    break;
                }
                else
                {
                    // この要求の方が優先度が高いため、既にインタフェースを確保しているクライアントに、
                    // インタフェースの解放を要求します。
                    g_pActive->m_LastError = ResultDeviceOccupied();
                    g_pActive->GetRequestStateChangeEvent().Signal();
                }
            }
            {
                // インタフェースが解放されるまで待機します。
                g_ReleasedEvent.Wait();
            }
        }
        return m_LastError;
    }

    void NetworkInterfaceRequestImpl::Release() NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(g_Mutex);
        if (g_pActive == this)
        {
            m_pNetworkInterface->~INetworkInterface();
            m_pNetworkInterface = nullptr;
            g_pActive = nullptr;
            m_LastError = ResultDeviceNotAvailable();
            m_SystemEvent.Signal();
            g_ReleasedEvent.Signal();
            std::memset(&m_Param, 0, sizeof(NetworkInterfaceRequestParameter));
        }
    }

    INetworkInterface* NetworkInterfaceRequestImpl::GetNetworkInterface() NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(m_pNetworkInterface);
        return m_pNetworkInterface;
    }

    Result NetworkInterfaceRequestImpl::GetResult() const NN_NOEXCEPT
    {
        return m_LastError;
    }

    nn::os::SystemEvent& NetworkInterfaceRequestImpl::GetRequestStateChangeEvent() NN_NOEXCEPT
    {
        return m_SystemEvent;
    }

}}}} // namespace nn::ldn::detail::impl
