﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <new>
#include <nn/ldn/ldn_PrivateResult.h>
#include <nn/ldn/ldn_PrivateTypes.h>
#include <nn/ldn/ldn_Result.h>
#include <nn/ldn/detail/Debug/ldn_Log.h>
#include <nn/ldn/detail/NetworkInterface/ldn_NetworkInterfaceRequestImpl-os.horizon.h>
#include <nn/ldn/detail/NetworkInterface/ldn_Wlan.h>
#include <nn/nifm/nifm_ApiForSystem.h>
#include <nn/nifm/nifm_ApiRequest.h>
#include <nn/nifm/nifm_ApiRequestPrivate.h>
#include <nn/nifm/nifm_Request.h>
#include <nn/nifm/nifm_ResultPrivate.h>
#include <nn/nifm/nifm_TypesRequirement.h>
#include <nn/util/util_TypedStorage.h>

namespace nn { namespace ldn { namespace detail { namespace impl { namespace
{
    typedef Wlan NetworkInterface;
    NN_OS_ALIGNAS_THREAD_STACK char g_InterfaceBuffer[20 * 1024];
    std::aligned_storage<sizeof(NetworkInterface),
        NN_ALIGNOF(NetworkInterface)>::type g_InterfaceStorage;

    Result ConvertNifmResultToLdnResult(Result result) NN_NOEXCEPT
    {
        if (result.IsSuccess())
        {
            return ResultSuccess();
        }
        else if (nn::nifm::ResultLowPriority::Includes(result))
        {
            return ResultDeviceOccupied();
        }
        else if (nn::nifm::ResultDeviceShutdown::Includes(result))
        {
            return ResultDeviceShutdown();
        }
        else if (nn::nifm::ResultNetworkInterfaceNoLongerAvailable::Includes(result) ||
                 nn::nifm::ResultNetworkInterfaceNotAvailable::Includes(result))
        {
            return ResultWifiOff();
        }
        else if (nn::nifm::ResultDevicePutToSleep::Includes(result))
        {
            return ResultSleep();
        }
        else if (nn::nifm::ResultLocked::Includes(result))
        {
            return ResultLocked();
        }
        else
        {
            // 念の為、適当なエラーを返して処理を継続できるようにしておきます。
            NN_SDK_ASSERT(false, "Unknown nifm result: %08X\n", result.GetInnerValueForDebug());
            return ResultDeviceOccupied();
        }
    }

}}}}} // namespace nn::ldn::detail::impl::<unnamed>

namespace nn { namespace ldn { namespace detail { namespace impl
{
    NetworkInterfaceRequestImpl::NetworkInterfaceRequestImpl() NN_NOEXCEPT
        : m_pRequest(nullptr),
          m_pNetworkInterface(nullptr)
    {
    }

    NetworkInterfaceRequestImpl::NetworkInterfaceRequestImpl(
        NetworkInterfaceRequestImpl&& other) NN_NOEXCEPT
    {
        *this = std::move(other);
    }

    NetworkInterfaceRequestImpl::~NetworkInterfaceRequestImpl() NN_NOEXCEPT
    {
        if (m_pRequest != nullptr)
        {
            Release();
        }
    }

    NetworkInterfaceRequestImpl& NetworkInterfaceRequestImpl::operator = (
        NetworkInterfaceRequestImpl&& other) NN_NOEXCEPT
    {
        if (this != &other)
        {
            m_pRequest = other.m_pRequest;
            m_pNetworkInterface = other.m_pNetworkInterface;
            m_Param = other.m_Param;
            other.m_pRequest = nullptr;
            other.m_pNetworkInterface = nullptr;
            std::memset(&other.m_Param, 0, sizeof(NetworkInterfaceRequestParameter));
        }
        return *this;
    }

    Result NetworkInterfaceRequestImpl::Submit(
        const NetworkInterfaceRequestParameter& param) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_pRequest == nullptr);
        NN_SDK_ASSERT(m_pNetworkInterface == nullptr);

        // ネットワークインタフェースの占有を要求するためのリクエストを生成します。
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::InitializeSystem());
        nn::nifm::RequestParameters requestParams;
        if (UserPriority <= param.priority)
        {
            requestParams.requirementPreset = nn::nifm::RequirementPreset_LocalGeneric;
        }
        else
        {
            requestParams.requirementPreset = nn::nifm::RequirementPreset_LocalForApplet;
        }
        m_pRequest = new nn::nifm::Request(requestParams);

        // ネットワークインタフェースの占有が可能か NIFM に問い合わせます。
        m_pRequest->SubmitAndWait();

        // ネットワークインタフェース占有の成否を確認します。
        Result result = GetResult();
        if (result.IsSuccess())
        {
            m_pNetworkInterface = new (&g_InterfaceStorage) NetworkInterface(
                g_InterfaceBuffer, sizeof(g_InterfaceBuffer));
            m_Param = param;
        }
        else
        {
            delete m_pRequest;
            m_pRequest = nullptr;
        }
        return result;
    }

    void NetworkInterfaceRequestImpl::Release() NN_NOEXCEPT
    {
        if (m_pRequest != nullptr)
        {
            m_pRequest->Cancel();
            m_pNetworkInterface->~INetworkInterface();
            m_pNetworkInterface = nullptr;
            delete m_pRequest;
            m_pRequest = nullptr;
            std::memset(&m_Param, 0, sizeof(NetworkInterfaceRequestParameter));
        }
    }

    INetworkInterface* NetworkInterfaceRequestImpl::GetNetworkInterface() NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(m_pNetworkInterface);
        return m_pNetworkInterface;
    }

    Result NetworkInterfaceRequestImpl::GetResult() const NN_NOEXCEPT
    {
        if (m_pRequest != nullptr)
        {
            return ConvertNifmResultToLdnResult(m_pRequest->GetResult());
        }
        else
        {
            return ResultDeviceNotAvailable();
        }
    }

    nn::os::SystemEvent& NetworkInterfaceRequestImpl::GetRequestStateChangeEvent() NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(m_pRequest);
        return m_pRequest->GetSystemEvent();
    }

}}}} // namespace nn::ldn::detail::impl
