﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <mutex>
#include <nn/ldn/detail/Debug/ldn_Assert.h>
#include <nn/ldn/detail/Debug/ldn_Log.h>
#include <nn/ldn/detail/NetworkInterface/ldn_FrameQueue.h>

namespace nn { namespace ldn { namespace detail { namespace
{

}}}} // namespace nn::ldn::detail::<unnamed>

namespace nn { namespace ldn { namespace detail
{
    FrameQueue::FrameQueue() NN_NOEXCEPT
        : m_Buffer(nullptr),
          m_Size(0),
          m_Capacity(0),
          m_Head(0),
          m_Tail(0),
          m_Count(0),
          m_Mutex(false)
    {
    }

    FrameQueue::~FrameQueue() NN_NOEXCEPT
    {
        if (m_Buffer)
        {
            Finalize();
        }
    }

    size_t FrameQueue::GetRequiredBufferSize(size_t size, int capacity) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(0U < size);
        NN_SDK_ASSERT(0 < capacity);
        return (sizeof(FrameQueueItemHeader) + size) * capacity;
    }

    void FrameQueue::Initialize(
        void* buffer, size_t bufferSize, size_t size, int capacity) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_Buffer == nullptr);
        NN_SDK_ASSERT_NOT_NULL(buffer);
        NN_SDK_ASSERT(GetRequiredBufferSize(size, capacity) <= bufferSize);
        m_Buffer = static_cast<Bit8*>(buffer);
        m_Size = size;
        m_Capacity = std::min(
            capacity,
            static_cast<int>(bufferSize / (sizeof(FrameQueueItemHeader) + size)));
        m_Head = 0;
        m_Tail = 0;
        m_Count = 0;
    }

    void FrameQueue::Finalize() NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(m_Buffer);
        m_Buffer = nullptr;
        m_Size = 0;
        m_Capacity = 0;
        m_Head = 0;
        m_Tail = 0;
        m_Count = 0;
    }

    void FrameQueue::Enqueue(const void* data, size_t dataSize) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(TryEnqueue(data, dataSize));
    }

    bool FrameQueue::TryEnqueue(const void* data, size_t dataSize) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(m_Buffer);
        NN_SDK_ASSERT_NOT_NULL(data);
        NN_SDK_ASSERT(dataSize <= m_Size);
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);

        // 空き領域がありません。
        if (m_Count == m_Capacity)
        {
            return false;
        }

        // データをコピーします。
        auto& item = GetItem(m_Tail);
        item.header.dataSize = static_cast<uint16_t>(std::min(m_Size, dataSize));
        std::memcpy(item.data, data, item.header.dataSize);
        m_Tail = (m_Tail + 1) % m_Capacity;
        ++m_Count;
        return true;
    }

    void FrameQueue::Dequeue(void* buffer, size_t* pOutSize, size_t bufferSize) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(TryDequeue(buffer, pOutSize, bufferSize));
    }

    bool FrameQueue::TryDequeue(void* buffer, size_t* pOutSize, size_t bufferSize) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(m_Buffer);
        NN_SDK_ASSERT_NOT_NULL(buffer);
        NN_SDK_ASSERT_NOT_NULL(pOutSize);
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);

        // データが存在しません。
        if (m_Count == 0)
        {
            return false;
        }

        // データをコピーします。
        const auto& item = GetItem(m_Head);
        *pOutSize = std::min<size_t>(item.header.dataSize, bufferSize);
        std::memcpy(buffer, item.data, *pOutSize);
        m_Head = (m_Head + 1) % m_Capacity;
        --m_Count;
        return true;
    }

    void FrameQueue::Dequeue() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(TryDequeue());
    }

    bool FrameQueue::TryDequeue() NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(m_Buffer);
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);

        // データが存在しません。
        if (m_Count == 0)
        {
            return false;
        }

        // データを削除します。
        m_Head = (m_Head + 1) % m_Capacity;
        --m_Count;
        return true;
    }

    void FrameQueue::Clear() NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        m_Head = 0;
        m_Tail = 0;
        m_Count = 0;
    }

    const void* FrameQueue::GetFront(size_t* pOutSize) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(m_Buffer);
        NN_SDK_ASSERT_NOT_EQUAL(0, m_Count);
        NN_SDK_ASSERT_NOT_NULL(pOutSize);
        const auto& item = GetItem(m_Head);
        *pOutSize = item.header.dataSize;
        return item.data;
    }

    bool FrameQueue::IsEmpty() const NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(m_Buffer);
        return m_Count == 0;
    }

    FrameQueue::FrameQueueItem& FrameQueue::GetItem(int index) NN_NOEXCEPT
    {
        return *reinterpret_cast<FrameQueueItem*>(
            m_Buffer + (sizeof(FrameQueueItemHeader) + m_Size) * index);
    }

    const FrameQueue::FrameQueueItem& FrameQueue::GetItem(int index) const NN_NOEXCEPT
    {
        return *reinterpret_cast<FrameQueueItem*>(
            m_Buffer + (sizeof(FrameQueueItemHeader) + m_Size) * index);
    }

}}} // namespace nn::ldn::detail
