﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/hid/system/hid_Irsensor.h>
#include <nn/irsensor/irsensor_IIrSensorServer.h>
#include <nn/irsensor/irsensor_HandAnalysis.h>
#include <nn/irsensor/irsensor_TeraPluginProcessorApiPrivate.h>
#include <nn/irsensor/irsensor_HeartRateProcessor.h>
#include <../source/OteteLib/src/OteteLib/HeartRateDriver.h>
#include <../external/core/source/numerical/nerd_numerical_utils.h>
#include "detail/irsensor_IrCameraHandle.h"

float const nerd_gillette_internal::math_constants<float>::PI = 3.1415f;

namespace nn { namespace irsensor {

namespace detail {

struct _HeartRateProcessorWorkBuffer
{
    HeartRateDriver* pHeartRateDriver;
    nn::util::TypedStorage<HeartRateDriver, sizeof(HeartRateDriver), NN_ALIGNOF(HeartRateDriver)> heartRateDriverStorage;
    uint8_t workBuffer[HeartRateDriver::s_requiredMemSize];
};

_HeartRateProcessorWorkBuffer* g_UserWorkBufferShadowing[::nn::hid::system::IrSensorSupportedNpadIdsCount] = { nullptr };

}

NN_STATIC_ASSERT( sizeof(HeartRateProcessorWorkBuffer) >= sizeof(detail::_HeartRateProcessorWorkBuffer) );

void RunHeartRateProcessor(const IrCameraHandle& handle, const HeartRateProcessorConfig& config, HeartRateProcessorWorkBuffer& workBuffer) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(&workBuffer);

    auto playerNumber = detail::GetIrCameraHandlePlayerNumber(handle);
    NN_SDK_REQUIRES( playerNumber >= 0 && playerNumber < ::nn::hid::system::IrSensorSupportedNpadIdsCount);

    detail::g_UserWorkBufferShadowing[playerNumber] = reinterpret_cast<detail::_HeartRateProcessorWorkBuffer*>(&workBuffer);

    uint32_t measurementDurationInMs = static_cast<uint32_t>(config.measurementDuration.GetMilliSeconds());
    uint32_t measurementDurationExtensionInMs = static_cast<uint32_t>(config.measurementDurationExtension.GetMilliSeconds());
    HeartRateCaptureConfig heartRateCaptureConfig(measurementDurationInMs, measurementDurationExtensionInMs);

    detail::_HeartRateProcessorWorkBuffer* pHeartRateProcessorWorkBuffer = reinterpret_cast<detail::_HeartRateProcessorWorkBuffer*>(&workBuffer);
    pHeartRateProcessorWorkBuffer->pHeartRateDriver = new (&pHeartRateProcessorWorkBuffer->heartRateDriverStorage) HeartRateDriver(static_cast<unsigned char*>(pHeartRateProcessorWorkBuffer->workBuffer), static_cast<int>(HeartRateDriver::s_requiredMemSize), heartRateCaptureConfig);

    nn::irsensor::HandAnalysisConfig handAnalysisConfig;
    handAnalysisConfig.mode = HandAnalysisMode_Silhouette;
    pHeartRateProcessorWorkBuffer->pHeartRateDriver->StartCapture();
    nn::irsensor::RunHandAnalysis(handle, handAnalysisConfig);
}

::nn::Result GetHeartRateProcessorState(HeartRateProcessorState* pHeartRateCaptureState, nn::irsensor::IrCameraHandle const& handle) NN_NOEXCEPT
{
    using namespace nn::irsensor;

    NN_SDK_REQUIRES_NOT_NULL(pHeartRateCaptureState);

    auto playerNumber = detail::GetIrCameraHandlePlayerNumber(handle);
    NN_SDK_REQUIRES( playerNumber >= 0 && playerNumber < ::nn::hid::system::IrSensorSupportedNpadIdsCount);

    detail::_HeartRateProcessorWorkBuffer *pHeartRateProcessorWorkBuffer = detail::g_UserWorkBufferShadowing[playerNumber];
    NN_SDK_REQUIRES_NOT_NULL(pHeartRateProcessorWorkBuffer);

    const int HandAnalysis2PrefixSize = 2;
    const uint32_t HandAnalysis2PrefixSilhouette = 0x2;

    int count = 0;
    size_t const maxCount = 16;
    TeraPluginProcessorState states[maxCount];
    GetTeraPluginProcessorStates(states, &count, maxCount, 0, HandAnalysis2PrefixSilhouette, HandAnalysis2PrefixSize, handle);

    NN_RESULT_THROW_UNLESS(count != 0, ResultIrsensorNotReady());

    TeraPacket packets[maxCount];
    for (size_t i = 0; i < size_t(count); ++i)
    {
        auto& packet = packets[i];
        packet.id = static_cast<size_t>(states[i].samplingNumber);
        pHeartRateCaptureState->samplingNumber = states[i].samplingNumber;
        packet.data = states[i].data;
        packet.size = 300;
    }

    HeartRateCaptureState heartRateCaptureState;
    NN_RESULT_THROW_UNLESS( pHeartRateProcessorWorkBuffer->pHeartRateDriver->GetCaptureState(&heartRateCaptureState, packets, count) == true, ResultIrsensorNotReady() );

    {
        pHeartRateCaptureState->bpm = heartRateCaptureState.bpm;
        pHeartRateCaptureState->confidence = heartRateCaptureState.confidence;
        NN_SDK_REQUIRES_MINMAX(pHeartRateCaptureState->confidence, 0.0f, 1.0f);
        pHeartRateCaptureState->userId = heartRateCaptureState.id;
        pHeartRateCaptureState->newSignalValues = heartRateCaptureState.newSignalValues;
        pHeartRateCaptureState->newSignalValuesCount = heartRateCaptureState.newSignalValuesCount;
        pHeartRateCaptureState->signalQualityInstant = heartRateCaptureState.signalQualityInstant;
        pHeartRateCaptureState->signalQualityAverage = heartRateCaptureState.signalQualityAverage;
        pHeartRateCaptureState->progress = heartRateCaptureState.progress;
        NN_SDK_REQUIRES_MINMAX(pHeartRateCaptureState->progress, 0.0f, 1.0f);

        switch (heartRateCaptureState.status)
        {
        case HeartRateCaptureStatus::Invalid:
            pHeartRateCaptureState->status = HeartRateStatus_Invalid;
            break;
        case HeartRateCaptureStatus::Waiting_For_User:
            pHeartRateCaptureState->status = HeartRateStatus_WaitingForUser;
            break;
        case HeartRateCaptureStatus::Measurement_In_Progress:
            pHeartRateCaptureState->status = HeartRateStatus_MeasurementInProgress;
            break;
        case HeartRateCaptureStatus::Measurement_In_Progress_Extension:
            pHeartRateCaptureState->status = HeartRateStatus_MeasurementInProgressExtension;
            break;
        case HeartRateCaptureStatus::Measurement_Completed:
            pHeartRateCaptureState->status = HeartRateStatus_MeasurementCompleted;
            break;
        case HeartRateCaptureStatus::Measurement_Aborted:
            pHeartRateCaptureState->status = HeartRateStatus_MeasurementAborted;
            break;
        default:
            pHeartRateCaptureState->status = HeartRateStatus_Invalid;
            break;
        }
    }

    NN_RESULT_SUCCESS;
}

void ResetHeartRateProcessorState(nn::irsensor::IrCameraHandle const& handle) NN_NOEXCEPT
{
    auto playerNumber = detail::GetIrCameraHandlePlayerNumber(handle);
    NN_SDK_REQUIRES( playerNumber >= 0 && playerNumber < ::nn::hid::system::IrSensorSupportedNpadIdsCount);

    detail::_HeartRateProcessorWorkBuffer* pHeartRateProcessorWorkBuffer = detail::g_UserWorkBufferShadowing[playerNumber];
    NN_SDK_REQUIRES_NOT_NULL(pHeartRateProcessorWorkBuffer);

    pHeartRateProcessorWorkBuffer->pHeartRateDriver->StartCapture();
}

} }
