﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nn/nn_StaticAssert.h>

#include <nn/result//result_HandlingUtility.h>

#include <algorithm>
#include <gillette_Api.h>


#include <nn/irsensor/irsensor_AdaptiveClusteringProcessor.h>
#include <nn/irsensor/irsensor_TeraPluginProcessorTypesPrivate.h>
#include <nn/irsensor/irsensor_TeraPluginProcessorApiPrivate.h>
#include <nn/irsensor/irsensor_ImageProcessorTypes.h>
#include <nn/irsensor/irsensor_ResultPrivate.h>
#include <nn/irsensor/irsensor_Result.public.h>

#include <nn/xcd/xcd_IrsensorTypes.h>

NN_STATIC_ASSERT(nerd::otete::plato::PlatoPacketHeader::MAX_REFLECTION_COUNT == nn::irsensor::AdaptiveClusteringProcessorObjectCountMax);
NN_STATIC_ASSERT(nn::irsensor::AdaptiveClusteringProcessorStateCountMax == nn::irsensor::TeraPluginProcessorStateCountMax);


namespace nn {
namespace irsensor {

struct TeraPluginParamForAdaptiveClustering
{
    uint8_t maximumGain;
    uint8_t _reserved;
};

NN_STATIC_ASSERT(sizeof(TeraPluginParamForAdaptiveClustering) == sizeof(TeraPluginParameter));

const uint8_t MaxGainForMiddleTargetDistance = 3;
const uint8_t MaxGainForFarTargetDistance    = 8;

::nn::Result DecodeMarkerDetectionState(AdaptiveClusteringProcessorState* pOutState, const TeraPluginProcessorState& markerDetectionState, size_t* pReadCursor, const unsigned char* pPacketData)
{
    NN_SDK_REQUIRES_NOT_NULL(pOutState);

    bool ok = true;
    ::nerd::otete::plato::PlatoPacket platoPacket;

    const size_t packetSize = sizeof(((TeraPluginProcessorState*)(0))->data);
    size_t& readCursor = *pReadCursor;

    pOutState->samplingNumber = markerDetectionState.samplingNumber;
    pOutState->ambientNoiseLevel = markerDetectionState.ambientNoiseLevel;

    {
        ok = ::nerd::otete::plato::DecodePacketROK(&platoPacket, pPacketData, packetSize, &readCursor);

        if (!ok)
        {
            NN_RESULT_THROW(ResultIrsensorNotReady());
        }
    }

    switch(platoPacket.header.type)
    {
        case ::nerd::otete::plato::PlatoPacketHeader::TYPE_WIDE:
            pOutState->accuracyLevel = AdaptiveClusteringAccuracyLevel_Middle;
            break;
        case ::nerd::otete::plato::PlatoPacketHeader::TYPE_SCAN:
            pOutState->accuracyLevel = AdaptiveClusteringAccuracyLevel_Low;
            break;
        case ::nerd::otete::plato::PlatoPacketHeader::TYPE_NARROW:
            pOutState->accuracyLevel = AdaptiveClusteringAccuracyLevel_High;
            break;
        default:
            NN_RESULT_THROW(ResultIrsensorNotReady());
    }

    pOutState->objectCount = platoPacket.header.reflectionCount;
    NN_SDK_ASSERT_MINMAX(pOutState->objectCount, 0, nn::irsensor::AdaptiveClusteringProcessorObjectCountMax);

    pOutState->backgroundIntensity = platoPacket.header.backgroundMean;
    NN_SDK_ASSERT_MINMAX(pOutState->backgroundIntensity, 0, nn::irsensor::IrCameraIntensityMax);

    pOutState->timeStamp = nn::TimeSpanType::FromMicroSeconds(platoPacket.header.samplingNumber * platoPacket.header.samplingPeriod * 100);
    NN_SDK_ASSERT_GREATER(pOutState->timeStamp.GetMicroSeconds(), 0);

    for(int i = 0; i< platoPacket.header.reflectionCount; ++i)
    {
        AdaptiveClusteringData& element = pOutState->objects[i];
        ::nerd::otete::plato::PlatoReflectionPacket& reflection = platoPacket.reflections[i];
        element.centroid.x =  reflection.centroid.x;
        element.centroid.y =  reflection.centroid.y;
        element.isIncomplete = reflection.touchBorder;

        element.area = reflection.areaSqrt * reflection.areaSqrt;
        NN_SDK_ASSERT_MINMAX(element.centroid.x, 0, nn::irsensor::IrCameraImageWidth);
        NN_SDK_ASSERT_MINMAX(element.centroid.y, 0, nn::irsensor::IrCameraImageHeight);
        NN_SDK_ASSERT_MINMAX(element.area, 0, nn::irsensor::IrCameraImageWidth * nn::irsensor::IrCameraImageHeight);
    }

    NN_RESULT_SUCCESS;
}

::nn::Result DecodeOneMarkerDetectionState(AdaptiveClusteringProcessorState* pOutValue, const TeraPluginProcessorState& markerDetectionState)
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    const unsigned char* pPacketData = markerDetectionState.data;
    size_t readCursor = 0;

    NN_RESULT_DO(DecodeMarkerDetectionState(pOutValue, markerDetectionState, &readCursor, pPacketData));

    NN_RESULT_SUCCESS;
}

void RunAdaptiveClusteringProcessor(const IrCameraHandle& handle, const AdaptiveClusteringProcessorConfig& config) NN_NOEXCEPT
{
    int teraMode = AdaptiveClusteringStaticFov;
    if(config.mode == AdaptiveClusteringMode::AdaptiveClusteringMode_StaticFov)
    {
        teraMode = AdaptiveClusteringStaticFov;
    }
    else if (config.mode == AdaptiveClusteringMode::AdaptiveClusteringMode_DynamicFov)
    {
        teraMode = AdaptiveClusteringDynamicFov;
    }

    TeraPluginProcessorConfig teraConfig = {};
    teraConfig.modeOffset = static_cast<uint8_t>(teraMode - static_cast<int>(nn::xcd::IrProcessorType::TeraPlugin));

    TeraPluginParamForAdaptiveClustering param = {};
    switch (config.distance)
    {
    case AdaptiveClusteringTargetDistance_Near:
        teraConfig.isParameterEnabled = false;
        break;
    case AdaptiveClusteringTargetDistance_Middle:
        teraConfig.isParameterEnabled = true;
        param.maximumGain = MaxGainForMiddleTargetDistance;
        memcpy(&teraConfig.param, &param, sizeof(TeraPluginParamForAdaptiveClustering));
        break;
    case AdaptiveClusteringTargetDistance_Far:
        teraConfig.isParameterEnabled = true;
        param.maximumGain = MaxGainForFarTargetDistance;
        memcpy(&teraConfig.param, &param, sizeof(TeraPluginParamForAdaptiveClustering));
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
    RunTeraPluginProcessor(handle, teraConfig);
}

::nn::Result GetAdaptiveClusteringProcessorStates(AdaptiveClusteringProcessorState* pOutStates, int* pOutCount, int countMax, int64_t infSamplingNumber, const IrCameraHandle& handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutStates);
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);

    // NOTE(nerd_marc.delorme): I cannot allocate a dynamic array of TeraPluginProcessorState and I do not want the
    // application developer to provide me some work memory. So I will use the AdaptiveClusteringProcessorState array memory as
    // working memory. If the AdaptiveClusteringProcessorState size is more than twice bigger than TeraPluginProcessorState I am sure for
    // any index except the last one AdaptiveClusteringProcessorState[index] will not overlap with TeraPluginProcessorState[index] if I put
    // the TeraPluginProcessorState array at the end of the AdaptiveCLusteringProcessorState buffer. Example:
    //
    // <------------ StateType ------------><------------ StateType ------------><------------ StateType ------------>
    //                                                             <- HndAnalysis -><- HndAnalysis -><- HndAnalysis ->
    //
    // I will just allocate one HandAnalysisState on the stack for the last element.
    NN_STATIC_ASSERT(sizeof(AdaptiveClusteringProcessorState) >= 2 * sizeof(TeraPluginProcessorState));

    countMax = ::std::min(countMax, AdaptiveClusteringProcessorStateCountMax);

    const int outValueArraySize          = countMax * sizeof(AdaptiveClusteringProcessorState);
    const int markerDetectionStateStateArraySize = countMax * sizeof(TeraPluginProcessorState);

    TeraPluginProcessorState* pMarkerDetectionState = reinterpret_cast<TeraPluginProcessorState*>(reinterpret_cast<char*>(pOutStates) + outValueArraySize - markerDetectionStateStateArraySize);

    NN_RESULT_DO(GetTeraPluginProcessorStates(pMarkerDetectionState, pOutCount, countMax, infSamplingNumber, 0, 0, handle));

    if (*pOutCount > 0)
    {
        const int lastIndex   = *pOutCount - 1;

        for (int i = 0; i < lastIndex; i++)
        {
            NN_RESULT_DO(DecodeOneMarkerDetectionState(&pOutStates[i], pMarkerDetectionState[i]));
        }

        {
            TeraPluginProcessorState markerState = pMarkerDetectionState[lastIndex];
            NN_RESULT_DO(DecodeOneMarkerDetectionState(&pOutStates[lastIndex], markerState));
        }
    }

    NN_RESULT_SUCCESS;
}

}} // namespace nn::irsensor
