﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/os/os_MemoryFence.h>
#include <nn/result/result_HandlingUtility.h>

#include "irsensor_StatusManager.h"
#include "irsensor_IrCameraHandle.h"

namespace nn { namespace irsensor { namespace detail {

StatusManager::StatusManager() NN_NOEXCEPT
{
    for (auto i = 0; i < ::nn::hid::system::IrSensorSupportedNpadIdsCount; i++)
    {
        m_SharedMemoryFormat.deviceFormat[i].cameraStatus = IrCameraStatus_Unconnected;
        // 互換性維持のため初期状態は Available 状態としている
        m_SharedMemoryFormat.deviceFormat[i].internalCameraStatus = IrCameraInternalStatus_Available;
        m_SharedMemoryFormat.deviceFormat[i].mode = IrSensorMode::None;
        memset(&(m_SharedMemoryFormat.deviceFormat[i].ringLifoStorage), 0, sizeof(RingLifoStorage));
    }
    for (auto i = 0; i < AppletStatusEntryCountMax; i++)
    {
        m_SharedMemoryFormat.appletFormat[i].aruid = nn::applet::AppletResourceUserId::GetInvalidId();
        m_SharedMemoryFormat.appletFormat[i].appletStatusFlag = AppletStatusFlagSet();
    }
}

StatusManager::~StatusManager() NN_NOEXCEPT
{
    // 何もしない
}

//!< IrCameraStatus を取得します。
IrCameraStatus StatusManager::GetIrCameraStatus(const IrCameraHandle& handle) const NN_NOEXCEPT
{
    auto playerNumber = GetIrCameraHandlePlayerNumber(handle);
    int status = static_cast<int>(m_SharedMemoryFormat.deviceFormat[playerNumber].cameraStatus);
    switch (status)
    {
    case IrCameraStatus_Available:
        return IrCameraStatus_Available;
    case IrCameraStatus_Unconnected:
        return IrCameraStatus_Unconnected;
    case IrCameraStatus_Unsupported:
        return IrCameraStatus_Unsupported;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

//!< IrCameraInternalStatus を設定します。
void StatusManager::SetIrCameraStatus(const IrCameraHandle& handle, IrCameraStatus status) NN_NOEXCEPT
{
    auto playerNumber = GetIrCameraHandlePlayerNumber(handle);
    m_SharedMemoryFormat.deviceFormat[playerNumber].cameraStatus= status;
}

//!< IrCameraInternalStatus を取得します。
IrCameraInternalStatus StatusManager::GetIrCameraInternalStatus(const IrCameraHandle& handle) const NN_NOEXCEPT
{
    auto playerNumber = GetIrCameraHandlePlayerNumber(handle);
    int status = static_cast<int>(m_SharedMemoryFormat.deviceFormat[playerNumber].internalCameraStatus);
    switch (status)
    {
    case IrCameraInternalStatus_Available:
        return IrCameraInternalStatus_Available;
    case IrCameraInternalStatus_NeedUpdate:
        return IrCameraInternalStatus_NeedUpdate;
    case IrCameraInternalStatus_HardwareError:
        return IrCameraInternalStatus_HardwareError;
    case IrCameraInternalStatus_NeedCharge:
        return IrCameraInternalStatus_NeedCharge;
    case IrCameraInternalStatus_HidResourceError:
        return IrCameraInternalStatus_HidResourceError;
    case IrCameraInternalStatus_CheckVersionIncompleted:
        return IrCameraInternalStatus_CheckVersionIncompleted;
    case IrCameraInternalStatus_NeedUpdateOnInitial:
        return IrCameraInternalStatus_NeedUpdateOnInitial;
    case IrCameraInternalStatus_SuccessfulTerminated:
        return IrCameraInternalStatus_SuccessfulTerminated;
    case IrCameraInternalStatus_Setting:
        return IrCameraInternalStatus_Setting;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

//!< IrCameraInternalStatus を設定します。
void StatusManager::SetIrCameraInternalStatus(const IrCameraHandle& handle, IrCameraInternalStatus status) NN_NOEXCEPT
{
    auto playerNumber = GetIrCameraHandlePlayerNumber(handle);
    m_SharedMemoryFormat.deviceFormat[playerNumber].internalCameraStatus= status;
}

//!< AruidStatus を取得します。
bool StatusManager::GetIrSensorAruidStatus(
    AppletStatusFlagSet* pOutValue,
    const ::nn::applet::AppletResourceUserId& aruid) NN_NOEXCEPT
{
    bool isRegistered = false;
    for (auto i = 0; i < AppletStatusEntryCountMax; i++)
    {
        if (m_SharedMemoryFormat.appletFormat[i].aruid == aruid)
        {
            isRegistered = true;
            *pOutValue = m_SharedMemoryFormat.appletFormat[i].appletStatusFlag;
            break;
        }
    }
    return isRegistered;
}

//!< AruidStatus を設定します。
void StatusManager::SetIrSensorAruidStatus(
    const ::nn::applet::AppletResourceUserId& aruid,
    AppletStatusFlagSet flag) NN_NOEXCEPT
{
    bool isRegistered = false;
    // 既に登録済みの Aruid の場合はフラグを更新
    for (auto i = 0; i < AppletStatusEntryCountMax; i++)
    {
        if (m_SharedMemoryFormat.appletFormat[i].aruid == aruid)
        {
            isRegistered = true;
            m_SharedMemoryFormat.appletFormat[i].appletStatusFlag = flag;
            break;
        }
    }

    // AppletResourceManager で異常時のハンドリングは行われているため、登録されていない場合はアボートする。
    NN_ABORT_UNLESS_EQUAL(isRegistered, true);
    return;
}

//!< AruidStatus を登録します。
void StatusManager::RegisterIrSensorAruid(const ::nn::applet::AppletResourceUserId& aruid) NN_NOEXCEPT
{
    bool isSpaceAvailable = false;
    // 未登録の場合は空きのある場所に登録していく
    for (auto i = 0; i < AppletStatusEntryCountMax; i++)
    {
        if (m_SharedMemoryFormat.appletFormat[i].aruid == ::nn::applet::AppletResourceUserId::GetInvalidId())
        {
            isSpaceAvailable = true;
            m_SharedMemoryFormat.appletFormat[i].aruid = aruid;
            break;
        }
    }

    // AppletResourceManager で異常時のハンドリングは行われているため、登録できない場合はアボートする。
    NN_ABORT_UNLESS_EQUAL(isSpaceAvailable, true);
    return;
}

//!< AruidStatus を破棄します。
void StatusManager::UnregisterIrSensorAruid(const ::nn::applet::AppletResourceUserId& aruid) NN_NOEXCEPT
{
    bool isRegistered = false;
    // 未登録の場合は空きのある場所に登録していく
    for (auto i = 0; i < AppletStatusEntryCountMax; i++)
    {
        if (m_SharedMemoryFormat.appletFormat[i].aruid == aruid)
        {
            isRegistered = true;
            m_SharedMemoryFormat.appletFormat[i].aruid = ::nn::applet::AppletResourceUserId::GetInvalidId();
            break;
        }
    }
    // AppletResourceManager で異常時のハンドリングは行われているため、登録されていない場合はアボートする。
    NN_ABORT_UNLESS_EQUAL(isRegistered, true);
    return;
}

void StatusManager::AllocateMomentLifo(const IrCameraHandle& handle) NN_NOEXCEPT
{
    auto playerNumber = GetIrCameraHandlePlayerNumber(handle);
    NN_SDK_REQUIRES_EQUAL(m_SharedMemoryFormat.deviceFormat[playerNumber].mode, IrSensorMode::None);

    auto pLifo = &::nn::util::Get(m_SharedMemoryFormat.deviceFormat[playerNumber].ringLifoStorage.moment);
    NN_SDK_ASSERT_NOT_NULL(pLifo);
    new (pLifo) MomentLifo();

    m_SharedMemoryFormat.deviceFormat[playerNumber].mode = IrSensorMode::Moment;
    // LIFO の構築の完了を保証
    ::nn::os::FenceMemoryStoreLoad();
}

void StatusManager::AllocateClusteringLifo(const IrCameraHandle& handle) NN_NOEXCEPT
{
    auto playerNumber = GetIrCameraHandlePlayerNumber(handle);
    NN_SDK_REQUIRES_EQUAL(m_SharedMemoryFormat.deviceFormat[playerNumber].mode, IrSensorMode::None);

    auto pLifo = &::nn::util::Get(m_SharedMemoryFormat.deviceFormat[playerNumber].ringLifoStorage.clustering);
    NN_SDK_ASSERT_NOT_NULL(pLifo);
    new (pLifo) ClusteringLifo();

    m_SharedMemoryFormat.deviceFormat[playerNumber].mode = IrSensorMode::Clustering;
    // LIFO の構築の完了を保証
    ::nn::os::FenceMemoryStoreLoad();
}

void StatusManager::AllocatePointingLifo(const IrCameraHandle& handle) NN_NOEXCEPT
{
    auto playerNumber = GetIrCameraHandlePlayerNumber(handle);
    NN_SDK_REQUIRES_EQUAL(m_SharedMemoryFormat.deviceFormat[playerNumber].mode, IrSensorMode::None);

    auto pLifo = &::nn::util::Get(m_SharedMemoryFormat.deviceFormat[playerNumber].ringLifoStorage.pointing);
    NN_SDK_ASSERT_NOT_NULL(pLifo);
    new (pLifo) PointingLifo();

    m_SharedMemoryFormat.deviceFormat[playerNumber].mode = IrSensorMode::Pointing;
    // LIFO の構築の完了を保証
    ::nn::os::FenceMemoryStoreLoad();
}

void StatusManager::AllocateTeraPluginLifo(const IrCameraHandle& handle) NN_NOEXCEPT
{
    auto playerNumber = GetIrCameraHandlePlayerNumber(handle);
    NN_SDK_REQUIRES_EQUAL(m_SharedMemoryFormat.deviceFormat[playerNumber].mode, IrSensorMode::None);

    auto pLifo = &::nn::util::Get(m_SharedMemoryFormat.deviceFormat[playerNumber].ringLifoStorage.teraPlugin);
    NN_SDK_ASSERT_NOT_NULL(pLifo);
    new (pLifo) TeraPluginLifo();

    m_SharedMemoryFormat.deviceFormat[playerNumber].mode = IrSensorMode::TeraPlugin;
    // LIFO の構築の完了を保証
    ::nn::os::FenceMemoryStoreLoad();
}

void StatusManager::FreeLifo(const IrCameraHandle& handle) NN_NOEXCEPT
{
    auto playerNumber = GetIrCameraHandlePlayerNumber(handle);
    switch (m_SharedMemoryFormat.deviceFormat[playerNumber].mode)
    {
    case IrSensorMode::None:
        // 何もしない
        break;
    case IrSensorMode::Moment:
        {
            auto pLifo = GetMomentLifo(handle);
            NN_SDK_ASSERT_NOT_NULL(pLifo);
            pLifo->~MomentLifo();
            pLifo = nullptr;
            // LIFO に対するアクセスの完了を保証
            ::nn::os::FenceMemoryAnyStore();
        }
        break;
    case IrSensorMode::Clustering:
        {
            auto pLifo = GetClusteringLifo(handle);
            NN_SDK_ASSERT_NOT_NULL(pLifo);
            pLifo->~ClusteringLifo();
            pLifo = nullptr;
            // LIFO に対するアクセスの完了を保証
            ::nn::os::FenceMemoryAnyStore();
        }
        break;
    case IrSensorMode::Pointing:
        {
            auto pLifo = GetPointingLifo(handle);
            NN_SDK_ASSERT_NOT_NULL(pLifo);
            pLifo->~PointingLifo();
            pLifo = nullptr;
            // LIFO に対するアクセスの完了を保証
            ::nn::os::FenceMemoryAnyStore();
        }
        break;
    case IrSensorMode::TeraPlugin:
        {
            auto pLifo = GetTeraPluginLifo(handle);
            NN_SDK_ASSERT_NOT_NULL(pLifo);
            pLifo->~TeraPluginLifo();
            pLifo = nullptr;
            // LIFO に対するアクセスの完了を保証
            ::nn::os::FenceMemoryAnyStore();
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    m_SharedMemoryFormat.deviceFormat[playerNumber].mode = IrSensorMode::None;
    ::nn::os::FenceMemoryStoreLoad();
}

void StatusManager::ClearLifo(const IrCameraHandle& handle) NN_NOEXCEPT
{
    auto playerNumber = GetIrCameraHandlePlayerNumber(handle);
    switch (m_SharedMemoryFormat.deviceFormat[playerNumber].mode)
    {
    case IrSensorMode::None:
        // 何もしない
        break;
    case IrSensorMode::Moment:
        {
            auto pLifo = GetMomentLifo(handle);
            NN_SDK_ASSERT_NOT_NULL(pLifo);
            pLifo->Clear();
            // LIFO に対するアクセスの完了を保証
            ::nn::os::FenceMemoryAnyStore();
        }
        break;
    case IrSensorMode::Clustering:
        {
            auto pLifo = GetClusteringLifo(handle);
            NN_SDK_ASSERT_NOT_NULL(pLifo);
            pLifo->Clear();
            // LIFO に対するアクセスの完了を保証
            ::nn::os::FenceMemoryAnyStore();
        }
        break;
    case IrSensorMode::Pointing:
        {
            auto pLifo = GetPointingLifo(handle);
            NN_SDK_ASSERT_NOT_NULL(pLifo);
            pLifo->Clear();
            // LIFO に対するアクセスの完了を保証
            ::nn::os::FenceMemoryAnyStore();
        }
        break;
    case IrSensorMode::TeraPlugin:
        {
            auto pLifo = GetTeraPluginLifo(handle);
            NN_SDK_ASSERT_NOT_NULL(pLifo);
            pLifo->Clear();
            // LIFO に対するアクセスの完了を保証
            ::nn::os::FenceMemoryAnyStore();
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    ::nn::os::FenceMemoryStoreLoad();
}

StatusManager::MomentLifo*
    StatusManager::GetMomentLifo(const IrCameraHandle& handle) const NN_NOEXCEPT
{
    auto playerNumber = GetIrCameraHandlePlayerNumber(handle);
    if (m_SharedMemoryFormat.deviceFormat[playerNumber].mode != IrSensorMode::Moment)
    {
        return nullptr;
    }
    auto pLifo = const_cast<MomentLifo*>(
        &::nn::util::Get(m_SharedMemoryFormat.deviceFormat[playerNumber].ringLifoStorage.moment));
    NN_SDK_REQUIRES_NOT_NULL(pLifo);

    return pLifo;
}

StatusManager::ClusteringLifo*
    StatusManager::GetClusteringLifo(const IrCameraHandle& handle) const NN_NOEXCEPT
{
    auto playerNumber = GetIrCameraHandlePlayerNumber(handle);
    if (m_SharedMemoryFormat.deviceFormat[playerNumber].mode != IrSensorMode::Clustering)
    {
        return nullptr;
    }
    auto pLifo = const_cast<ClusteringLifo*>(
        &::nn::util::Get(m_SharedMemoryFormat.deviceFormat[playerNumber].ringLifoStorage.clustering));
    NN_SDK_REQUIRES_NOT_NULL(pLifo);

    return pLifo;
}

StatusManager::PointingLifo*
    StatusManager::GetPointingLifo(const IrCameraHandle& handle) const NN_NOEXCEPT
{
    auto playerNumber = GetIrCameraHandlePlayerNumber(handle);
    if (m_SharedMemoryFormat.deviceFormat[playerNumber].mode != IrSensorMode::Pointing)
    {
        return nullptr;
    }
    auto pLifo = const_cast<PointingLifo*>(
        &::nn::util::Get(m_SharedMemoryFormat.deviceFormat[playerNumber].ringLifoStorage.pointing));
    NN_SDK_REQUIRES_NOT_NULL(pLifo);

    return pLifo;
}

StatusManager::TeraPluginLifo*
    StatusManager::GetTeraPluginLifo(const IrCameraHandle& handle) const NN_NOEXCEPT
{
    auto playerNumber = GetIrCameraHandlePlayerNumber(handle);
    if (m_SharedMemoryFormat.deviceFormat[playerNumber].mode != IrSensorMode::TeraPlugin)
    {
        return nullptr;
    }
    auto pLifo = const_cast<TeraPluginLifo*>(
        &::nn::util::Get(m_SharedMemoryFormat.deviceFormat[playerNumber].ringLifoStorage.teraPlugin));
    NN_SDK_REQUIRES_NOT_NULL(pLifo);

    return pLifo;
}

}}} // namespace nn::irsensor::detail
