﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/irsensor/irsensor_ResultPrivate.h>
#include <nn/os/os_NativeHandle.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_ISharedObject.h>
#include <nn/sf/sf_NativeHandle.h>
#include <nn/sf/sf_ObjectFactory.h>

#include "irsensor_ResourceManager.h"
#include "irsensor_IrSensorServer.h"
#include "irsensor_StaticObject.h"
#include "../../hid/detail/hid_NpadId.h"

#ifdef NN_BUILD_CONFIG_OS_HORIZON
#include "../../hid/detail/hid_ResourceManager-os.horizon.h"
#endif

#ifdef NN_BUILD_CONFIG_OS_WIN
#include "../../hid/detail/hid_ResourceManager-os.win.h"
#endif

namespace nn { namespace irsensor { namespace detail {

namespace {

//!< os ハンドルを管理するか否かを表す値
const bool NeedsToBeManaged =
#ifdef NN_BUILD_CONFIG_OS_HORIZON
    false;
#else
    false;
#endif

} // namespace

IrSensorServer::IrSensorServer() NN_NOEXCEPT
{
    // 何もしない
}

IrSensorServer::~IrSensorServer() NN_NOEXCEPT
{
    // 何もしない
}

::nn::Result IrSensorServer::GetNpadIrCameraHandle(
    ::nn::sf::Out<IrCameraHandle> outValue,
    ::nn::hid::NpadIdType npadIdType) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(
        outValue.GetPointer() != nullptr,
        ResultIrsensorNullPointer());

    NN_RESULT_DO(::nn::hid::detail::VerifyValidNpadId(npadIdType));

    IrCameraHandle handle;
    NN_RESULT_DO(
        GetResourceManager().GetIrCameraHandle(&handle, npadIdType));
    outValue.Set(handle);
    NN_RESULT_SUCCESS;
}

::nn::Result IrSensorServer::ActivateIrsensor(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    // データのクリア
    GetResourceManager().ClearStatusManager();
#if defined(NN_BUILD_TARGET_PLATFORM_OS_WIN)
    // Win の場合はリソーステーブルを生成
    GetResourceManager().RegisterAppletResourceUserId(aruid, true);
    GetResourceManager().SetAppletResourceUserId(aruid);
#endif
    // リソーステーブルに Irsensor 使用中と設定
    GetResourceManager().SetAppletResourceEntry(aruid);

    for (auto i = 0; i < ::nn::hid::system::IrSensorSupportedNpadIdsCount; i++)
    {
        IrCameraHandle handle;
        MakeIrCameraHandle(&handle, ::nn::hid::system::IrSensorSupportedNpadIds[i]);
        NN_RESULT_DO(GetResourceManager().AssertValidIrCameraHandle(handle));
        // 対応する Xcd Driver をハンドルに設定し、初期化処理をしておく。
        NN_RESULT_DO(GetResourceManager().BindIrCameraXcdDriver(handle));
        NN_RESULT_DO(
            GetResourceManager().GetRequestHandlerTask(handle).Activate());
    }
    NN_RESULT_SUCCESS;
}

::nn::Result IrSensorServer::ActivateIrsensorWithFunctionLevel(
    ::nn::applet::AppletResourceUserId aruid,
    const PackedFunctionLevel& functionLevel) NN_NOEXCEPT
{
    // データのクリア
    GetResourceManager().ClearStatusManager();
#if defined(NN_BUILD_TARGET_PLATFORM_OS_WIN)
    // Win の場合はリソーステーブルを生成
    GetResourceManager().RegisterAppletResourceUserId(aruid, true);
    GetResourceManager().SetAppletResourceUserId(aruid);
#endif
    // リソーステーブルに Irsensor 使用中と設定
    GetResourceManager().SetAppletResourceEntry(aruid);

    for (auto i = 0; i < ::nn::hid::system::IrSensorSupportedNpadIdsCount; i++)
    {
        IrCameraHandle handle;
        MakeIrCameraHandle(&handle, ::nn::hid::system::IrSensorSupportedNpadIds[i]);
        NN_RESULT_DO(GetResourceManager().AssertValidIrCameraHandle(handle));
        // 対応する Xcd Driver をハンドルに設定し、初期化処理をしておく。
        NN_RESULT_DO(GetResourceManager().BindIrCameraXcdDriver(handle));
        NN_RESULT_DO(
            GetResourceManager().GetRequestHandlerTask(handle).Activate(aruid, functionLevel));
    }
    NN_RESULT_SUCCESS;
}

::nn::Result IrSensorServer::DeactivateIrsensor(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    for (auto i = 0; i < ::nn::hid::system::IrSensorSupportedNpadIdsCount; i++)
    {
        IrCameraHandle handle;
        MakeIrCameraHandle(&handle, ::nn::hid::system::IrSensorSupportedNpadIds[i]);
        NN_RESULT_DO(GetResourceManager().AssertValidIrCameraHandle(handle));
        NN_RESULT_DO(
            GetResourceManager().GetRequestHandlerTask(handle).Deactivate(aruid));
        // Xcd Driver を Handle と離して、終了処理をしておく。
        NN_RESULT_DO(GetResourceManager().UnbindIrCameraXcdDriver(handle));
    }
    // リソーステーブルの Irsensor 使用中フラグをクリア
    GetResourceManager().ResetAppletResourceEntry(aruid);

#if defined(NN_BUILD_TARGET_PLATFORM_OS_WIN)
    // Win の場合はリソーステーブルを破棄
    GetResourceManager().UnregisterAppletResourceUserId(aruid);
#endif

    NN_RESULT_SUCCESS;
}

::nn::Result IrSensorServer::GetIrsensorSharedMemoryHandle(
    ::nn::sf::Out<::nn::sf::NativeHandle> outValue,
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_UNUSED(aruid);

    auto sharedMemoryHandle = ::nn::os::NativeHandle();
    NN_RESULT_DO(
        GetResourceManager().GetSharedMemoryHandle(&sharedMemoryHandle));

    outValue.Set(::nn::sf::NativeHandle(sharedMemoryHandle, NeedsToBeManaged));
    NN_RESULT_SUCCESS;
}

//!< IR センサーのバージョンチェックをします。
::nn::Result IrSensorServer::CheckFirmwareVersion(
    ::nn::applet::AppletResourceUserId aruid,
    IrCameraHandle handle,
    const ::nn::irsensor::PackedMcuVersion& requiredVersion) NN_NOEXCEPT
{
    NN_UNUSED(aruid);

    NN_RESULT_DO(GetResourceManager().AssertValidIrCameraHandle(handle));

    NN_RESULT_THROW(
        GetResourceManager().GetRequestHandlerTask(handle)
                            .RequestFirmwareVersion(requiredVersion));
}

::nn::Result IrSensorServer::SetFunctionLevel(
    ::nn::applet::AppletResourceUserId aruid,
    IrCameraHandle handle,
    const ::nn::irsensor::PackedFunctionLevel& functionLevel) NN_NOEXCEPT
{
    NN_RESULT_DO(GetResourceManager().AssertValidIrCameraHandle(handle));

    NN_RESULT_THROW(
        GetResourceManager().GetRequestHandlerTask(handle)
                            .SetFunctionLevel(aruid, functionLevel));
}

::nn::Result IrSensorServer::StopImageProcessor(
    ::nn::applet::AppletResourceUserId aruid,
    IrCameraHandle handle) NN_NOEXCEPT
{
    NN_UNUSED(aruid);

    NN_RESULT_DO(GetResourceManager().AssertValidIrCameraHandle(handle));

    NN_RESULT_THROW(
        GetResourceManager().GetRequestHandlerTask(handle)
                            .StopImageProcessor(aruid));
}

::nn::Result IrSensorServer::StopImageProcessorAsync(
    ::nn::applet::AppletResourceUserId aruid,
    IrCameraHandle handle) NN_NOEXCEPT
{
    NN_UNUSED(aruid);

    NN_RESULT_DO(GetResourceManager().AssertValidIrCameraHandle(handle));

    NN_RESULT_THROW(
        GetResourceManager().GetRequestHandlerTask(handle)
                            .StopImageProcessorAsync(aruid));
}

::nn::Result IrSensorServer::SuspendImageProcessor(
    ::nn::applet::AppletResourceUserId aruid,
    IrCameraHandle handle) NN_NOEXCEPT
{
    NN_UNUSED(aruid);

    NN_RESULT_DO(GetResourceManager().AssertValidIrCameraHandle(handle));

    NN_RESULT_THROW(GetResourceManager().GetRequestHandlerTask(handle)
        .SuspendImageProcessor(aruid));
}

::nn::Result IrSensorServer::RunMomentProcessor(
    ::nn::applet::AppletResourceUserId aruid,
    IrCameraHandle handle,
    const PackedMomentProcessorConfig& config
    ) NN_NOEXCEPT
{
    NN_UNUSED(aruid);

    NN_RESULT_DO(GetResourceManager().AssertValidIrCameraHandle(handle));

    NN_RESULT_THROW(
        GetResourceManager().GetRequestHandlerTask(handle)
                            .RunMomentProcessor(aruid, config));
}

::nn::Result IrSensorServer::RunClusteringProcessor(
    ::nn::applet::AppletResourceUserId aruid,
    IrCameraHandle handle,
    const PackedClusteringProcessorConfig& config
    ) NN_NOEXCEPT
{
    NN_UNUSED(aruid);

    NN_RESULT_DO(GetResourceManager().AssertValidIrCameraHandle(handle));

    NN_RESULT_THROW(
        GetResourceManager().GetRequestHandlerTask(handle)
                            .RunClusteringProcessor(aruid, config));
}

::nn::Result IrSensorServer::RunImageTransferProcessor(
    ::nn::applet::AppletResourceUserId aruid,
    IrCameraHandle handle,
    const PackedImageTransferProcessorConfig& config,
    ::nn::sf::NativeHandle&& transferMemoryHandle,
    uint64_t size
    ) NN_NOEXCEPT
{
    NN_UNUSED(aruid);

    NN_RESULT_DO(GetResourceManager().AssertValidIrCameraHandle(handle));

    auto osHandle = transferMemoryHandle.GetOsHandle();
    auto managed = transferMemoryHandle.IsManaged();
    transferMemoryHandle.Detach();

    // 設定値を Ex 版にキャストする。
    PackedImageTransferProcessorExConfig exConfig = {};
    exConfig.irCameraConfig = config.irCameraConfig;
    exConfig.requiredVersion = config.requiredVersion;
    exConfig.origFormat = config.format;
    exConfig.trimmingFormat = config.format;
    exConfig.trimmingStartX = 0;
    exConfig.trimmingStartY = 0;
    exConfig.isExternalLightFilterEnebled = true;

    NN_RESULT_THROW(GetResourceManager().GetRequestHandlerTask(handle)
        .RunImageTransferProcessor(aruid, exConfig, ::std::move(osHandle), static_cast<size_t>(size), managed));
}

::nn::Result IrSensorServer::RunImageTransferExProcessor(
    ::nn::applet::AppletResourceUserId aruid,
    IrCameraHandle handle,
    const PackedImageTransferProcessorExConfig& config,
    ::nn::sf::NativeHandle&& transferMemoryHandle,
    uint64_t size
    ) NN_NOEXCEPT
{
    NN_UNUSED(aruid);

    NN_RESULT_DO(GetResourceManager().AssertValidIrCameraHandle(handle));

    auto osHandle = transferMemoryHandle.GetOsHandle();
    auto managed = transferMemoryHandle.IsManaged();
    transferMemoryHandle.Detach();

    NN_RESULT_THROW(GetResourceManager().GetRequestHandlerTask(handle)
        .RunImageTransferProcessor(aruid, config, ::std::move(osHandle), static_cast<size_t>(size), managed));
}

nn::Result IrSensorServer::GetImageTransferProcessorState(
    ::nn::applet::AppletResourceUserId aruid,
    nn::sf::Out<nn::irsensor::ImageTransferProcessorState> outState,
    const nn::sf::OutBuffer& outBuffer,
    nn::irsensor::IrCameraHandle handle) NN_NOEXCEPT
{
    NN_UNUSED(aruid);

    NN_RESULT_THROW_UNLESS(
        outState.GetPointer() != nullptr,
        ResultIrsensorNullPointer());

    NN_RESULT_DO(GetResourceManager().AssertValidIrCameraHandle(handle));

    ImageTransferProcessorState state;

    auto& task = GetResourceManager().GetRequestHandlerTask(handle);
    NN_RESULT_DO(task.GetImageTransferProcessorState(
        &state, outBuffer.GetPointerUnsafe(), aruid, outBuffer.GetSize()));

    outState.Set(state);

    NN_RESULT_SUCCESS;
}

::nn::Result IrSensorServer::RunPointingProcessor(
    ::nn::applet::AppletResourceUserId aruid,
    IrCameraHandle handle,
    const PackedPointingProcessorConfig& config
    ) NN_NOEXCEPT
{
    NN_UNUSED(aruid);

    NN_RESULT_DO(GetResourceManager().AssertValidIrCameraHandle(handle));

    NN_RESULT_THROW(
        GetResourceManager().GetRequestHandlerTask(handle)
                            .RunPointingProcessor(aruid, config));
}

::nn::Result IrSensorServer::RunTeraPluginProcessor(
    ::nn::applet::AppletResourceUserId aruid,
    IrCameraHandle handle,
    PackedTeraPluginProcessorConfig config) NN_NOEXCEPT
{
    NN_UNUSED(aruid);

    NN_RESULT_DO(GetResourceManager().AssertValidIrCameraHandle(handle));

    NN_RESULT_THROW(
        GetResourceManager().GetRequestHandlerTask(handle)
                            .RunTeraPluginProcessor(aruid, config));
}

::nn::Result IrSensorServer::RunIrLedProcessor(
    ::nn::applet::AppletResourceUserId aruid,
    IrCameraHandle handle,
    PackedIrLedProcessorConfig config) NN_NOEXCEPT
{
    NN_UNUSED(aruid);

    NN_RESULT_DO(GetResourceManager().AssertValidIrCameraHandle(handle));

    NN_RESULT_THROW(
        GetResourceManager().GetRequestHandlerTask(handle)
                            .RunIrLedProcessor(aruid, config));
}

::nn::Result CreateIrSensorServerProxy(::nn::sf::SharedPointer<IIrSensorServer>* outValue
                                  ) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(
        outValue != nullptr,
        ResultIrsensorNullPointer());

    *outValue = StaticObject<
        ::nn::sf::UnmanagedServiceObject<IIrSensorServer, IrSensorServer>
    >::Get().GetShared();
    NN_RESULT_SUCCESS;
}

}}} // namespace nn::irsensor::detail
