﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkText.h>
#include <nn/hid/system/hid_Result.h>
#include <nn/hid/system/hid_Irsensor.h>
#include <nn/irsensor/irsensor_Result.h>
#include <nn/irsensor/irsensor_ResultPrivate.h>
#include <nn/os/os_NativeHandle.h>
#include <nn/os/os_TimerEvent.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

#include "irsensor_DriverManager.h"
#include "irsensor_ResourceManager.h"

namespace nn { namespace irsensor { namespace detail {

namespace {

size_t GetImageSize(ImageTransferProcessorFormat format)
{
    size_t size;

    switch (static_cast<ImageTransferProcessorFormat>(format))
    {
    case ImageTransferProcessorFormat_320x240:
        size = ImageTransferProcessorImageSize320x240;
        break;
    case ImageTransferProcessorFormat_160x120:
        size = ImageTransferProcessorImageSize160x120;
        break;
    case ImageTransferProcessorFormat_80x60:
        size = ImageTransferProcessorImageSize80x60;
        break;
    case ImageTransferProcessorFormat_40x30:
        size = ImageTransferProcessorImageSize40x30;
        break;
    case ImageTransferProcessorFormat_20x15:
        size = ImageTransferProcessorImageSize20x15;
        break;
    default:
        NN_ABORT(NN_TEXT("不正な ImageTransferProcessorFormat です"));
    }

    return size;
}

}

const ::nn::TimeSpan DriverManager::SamplingInterval =
    ::nn::TimeSpan::FromMilliSeconds(15);

// Pointing の1度にサンプリングできる数の上限値
const int DriverManager::PointingSamplingStatesCountMax = 6;

DriverManager::DriverManager() NN_NOEXCEPT
    : m_Handle()
    , m_pDriver(nullptr)
    , m_pStatusManagerHolder(nullptr)
    , m_pAppletResourceMutex(nullptr)
    , m_pAppletResourceManager(nullptr)
    , m_ActivationCount()
    , m_pTimerEvent(nullptr)
    , m_IsDeviceSuspended(false)
{
    // 何もしない
}

DriverManager::~DriverManager() NN_NOEXCEPT
{
    // 何もしない
}

void DriverManager::SetTimerEvent(
    ::nn::os::TimerEventType* pTimerEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pTimerEvent);
    NN_SDK_REQUIRES(m_ActivationCount.IsZero());
    m_pTimerEvent = pTimerEvent;
}

void DriverManager::SetDriver(IIrSensorDriver* pDriver) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDriver);
    NN_SDK_REQUIRES(m_ActivationCount.IsZero());
    m_pDriver = pDriver;
}

void DriverManager::SetStatusManagerHolder(
    StatusManagerHolder* pStatusManagerHolder) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pStatusManagerHolder);
    NN_SDK_REQUIRES(m_ActivationCount.IsZero());
    m_pStatusManagerHolder = pStatusManagerHolder;
}

void DriverManager::SetIrCameraHandle(IrCameraHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_ActivationCount.IsZero());
    m_Handle = handle;
}

void DriverManager::BindIrSensorEvent() NN_NOEXCEPT
{
    // デバイス状態通知イベントを hid にバインド
    auto playerNumber = GetIrCameraHandlePlayerNumber(m_Handle);
    ::nn::hid::system::BindIrSensorEvent(
        ::nn::hid::system::IrSensorSupportedNpadIds[playerNumber],
        &m_DeviceStatusEvent,
        ::nn::os::EventClearMode_ManualClear);
}

void DriverManager::SetAppletResourceManager(
    ::nn::os::Mutex* pAppletResourceMutex,
    AppletResourceManager* pAppletResourceManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_ActivationCount.IsZero());
    m_pAppletResourceMutex = pAppletResourceMutex;
    m_pAppletResourceManager = pAppletResourceManager;
}

bool DriverManager::IsDriverAvailable() NN_NOEXCEPT
{
    if (m_pDriver != nullptr)
    {
        return true;
    }
    else
    {
        return false;
    }
}

::nn::Result DriverManager::Activate() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pDriver);
    NN_SDK_REQUIRES_NOT_NULL(m_pStatusManagerHolder);

    if (m_ActivationCount.IsZero())
    {
        ActivateTimerEvent();
    }

    ++m_ActivationCount;
    m_IsDeviceSuspended = false;

    NN_RESULT_SUCCESS;
}

::nn::Result DriverManager::Deactivate() NN_NOEXCEPT
{
    --m_ActivationCount;

    if (m_ActivationCount.IsZero())
    {
        DeactivateTimerEvent();

        m_pStatusManagerHolder = nullptr;
        m_pDriver = nullptr;
    }
    m_IsDeviceSuspended = false;

    NN_RESULT_SUCCESS;
}

::nn::Result DriverManager::CheckFirmwareVersion(
    const PackedMcuVersion& requiredVersion) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pDriver);

    PackedFunctionLevel functionLevel = {};
    {
        ::std::lock_guard<decltype(*m_pAppletResourceMutex)> locker(*m_pAppletResourceMutex);
        // アプレットが登録、初期化されていない、または、 FG でない場合は、エラーを返す。
        NN_RESULT_THROW_UNLESS(m_pAppletResourceManager->IsAppletResourceAvailable(),
            ResultIrsensorAppletResourceNotAvailable());
        // FunctionLevel を取得
        auto latestAruid = m_pAppletResourceManager->GetLatestActivatedAruid();
        m_pAppletResourceManager->GetIrSensorFunctionLevel(latestAruid, &functionLevel, m_Handle);
    }

    NN_RESULT_DO(m_pDriver->CheckVersion(requiredVersion, functionLevel));
    NN_RESULT_SUCCESS;
}

::nn::Result DriverManager::RequestFirmwareVersion() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pDriver);

    // IR が搭載されているかに関わらずハンドルを取得する
    // 非搭載だった場合は Unsupported のエラーを返す
    auto playerNumber = GetIrCameraHandlePlayerNumber(m_Handle);
    ::nn::xcd::DeviceHandle handle;
    NN_RESULT_TRY(::nn::hid::system::GetXcdHandleForNpadWithIrSensor(
        &handle, ::nn::hid::system::IrSensorSupportedNpadIds[playerNumber]))
        NN_RESULT_CATCH(::nn::hid::system::ResultNoIrSensorDeviceFoundOnNpad)
        {
            NN_RESULT_THROW(::nn::irsensor::ResultIrsensorUnsupported());
        }
    NN_RESULT_END_TRY
    NN_RESULT_DO(m_pDriver->SetXcdDeviceHandle(handle));

    NN_RESULT_DO(m_pDriver->RequestVersion());
    NN_RESULT_SUCCESS;
}

::nn::Result DriverManager::SetFunctionLevel(
    const ::nn::applet::AppletResourceUserId& aruid,
    const PackedFunctionLevel& functionLevel) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_MINMAX(functionLevel.level,
        static_cast<uint8_t>(nn::irsensor::IrSensorFunctionLevel_0),
        static_cast<uint8_t>(nn::irsensor::IrSensorFunctionLevel_Latest));

    ::std::lock_guard<decltype(*m_pAppletResourceMutex)
    > locker(*m_pAppletResourceMutex);
    m_pAppletResourceManager->SetIrSensorFunctionLevel(aruid, functionLevel, m_Handle);
    NN_RESULT_SUCCESS;
}

::nn::Result DriverManager::SleepImageProcessor() NN_NOEXCEPT
{
    // Hid に使用しないと通知しておく。
    NN_RESULT_DO(DeactivateIrsensorForHid());

    // Aruid に紐づくデータはそのまま、デバイスだけ止める。
    NN_SDK_REQUIRES_NOT_NULL(m_pDriver);
    NN_RESULT_DO(m_pDriver->StopImageProcessor());
    NN_RESULT_SUCCESS;
}

::nn::Result DriverManager::StopImageProcessorWithCompatibility() NN_NOEXCEPT
{
    // 互換性を維持した版の Stop 関数
    // Addon 3.x系以前のSDKではこちらが呼び出される。
    NN_SDK_REQUIRES_NOT_NULL(m_pDriver);

    IrSensorMode mode = IrSensorMode::None;
    {
        ::std::lock_guard<decltype(*m_pAppletResourceMutex)
        > locker(*m_pAppletResourceMutex);
        m_pAppletResourceManager->GetIrSensorMode(&mode, m_pAppletResourceManager->GetLatestActivatedAruid(), m_Handle);
    }

    if (mode != IrSensorMode::None)
    {
        NN_RESULT_DO(DeactivateIrsensorForHid());

        NN_RESULT_DO(m_pDriver->StopImageProcessor());
        auto pManager = m_pStatusManagerHolder->GetStatusManager();
        NN_SDK_ASSERT_NOT_NULL(pManager);
        pManager->FreeLifo(m_Handle);
        {
            ::std::lock_guard<decltype(*m_pAppletResourceMutex)
            > locker(*m_pAppletResourceMutex);
            m_pAppletResourceManager->SetIrSensorMode(
                m_pAppletResourceManager->GetLatestActivatedAruid(), IrSensorMode::None, m_Handle);
        }
    }

    NN_RESULT_SUCCESS;
}

::nn::Result DriverManager::StopImageProcessor() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pDriver);

    IrSensorMode mode = IrSensorMode::None;
    {
        ::std::lock_guard<decltype(*m_pAppletResourceMutex)
        > locker(*m_pAppletResourceMutex);
        m_pAppletResourceManager->GetIrSensorMode(&mode, m_pAppletResourceManager->GetLatestActivatedAruid(), m_Handle);
    }

    // Suspend 直後に到着した場合も、再度終了処理を行う
    if (mode != IrSensorMode::None || m_IsDeviceSuspended)
    {
        NN_RESULT_DO(DeactivateIrsensorForHid());

        NN_RESULT_DO(m_pDriver->StopImageProcessor());
        auto pManager = m_pStatusManagerHolder->GetStatusManager();
        NN_SDK_ASSERT_NOT_NULL(pManager);
        pManager->FreeLifo(m_Handle);
        {
            ::std::lock_guard<decltype(*m_pAppletResourceMutex)
            > locker(*m_pAppletResourceMutex);
            m_pAppletResourceManager->SetIrSensorMode(
                m_pAppletResourceManager->GetLatestActivatedAruid(), IrSensorMode::None, m_Handle);
        }
    }

    NN_RESULT_SUCCESS;
}

::nn::Result DriverManager::SuspendImageProcessor() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pDriver);

    IrSensorMode mode = IrSensorMode::None;
    {
        ::std::lock_guard<decltype(*m_pAppletResourceMutex)
        > locker(*m_pAppletResourceMutex);
        m_pAppletResourceManager->GetIrSensorMode(&mode, m_pAppletResourceManager->GetLatestActivatedAruid(), m_Handle);
    }

    if (mode != IrSensorMode::None)
    {
        NN_RESULT_DO(m_pDriver->SuspendImageProcessor());
        auto pManager = m_pStatusManagerHolder->GetStatusManager();
        NN_SDK_ASSERT_NOT_NULL(pManager);
        pManager->FreeLifo(m_Handle);
        {
            ::std::lock_guard<decltype(*m_pAppletResourceMutex)
            > locker(*m_pAppletResourceMutex);
            m_pAppletResourceManager->SetIrSensorMode(
                m_pAppletResourceManager->GetLatestActivatedAruid(), IrSensorMode::None, m_Handle);
        }
        m_IsDeviceSuspended = true;
    }

    NN_RESULT_SUCCESS;
}

::nn::Result DriverManager::StopSampling(
    const ::nn::applet::AppletResourceUserId& aruid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pDriver);

    IrSensorMode mode = IrSensorMode::None;
    {
        ::std::lock_guard<decltype(*m_pAppletResourceMutex)
        > locker(*m_pAppletResourceMutex);
        m_pAppletResourceManager->GetIrSensorMode(&mode, aruid, m_Handle);
    }

    if (mode != IrSensorMode::None)
    {
        NN_RESULT_DO(m_pDriver->StopSampling());
    }

    NN_RESULT_SUCCESS;
}

::nn::Result DriverManager::PrepareMomentProcessor() NN_NOEXCEPT
{
    auto pManager = m_pStatusManagerHolder->GetStatusManager();
    NN_SDK_ASSERT_NOT_NULL(pManager);
    pManager->FreeLifo(m_Handle);
    pManager->AllocateMomentLifo(m_Handle);
    m_IsDeviceSuspended = false;

    NN_RESULT_SUCCESS;
}

::nn::Result DriverManager::PrepareClusteringProcessor() NN_NOEXCEPT
{
    auto pManager = m_pStatusManagerHolder->GetStatusManager();
    NN_SDK_ASSERT_NOT_NULL(pManager);
    pManager->FreeLifo(m_Handle);
    pManager->AllocateClusteringLifo(m_Handle);
    m_IsDeviceSuspended = false;

    NN_RESULT_SUCCESS;
}

::nn::Result DriverManager::PrepareImageTransferProcessor() NN_NOEXCEPT
{
    auto pManager = m_pStatusManagerHolder->GetStatusManager();
    NN_SDK_ASSERT_NOT_NULL(pManager);
    pManager->FreeLifo(m_Handle);
    {
        // ImageTransfer の画像を無効設定
        ::std::lock_guard<decltype(*m_pAppletResourceMutex)
        > locker(*m_pAppletResourceMutex);
        m_pAppletResourceManager->SetImageValidity(m_pAppletResourceManager->GetLatestActivatedAruid(), m_Handle, false);
    }
    m_IsDeviceSuspended = false;
    NN_RESULT_SUCCESS;
}

::nn::Result DriverManager::PreparePointingProcessor() NN_NOEXCEPT
{
    auto pManager = m_pStatusManagerHolder->GetStatusManager();
    NN_SDK_ASSERT_NOT_NULL(pManager);
    pManager->FreeLifo(m_Handle);
    pManager->AllocatePointingLifo(m_Handle);
    m_IsDeviceSuspended = false;

    NN_RESULT_SUCCESS;
}

::nn::Result DriverManager::PrepareTeraPluginProcessor() NN_NOEXCEPT
{
    auto pManager = m_pStatusManagerHolder->GetStatusManager();
    NN_SDK_ASSERT_NOT_NULL(pManager);
    pManager->FreeLifo(m_Handle);
    pManager->AllocateTeraPluginLifo(m_Handle);
    m_IsDeviceSuspended = false;

    NN_RESULT_SUCCESS;
}

::nn::Result DriverManager::PrepareIrLedProcessor() NN_NOEXCEPT
{
    NN_RESULT_DO(PrepareMomentProcessor());
    NN_RESULT_SUCCESS;
}

::nn::Result DriverManager::RunImageProcessor(
    const PackedMomentProcessorConfig& config
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pDriver);

    PackedFunctionLevel functionLevel = {};
    {
        ::std::lock_guard<decltype(*m_pAppletResourceMutex)> locker(*m_pAppletResourceMutex);
        // アプレットが登録、初期化されていない、または、 FG でない場合は、サンプリングを停止しておく。
        NN_RESULT_THROW_UNLESS(m_pAppletResourceManager->IsAppletResourceAvailable(),
            ResultIrsensorAppletResourceNotAvailable());
        // FunctionLevel を取得
        auto latestAruid = m_pAppletResourceManager->GetLatestActivatedAruid();
        m_pAppletResourceManager->GetIrSensorFunctionLevel(latestAruid, &functionLevel, m_Handle);
    }

    NN_RESULT_DO(ActivateIrsensorForHid());

    NN_RESULT_DO(m_pDriver->RunImageProcessor(config, functionLevel));
    {
        ::std::lock_guard<decltype(*m_pAppletResourceMutex)
        > locker(*m_pAppletResourceMutex);
        m_pAppletResourceManager->SetIrSensorMode(
            m_pAppletResourceManager->GetLatestActivatedAruid(), IrSensorMode::Moment, m_Handle);
    }

    NN_RESULT_SUCCESS;
}

::nn::Result DriverManager::ResumeImageProcessor(
    const PackedMomentProcessorConfig& config
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pDriver);

    PackedFunctionLevel functionLevel = {};
    {
        ::std::lock_guard<decltype(*m_pAppletResourceMutex)> locker(*m_pAppletResourceMutex);
        // アプレットが登録、初期化されていない、または、 FG でない場合は、サンプリングを停止しておく。
        NN_RESULT_THROW_UNLESS(m_pAppletResourceManager->IsAppletResourceAvailable(),
            ResultIrsensorAppletResourceNotAvailable());
        // FunctionLevel を取得
        auto latestAruid = m_pAppletResourceManager->GetLatestActivatedAruid();
        m_pAppletResourceManager->GetIrSensorFunctionLevel(latestAruid, &functionLevel, m_Handle);
    }

    NN_RESULT_DO(m_pDriver->ResumeImageProcessor(config, functionLevel));
    {
        ::std::lock_guard<decltype(*m_pAppletResourceMutex)
        > locker(*m_pAppletResourceMutex);
        m_pAppletResourceManager->SetIrSensorMode(
            m_pAppletResourceManager->GetLatestActivatedAruid(), IrSensorMode::Moment, m_Handle);
    }

    NN_RESULT_SUCCESS;
}

::nn::Result DriverManager::RunImageProcessor(
    const PackedClusteringProcessorConfig& config
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pDriver);

    PackedFunctionLevel functionLevel = {};
    {
        ::std::lock_guard<decltype(*m_pAppletResourceMutex)> locker(*m_pAppletResourceMutex);
        // アプレットが登録、初期化されていない、または、 FG でない場合は、サンプリングを停止しておく。
        NN_RESULT_THROW_UNLESS(m_pAppletResourceManager->IsAppletResourceAvailable(),
            ResultIrsensorAppletResourceNotAvailable());
        // FunctionLevel を取得
        auto latestAruid = m_pAppletResourceManager->GetLatestActivatedAruid();
        m_pAppletResourceManager->GetIrSensorFunctionLevel(latestAruid, &functionLevel, m_Handle);
    }

    NN_RESULT_DO(ActivateIrsensorForHid());

    NN_RESULT_DO(m_pDriver->RunImageProcessor(config, functionLevel));
    {
        ::std::lock_guard<decltype(*m_pAppletResourceMutex)
        > locker(*m_pAppletResourceMutex);
        m_pAppletResourceManager->SetIrSensorMode(
            m_pAppletResourceManager->GetLatestActivatedAruid(), IrSensorMode::Clustering, m_Handle);
    }

    NN_RESULT_SUCCESS;
}

::nn::Result DriverManager::ResumeImageProcessor(
    const PackedClusteringProcessorConfig& config
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pDriver);

    PackedFunctionLevel functionLevel = {};
    {
        ::std::lock_guard<decltype(*m_pAppletResourceMutex)> locker(*m_pAppletResourceMutex);
        // アプレットが登録、初期化されていない、または、 FG でない場合は、サンプリングを停止しておく。
        NN_RESULT_THROW_UNLESS(m_pAppletResourceManager->IsAppletResourceAvailable(),
            ResultIrsensorAppletResourceNotAvailable());
        // FunctionLevel を取得
        auto latestAruid = m_pAppletResourceManager->GetLatestActivatedAruid();
        m_pAppletResourceManager->GetIrSensorFunctionLevel(latestAruid, &functionLevel, m_Handle);
    }

    NN_RESULT_DO(m_pDriver->ResumeImageProcessor(config, functionLevel));
    {
        ::std::lock_guard<decltype(*m_pAppletResourceMutex)
        > locker(*m_pAppletResourceMutex);
        m_pAppletResourceManager->SetIrSensorMode(
            m_pAppletResourceManager->GetLatestActivatedAruid(), IrSensorMode::Clustering, m_Handle);
    }

    NN_RESULT_SUCCESS;
}

::nn::Result DriverManager::RunImageProcessor(
    const PackedImageTransferProcessorExConfig& config,
    void* buffer
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pDriver);

    PackedFunctionLevel functionLevel = {};
    {
        ::std::lock_guard<decltype(*m_pAppletResourceMutex)> locker(*m_pAppletResourceMutex);
        // アプレットが登録、初期化されていない、または、 FG でない場合は、サンプリングを停止しておく。
        NN_RESULT_THROW_UNLESS(m_pAppletResourceManager->IsAppletResourceAvailable(),
            ResultIrsensorAppletResourceNotAvailable());
        // FunctionLevel を取得
        auto latestAruid = m_pAppletResourceManager->GetLatestActivatedAruid();
        m_pAppletResourceManager->GetIrSensorFunctionLevel(latestAruid, &functionLevel, m_Handle);
    }

    NN_RESULT_DO(ActivateIrsensorForHid());

    NN_RESULT_DO(m_pDriver->RunImageProcessor(config, functionLevel, buffer));
    {
        ::std::lock_guard<decltype(*m_pAppletResourceMutex)
        > locker(*m_pAppletResourceMutex);
        m_pAppletResourceManager->SetIrSensorMode(
            m_pAppletResourceManager->GetLatestActivatedAruid(), IrSensorMode::ImageTransfer, m_Handle);
    }

    // バッファの末尾は結果取得に利用する
    auto size = GetImageSize(
        static_cast<ImageTransferProcessorFormat>(config.trimmingFormat));

    {
        ::std::lock_guard<decltype(*m_pAppletResourceMutex)
        > locker(*m_pAppletResourceMutex);
        // ImageTranfser のバッファアドレスを保存
        m_pAppletResourceManager->SetImageTransferBuffer(m_Handle, reinterpret_cast<uint8_t*>(buffer) + size);
    }

    NN_RESULT_SUCCESS;
}

::nn::Result DriverManager::ResumeImageProcessor(
    const PackedImageTransferProcessorExConfig& config,
    void* buffer
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pDriver);

    PackedFunctionLevel functionLevel = {};
    {
        ::std::lock_guard<decltype(*m_pAppletResourceMutex)> locker(*m_pAppletResourceMutex);
        // アプレットが登録、初期化されていない、または、 FG でない場合は、サンプリングを停止しておく。
        NN_RESULT_THROW_UNLESS(m_pAppletResourceManager->IsAppletResourceAvailable(),
            ResultIrsensorAppletResourceNotAvailable());
        // FunctionLevel を取得
        auto latestAruid = m_pAppletResourceManager->GetLatestActivatedAruid();
        m_pAppletResourceManager->GetIrSensorFunctionLevel(latestAruid, &functionLevel, m_Handle);
    }

    NN_RESULT_DO(m_pDriver->ResumeImageProcessor(config, functionLevel, buffer));
    {
        ::std::lock_guard<decltype(*m_pAppletResourceMutex)
        > locker(*m_pAppletResourceMutex);
        m_pAppletResourceManager->SetIrSensorMode(
            m_pAppletResourceManager->GetLatestActivatedAruid(), IrSensorMode::ImageTransfer, m_Handle);
    }

    // バッファの末尾は結果取得に利用する
    auto size = GetImageSize(
        static_cast<ImageTransferProcessorFormat>(config.trimmingFormat));

    {
        ::std::lock_guard<decltype(*m_pAppletResourceMutex)
        > locker(*m_pAppletResourceMutex);
        // ImageTranfser のバッファアドレスを保存
        m_pAppletResourceManager->SetImageTransferBuffer(m_Handle, reinterpret_cast<uint8_t*>(buffer) + size);
    }

    NN_RESULT_SUCCESS;
}

::nn::Result DriverManager::GetImageProcessorState(
    ImageTransferProcessorState* pOutState,
    void* pOutImage,
    size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutState);
    NN_SDK_REQUIRES_NOT_NULL(pOutImage);

    IrSensorMode mode = IrSensorMode::None;
    {
        ::std::lock_guard<decltype(*m_pAppletResourceMutex)
        > locker(*m_pAppletResourceMutex);
        // アプレットが登録、初期化されていない、または、 FG でない場合は、エラーを返す。
        NN_RESULT_THROW_UNLESS(m_pAppletResourceManager->IsAppletResourceAvailable(),
            ResultIrsensorAppletResourceNotAvailable());
        m_pAppletResourceManager->GetIrSensorMode(&mode, m_pAppletResourceManager->GetLatestActivatedAruid(), m_Handle);
    }
    NN_RESULT_THROW_UNLESS(
        mode == IrSensorMode::ImageTransfer,
        ResultIrsensorNotReady());

    // 最新のdeviceResource を取得
    IrSensorDeviceResource deviceResource;
    {
        ::std::lock_guard<decltype(*m_pAppletResourceMutex)
        > locker(*m_pAppletResourceMutex);

        m_pAppletResourceManager->GetIrSensorDeviceResource(
            &deviceResource, m_pAppletResourceManager->GetLatestActivatedAruid(), m_Handle);
    }

    auto expectedSize = GetImageSize(
        static_cast<ImageTransferProcessorFormat>(deviceResource.config.imageTransfer.trimmingFormat));
    NN_RESULT_THROW_UNLESS(
        size >= expectedSize,
        ResultIrsensorBufferSizeTooSmall());

    NN_RESULT_THROW_UNLESS(
        deviceResource.imageTransferStatus.isImageValid,
        ResultIrsensorNotReady());

    ::std::memcpy(pOutImage, deviceResource.imageTransferStatus.imageBufferAddress, expectedSize);
    *pOutState = deviceResource.imageTransferStatus.imageTransferState;

    NN_RESULT_SUCCESS;
}

::nn::Result DriverManager::RunImageProcessor(
    const PackedPointingProcessorConfig& config
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pDriver);

    PackedFunctionLevel functionLevel = {};
    {
        ::std::lock_guard<decltype(*m_pAppletResourceMutex)> locker(*m_pAppletResourceMutex);
        // アプレットが登録、初期化されていない、または、 FG でない場合は、サンプリングを停止しておく。
        NN_RESULT_THROW_UNLESS(m_pAppletResourceManager->IsAppletResourceAvailable(),
            ResultIrsensorAppletResourceNotAvailable());
        // FunctionLevel を取得
        auto latestAruid = m_pAppletResourceManager->GetLatestActivatedAruid();
        m_pAppletResourceManager->GetIrSensorFunctionLevel(latestAruid, &functionLevel, m_Handle);
    }

    NN_RESULT_DO(ActivateIrsensorForHid());

    NN_RESULT_DO(m_pDriver->RunImageProcessor(config, functionLevel));
    {
        ::std::lock_guard<decltype(*m_pAppletResourceMutex)
        > locker(*m_pAppletResourceMutex);
        m_pAppletResourceManager->SetIrSensorMode(
            m_pAppletResourceManager->GetLatestActivatedAruid(), IrSensorMode::Pointing, m_Handle);
    }

    NN_RESULT_SUCCESS;
}

::nn::Result DriverManager::ResumeImageProcessor(
    const PackedPointingProcessorConfig& config
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pDriver);

    PackedFunctionLevel functionLevel = {};
    {
        ::std::lock_guard<decltype(*m_pAppletResourceMutex)> locker(*m_pAppletResourceMutex);
        // アプレットが登録、初期化されていない、または、 FG でない場合は、サンプリングを停止しておく。
        NN_RESULT_THROW_UNLESS(m_pAppletResourceManager->IsAppletResourceAvailable(),
            ResultIrsensorAppletResourceNotAvailable());
        // FunctionLevel を取得
        auto latestAruid = m_pAppletResourceManager->GetLatestActivatedAruid();
        m_pAppletResourceManager->GetIrSensorFunctionLevel(latestAruid, &functionLevel, m_Handle);
    }

    NN_RESULT_DO(m_pDriver->ResumeImageProcessor(config, functionLevel));
    {
        ::std::lock_guard<decltype(*m_pAppletResourceMutex)
        > locker(*m_pAppletResourceMutex);
        m_pAppletResourceManager->SetIrSensorMode(
            m_pAppletResourceManager->GetLatestActivatedAruid(), IrSensorMode::Pointing, m_Handle);
    }

    NN_RESULT_SUCCESS;
}

::nn::Result DriverManager::RunImageProcessor(
    const PackedTeraPluginProcessorConfig& config
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pDriver);

    PackedFunctionLevel functionLevel = {};
    {
        ::std::lock_guard<decltype(*m_pAppletResourceMutex)> locker(*m_pAppletResourceMutex);
        // アプレットが登録、初期化されていない、または、 FG でない場合は、サンプリングを停止しておく。
        NN_RESULT_THROW_UNLESS(m_pAppletResourceManager->IsAppletResourceAvailable(),
            ResultIrsensorAppletResourceNotAvailable());
        // FunctionLevel を取得
        auto latestAruid = m_pAppletResourceManager->GetLatestActivatedAruid();
        m_pAppletResourceManager->GetIrSensorFunctionLevel(latestAruid, &functionLevel, m_Handle);
    }

    NN_RESULT_DO(ActivateIrsensorForHid());

    NN_RESULT_DO(m_pDriver->RunImageProcessor(config, functionLevel));
    {
        ::std::lock_guard<decltype(*m_pAppletResourceMutex)
        > locker(*m_pAppletResourceMutex);
        m_pAppletResourceManager->SetIrSensorMode(
            m_pAppletResourceManager->GetLatestActivatedAruid(), IrSensorMode::TeraPlugin, m_Handle);
    }

    NN_RESULT_SUCCESS;
}

::nn::Result DriverManager::ResumeImageProcessor(
    const PackedTeraPluginProcessorConfig& config
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pDriver);

    PackedFunctionLevel functionLevel = {};
    {
        ::std::lock_guard<decltype(*m_pAppletResourceMutex)> locker(*m_pAppletResourceMutex);
        // アプレットが登録、初期化されていない、または、 FG でない場合は、サンプリングを停止しておく。
        NN_RESULT_THROW_UNLESS(m_pAppletResourceManager->IsAppletResourceAvailable(),
            ResultIrsensorAppletResourceNotAvailable());
        // FunctionLevel を取得
        auto latestAruid = m_pAppletResourceManager->GetLatestActivatedAruid();
        m_pAppletResourceManager->GetIrSensorFunctionLevel(latestAruid, &functionLevel, m_Handle);
    }

    NN_RESULT_DO(m_pDriver->ResumeImageProcessor(config, functionLevel));
    {
        ::std::lock_guard<decltype(*m_pAppletResourceMutex)
        > locker(*m_pAppletResourceMutex);
        m_pAppletResourceManager->SetIrSensorMode(
            m_pAppletResourceManager->GetLatestActivatedAruid(), IrSensorMode::TeraPlugin, m_Handle);
    }

    NN_RESULT_SUCCESS;
}

::nn::Result DriverManager::RunImageProcessor(
    const PackedIrLedProcessorConfig& config
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pDriver);

    PackedFunctionLevel functionLevel = {};
    {
        ::std::lock_guard<decltype(*m_pAppletResourceMutex)> locker(*m_pAppletResourceMutex);
        // アプレットが登録、初期化されていない、または、 FG でない場合は、サンプリングを停止しておく。
        NN_RESULT_THROW_UNLESS(m_pAppletResourceManager->IsAppletResourceAvailable(),
            ResultIrsensorAppletResourceNotAvailable());
        // FunctionLevel を取得
        auto latestAruid = m_pAppletResourceManager->GetLatestActivatedAruid();
        m_pAppletResourceManager->GetIrSensorFunctionLevel(latestAruid, &functionLevel, m_Handle);
    }

    NN_RESULT_DO(ActivateIrsensorForHid());

    NN_RESULT_DO(m_pDriver->RunImageProcessor(config, functionLevel));
    {
        ::std::lock_guard<decltype(*m_pAppletResourceMutex)
        > locker(*m_pAppletResourceMutex);
        m_pAppletResourceManager->SetIrSensorMode(
            m_pAppletResourceManager->GetLatestActivatedAruid(), IrSensorMode::IrLed, m_Handle);
    }

    NN_RESULT_SUCCESS;
}

::nn::Result DriverManager::ResumeImageProcessor(
    const PackedIrLedProcessorConfig& config
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pDriver);

    PackedFunctionLevel functionLevel = {};
    {
        ::std::lock_guard<decltype(*m_pAppletResourceMutex)> locker(*m_pAppletResourceMutex);
        // アプレットが登録、初期化されていない、または、 FG でない場合は、サンプリングを停止しておく。
        NN_RESULT_THROW_UNLESS(m_pAppletResourceManager->IsAppletResourceAvailable(),
            ResultIrsensorAppletResourceNotAvailable());
        // FunctionLevel を取得
        auto latestAruid = m_pAppletResourceManager->GetLatestActivatedAruid();
        m_pAppletResourceManager->GetIrSensorFunctionLevel(latestAruid, &functionLevel, m_Handle);
    }

    NN_RESULT_DO(m_pDriver->ResumeImageProcessor(config, functionLevel));
    {
        ::std::lock_guard<decltype(*m_pAppletResourceMutex)
        > locker(*m_pAppletResourceMutex);
        m_pAppletResourceManager->SetIrSensorMode(
            m_pAppletResourceManager->GetLatestActivatedAruid(), IrSensorMode::IrLed, m_Handle);
    }

    NN_RESULT_SUCCESS;
}

::nn::Result DriverManager::GetLifoHandle(
    ::nn::os::NativeHandle* outValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outValue);

    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsZero(),
        ResultIrsensorActivationUpperLimitOver());
    *outValue = m_pStatusManagerHolder->GetSharedMemoryHandle();

    NN_RESULT_SUCCESS;
}

void DriverManager::UpdateIrCameraStatus() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pDriver);

    auto playerNumber = GetIrCameraHandlePlayerNumber(m_Handle);
    ::nn::hid::system::IrSensorState irSensorState;
    irSensorState = ::nn::hid::system::GetIrSensorState(
        ::nn::hid::system::IrSensorSupportedNpadIds[playerNumber]);

    IrCameraStatus status;
    switch (irSensorState)
    {
    case ::nn::hid::system::IrSensorState_NotConnected:
        {
            status = IrCameraStatus_Unconnected;
        }
        break;
    case ::nn::hid::system::IrSensorState_NotSupported:
        {
            status = IrCameraStatus_Unsupported;
        }
        break;
    case ::nn::hid::system::IrSensorState_Deactivated:
    case ::nn::hid::system::IrSensorState_Activated:
        {
            status = IrCameraStatus_Available;
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    // ドライバ側に通知
    m_pDriver->SetIrCameraStatus(status);

    // 共有メモリを更新
    auto pManager = m_pStatusManagerHolder->GetStatusManager();
    NN_SDK_ASSERT_NOT_NULL(pManager);
    pManager->SetIrCameraStatus(m_Handle, status);
}

void DriverManager::SetIrCameraInternalStatus(IrCameraInternalStatus status) NN_NOEXCEPT
{
    auto pManager = m_pStatusManagerHolder->GetStatusManager();
    NN_SDK_ASSERT_NOT_NULL(pManager);

    // FunctionLevel ごとに、共有メモリに埋める InternalStatus がかわるため、
    // 適切に互換性がとれているかをチェックして、必要に応じてキャストします。
    PackedFunctionLevel functionLevel = {};
    {
        ::std::lock_guard<decltype(*m_pAppletResourceMutex)> locker(*m_pAppletResourceMutex);
        // FunctionLevel を取得
        auto latestAruid = m_pAppletResourceManager->GetLatestActivatedAruid();
        m_pAppletResourceManager->GetIrSensorFunctionLevel(latestAruid, &functionLevel, m_Handle);
    }
    if (functionLevel.level <= IrSensorFunctionLevel_0)
    {
        if (status == nn::irsensor::IrCameraInternalStatus_SuccessfulTerminated
            || status == nn::irsensor::IrCameraInternalStatus_Setting)
        {
            // 互換性維持のためキャストする
            status = nn::irsensor::IrCameraInternalStatus_Available;
        }
        // FunctionLevel 0 の の場合は SuccessfulTerminated より小さい値の設定のみ可能
        NN_SDK_ASSERT_LESS(status, nn::irsensor::IrCameraInternalStatus_Setting);
    }
    pManager->SetIrCameraInternalStatus(m_Handle, status);
}

void DriverManager::UpdateIrSensorAppletStatus() NN_NOEXCEPT
{
    auto pManager = m_pStatusManagerHolder->GetStatusManager();
    NN_SDK_ASSERT_NOT_NULL(pManager);

    {
        ::std::lock_guard<decltype(*m_pAppletResourceMutex)
        > locker(*m_pAppletResourceMutex);

        for (const AppletResourceEntry& entry :
            m_pAppletResourceManager->GetAppletResourceEntries())
        {
            // IR センサーを利用しているアプレットのみ共有メモリのテーブルを更新
            if (entry.flags.Test<AppletResourceFlag::IsAvailable>())
            {
                // Aruid に対応するエントリを取得する
                AppletStatusFlagSet statusFlag = AppletStatusFlagSet();
                if (!pManager->GetIrSensorAruidStatus(&statusFlag, entry.aruid))
                {
                    // テーブルに対応するエントリが存在しなければ生成する
                    pManager->RegisterIrSensorAruid(entry.aruid);
                }
                // 共有メモリのテーブルをチェックして、情報を更新後、書き戻し
                bool isForeground = entry.flags.Test<AppletResourceFlag::IsForeground>();
                statusFlag.Set<AppletStatusFlag::IsForeground>(isForeground);
                pManager->SetIrSensorAruidStatus(entry.aruid, statusFlag);
            }
            else
            {
                // IR センサーを利用していないアプレットのテーブルは存在すれば破棄する
                AppletStatusFlagSet statusFlag;
                if (pManager->GetIrSensorAruidStatus(&statusFlag, entry.aruid))
                {
                    pManager->UnregisterIrSensorAruid(entry.aruid);
                }
            }
        }
    }
}

::nn::Result DriverManager::Sample() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pDriver);

    IrSensorMode mode = IrSensorMode::None;
    {
        ::std::lock_guard<decltype(*m_pAppletResourceMutex)
        > locker(*m_pAppletResourceMutex);
        // アプレットが登録、初期化されていない、または、 FG でない場合は、サンプリングを停止しておく。
        NN_RESULT_THROW_UNLESS(m_pAppletResourceManager->IsAppletResourceAvailable(),
            ResultIrsensorAppletResourceNotAvailable());
        m_pAppletResourceManager->GetIrSensorMode(&mode, m_pAppletResourceManager->GetLatestActivatedAruid(), m_Handle);
    }
    if ((mode == IrSensorMode::Moment)
        || (mode == IrSensorMode::IrLed))
    {
        MomentProcessorState state;
        int count;
        auto result = m_pDriver->GetImageProcessorStates(&state, &count, 1);
        if (result.IsSuccess())
        {
            auto pManager = m_pStatusManagerHolder->GetStatusManager();
            NN_SDK_ASSERT_NOT_NULL(pManager);
            auto pLifo = pManager->GetMomentLifo(m_Handle);
            NN_SDK_ASSERT_NOT_NULL(pLifo);
            pLifo->Append(state);
        }
        return result;
    }
    if (mode == IrSensorMode::Clustering)
    {
        ClusteringProcessorState state;
        int count;
        auto result = m_pDriver->GetImageProcessorStates(&state, &count, 1);
        if (result.IsSuccess())
        {
            auto pManager = m_pStatusManagerHolder->GetStatusManager();
            NN_SDK_ASSERT_NOT_NULL(pManager);
            auto pLifo = pManager->GetClusteringLifo(m_Handle);
            NN_SDK_ASSERT_NOT_NULL(pLifo);
            pLifo->Append(state);
        }
        return result;
    }
    if (mode == IrSensorMode::ImageTransfer)
    {
        // 最新のdeviceResource を取得
        IrSensorDeviceResource deviceResource;
        ::std::lock_guard<decltype(*m_pAppletResourceMutex)
        > locker(*m_pAppletResourceMutex);
        m_pAppletResourceManager->GetIrSensorDeviceResource(
            &deviceResource, m_pAppletResourceManager->GetLatestActivatedAruid(), m_Handle);

        // すでに TransferMemory が開放済みの場合はエラーを返す
        NN_RESULT_THROW_UNLESS(deviceResource.imageTransferStatus.workBufferAddress != nullptr, nn::irsensor::ResultIrsensorSamplingIncompleted());

        // データを取得 (AppletResource を操作しているため、ロックしておく必要がある)
        ImageTransferProcessorState state;
        auto result = m_pDriver->GetImageProcessorState(&state, deviceResource.imageTransferStatus.imageBufferAddress);
        if (result.IsSuccess())
        {
            m_pAppletResourceManager->SetImageTransferState(m_Handle, state);
            m_pAppletResourceManager->SetImageValidity(m_pAppletResourceManager->GetLatestActivatedAruid(), m_Handle, true);
        }

        return result;
    }
    if (mode == IrSensorMode::Pointing)
    {
        PointingProcessorMarkerState state[PointingSamplingStatesCountMax];
        int count;
        auto result = m_pDriver->GetImageProcessorStates(&state[0], &count, PointingSamplingStatesCountMax);
        if (result.IsSuccess())
        {
            auto pManager = m_pStatusManagerHolder->GetStatusManager();
            NN_SDK_ASSERT_NOT_NULL(pManager);
            auto pLifo = pManager->GetPointingLifo(m_Handle);
            NN_SDK_ASSERT_NOT_NULL(pLifo);
            for (auto i = 0; i < count; i++)
            {
                pLifo->Append(state[i]);
            }
        }
        return result;
    }
    if (mode == IrSensorMode::TeraPlugin)
    {
        TeraPluginProcessorState state;
        int count;
        auto result = m_pDriver->GetImageProcessorStates(&state, &count, 1);
        if (result.IsSuccess())
        {
            auto pManager = m_pStatusManagerHolder->GetStatusManager();
            NN_SDK_ASSERT_NOT_NULL(pManager);

            auto pLifo = pManager->GetTeraPluginLifo(m_Handle);
            NN_SDK_ASSERT_NOT_NULL(pLifo);
            pLifo->Append(state);

        }
        return result;
    }
    if (mode == IrSensorMode::None)
    {
        NN_RESULT_SUCCESS;
    }
    NN_ABORT(NN_TEXT("予期しない実行モードです"));

} // NOLINT(impl/function_size)

::nn::Result DriverManager::ClearLifo() NN_NOEXCEPT
{
    auto pManager = m_pStatusManagerHolder->GetStatusManager();
    NN_SDK_ASSERT_NOT_NULL(pManager);
    pManager->ClearLifo(m_Handle);

    NN_RESULT_SUCCESS;
}

::nn::Result DriverManager::SetIrSensorConfig(
    const ::nn::applet::AppletResourceUserId& aruid,
    IrSensorConfig config) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pAppletResourceMutex);
    NN_SDK_REQUIRES_NOT_NULL(m_pAppletResourceManager);

    ::std::lock_guard<decltype(*m_pAppletResourceMutex)
    > locker(*m_pAppletResourceMutex);
    m_pAppletResourceManager->SetIrSensorConfig(aruid, m_Handle, config);

    NN_RESULT_SUCCESS;
}

::nn::Result DriverManager::GetIrSensorConfig(
    IrSensorConfig* pConfig) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pAppletResourceMutex);
    NN_SDK_REQUIRES_NOT_NULL(m_pAppletResourceManager);

    ::std::lock_guard<decltype(*m_pAppletResourceMutex)
    > locker(*m_pAppletResourceMutex);

    IrSensorDeviceResource driverResource;
    m_pAppletResourceManager->GetIrSensorDeviceResource(
        &driverResource, m_pAppletResourceManager->GetLatestActivatedAruid(), m_Handle);
    *pConfig = driverResource.config;

    NN_RESULT_SUCCESS;
}

::nn::Result DriverManager::GetTransferMemoryType(
    ::nn::os::TransferMemoryType** pTransferMemory,
    const ::nn::applet::AppletResourceUserId& aruid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pAppletResourceMutex);
    NN_SDK_REQUIRES_NOT_NULL(m_pAppletResourceManager);

    ::std::lock_guard<decltype(*m_pAppletResourceMutex)
    > locker(*m_pAppletResourceMutex);

    m_pAppletResourceManager->GetTransferMemoryType(pTransferMemory, aruid, m_Handle);

    NN_RESULT_SUCCESS;
}

::nn::Result DriverManager::SetTransferMemoryWorkBufferAddress(
    const ::nn::applet::AppletResourceUserId& aruid,
    void* address) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pAppletResourceMutex);
    NN_SDK_REQUIRES_NOT_NULL(m_pAppletResourceManager);

    ::std::lock_guard<decltype(*m_pAppletResourceMutex)
    > locker(*m_pAppletResourceMutex);

    m_pAppletResourceManager->SetTransferMemoryWorkBufferAddress(aruid, address, m_Handle);

    // XCD 側に最新の有効バッファの先頭アドレスを同期的に通知する
    NN_RESULT_DO(m_pDriver->SetLatestTransferMemoryAddress(address));

    NN_RESULT_SUCCESS;
}

::nn::Result DriverManager::GetTransferMemoryWorkBufferAddressWithAruid(
    void** pAddress,
    const ::nn::applet::AppletResourceUserId& aruid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pAppletResourceMutex);
    NN_SDK_REQUIRES_NOT_NULL(m_pAppletResourceManager);

    ::std::lock_guard<decltype(*m_pAppletResourceMutex)
    > locker(*m_pAppletResourceMutex);

    IrSensorDeviceResource driverResource;
    m_pAppletResourceManager->GetIrSensorDeviceResource(
        &driverResource, aruid, m_Handle);
    *pAddress = driverResource.imageTransferStatus.workBufferAddress;

    NN_RESULT_SUCCESS;
}

::nn::Result DriverManager::GetTransferMemoryWorkBufferAddress(
    void** pAddress) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pAppletResourceMutex);
    NN_SDK_REQUIRES_NOT_NULL(m_pAppletResourceManager);

    ::std::lock_guard<decltype(*m_pAppletResourceMutex)
    > locker(*m_pAppletResourceMutex);

    IrSensorDeviceResource driverResource;
    m_pAppletResourceManager->GetIrSensorDeviceResource(
        &driverResource, m_pAppletResourceManager->GetLatestActivatedAruid(), m_Handle);
    *pAddress = driverResource.imageTransferStatus.workBufferAddress;

    NN_RESULT_SUCCESS;
}

::nn::Result DriverManager::SetImageValidity(
    const ::nn::applet::AppletResourceUserId& aruid,
    bool isValid) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(*m_pAppletResourceMutex)
    > locker(*m_pAppletResourceMutex);

    m_pAppletResourceManager->SetImageValidity(aruid, m_Handle, isValid);

    NN_RESULT_SUCCESS;
}

void DriverManager::ActivateTimerEvent() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pTimerEvent);
    NN_SDK_REQUIRES(m_ActivationCount.IsZero());

    ::nn::os::StartPeriodicTimerEvent(
        m_pTimerEvent, SamplingInterval, SamplingInterval);
}

void DriverManager::DeactivateTimerEvent() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pTimerEvent);
    NN_SDK_REQUIRES(m_ActivationCount.IsZero());

    ::nn::os::StopTimerEvent(m_pTimerEvent);
}

::nn::Result DriverManager::ActivateIrsensorForHid() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pAppletResourceManager);
    NN_SDK_REQUIRES_NOT_NULL(m_pDriver);

    // hid に IR センサーの利用を通知
    auto playerNumber = GetIrCameraHandlePlayerNumber(m_Handle);
    auto state = ::nn::hid::system::GetIrSensorState(::nn::hid::system::IrSensorSupportedNpadIds[playerNumber]);

    switch(state)
    {
    case ::nn::hid::system::IrSensorState_Deactivated:
        {
            auto aruid = m_pAppletResourceManager->GetLatestActivatedAruid();
            NN_RESULT_TRY(::nn::hid::system::ActivateIrSensor(
                ::nn::hid::system::IrSensorSupportedNpadIds[playerNumber], aruid))
                NN_RESULT_CATCH(::nn::hid::system::ResultIrSensorActivateFailureNpadBusy)
                {
                    NN_RESULT_THROW(::nn::irsensor::ResultIrSensorActivateFailureNpadBusy());
                }
                NN_RESULT_CATCH(::nn::hid::system::ResultIrSensorActivationLimitOver)
                {
                    NN_RESULT_THROW(::nn::irsensor::ResultIrSensorActivationNpadLimitOver());
                }
                NN_RESULT_CATCH(::nn::hid::system::ResultNoIrSensorDeviceFoundOnNpad)
                {
                    NN_RESULT_THROW(::nn::irsensor::ResultNoIrSensorDeviceFoundOnNpad());
                }
            NN_RESULT_END_TRY
        }
        break;
    case ::nn::hid::system::IrSensorState_Activated:
        // 既に Activate 状態の時は何もしない
        break;
    case ::nn::hid::system::IrSensorState_NotConnected:
        {
            NN_RESULT_THROW(::nn::irsensor::ResultIrsensorUnconnected());
        }
        break;
    case ::nn::hid::system::IrSensorState_NotSupported:
        {
            NN_RESULT_THROW(::nn::irsensor::ResultIrsensorUnsupported());
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    // Activate になったことが確認できた後に、使用するコントローラのデバイスハンドルを通知する
    ::nn::xcd::DeviceHandle handle;
    NN_RESULT_TRY(::nn::hid::system::GetXcdHandleForNpadWithIrSensor(
        &handle, ::nn::hid::system::IrSensorSupportedNpadIds[playerNumber]))
        NN_RESULT_CATCH(::nn::hid::system::ResultNoIrSensorDeviceFoundOnNpad)
        {
            NN_RESULT_THROW(::nn::irsensor::ResultNoIrSensorDeviceFoundOnNpad());
        }
    NN_RESULT_END_TRY
    NN_RESULT_DO(m_pDriver->SetXcdDeviceHandle(handle));

    NN_RESULT_SUCCESS;
}

::nn::Result DriverManager::DeactivateIrsensorForHid() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pAppletResourceManager);

    // hid に IR センサーの使用終了を通知
    auto playerNumber = GetIrCameraHandlePlayerNumber(m_Handle);
    if (::nn::hid::system::GetIrSensorState(::nn::hid::system::IrSensorSupportedNpadIds[playerNumber])
        == ::nn::hid::system::IrSensorState_Activated)
    {
        auto aruid = m_pAppletResourceManager->GetLatestActivatedAruid();
        NN_RESULT_TRY(::nn::hid::system::DeactivateIrSensor(
            ::nn::hid::system::IrSensorSupportedNpadIds[playerNumber], aruid))
            NN_RESULT_CATCH(::nn::hid::system::ResultNoIrSensorDeviceFoundOnNpad)
            {
                NN_RESULT_THROW(::nn::irsensor::ResultNoIrSensorDeviceFoundOnNpad());
            }
        NN_RESULT_END_TRY
    }
    NN_RESULT_SUCCESS;
}

}}} // namespace nn::irsensor::detail
