﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_Thread.h>

#include "hidbus_ActivationCount.h"
#include "hidbus_MultiWaitEvent.h"

namespace nn { namespace hidbus { namespace server {

//!< タスクを扱うクラスです。
class Task
{
    NN_DISALLOW_COPY(Task);
    NN_DISALLOW_MOVE(Task);

private:
    //!< スレッド操作排他用のミューテックス
    ::nn::os::Mutex m_ThreadMutex;

    //!< アクティブ化回数
    ActivationCount m_ActivationCount;

    //!< スレッドオブジェクト
    ::nn::os::ThreadType m_Thread;

    //!< スレッドのスタック領域
    void* m_ThreadStack;

    //!< スレッドのスタック領域のサイズ
    size_t m_ThreadStackSize;

    //!< スレッドの優先度
    int m_ThreadPriority;

    //!< 多重待ちオブジェクト
    ::nn::os::MultiWaitType m_MultiWait;

    //!< 停止イベント
    MultiWaitEvent m_TerminationEvent;

public:
    Task() NN_NOEXCEPT;

    virtual ~Task() NN_NOEXCEPT;

    //!< スレッドのスタック領域を設定します。
    void SetThreadStack(void* stack, size_t stackSize) NN_NOEXCEPT;

    //!< スレッドの優先度を設定します。
    void SetThreadPriority(int priority) NN_NOEXCEPT;

protected:
    //!< スレッド名を登録します。
    virtual void SetThreadName(::nn::os::ThreadType* pThreadType) NN_NOEXCEPT = 0;

    //!< イベントをリンクします。
    virtual void LinkEvent(::nn::os::MultiWaitType* pMultiWait) NN_NOEXCEPT = 0;

    //!< イベントをアンリンクします。
    virtual void UnlinkEvent() NN_NOEXCEPT = 0;

    //!< イベントの処理を実行します。
    virtual void HandleEvent(const ::nn::os::MultiWaitHolderType* waitId
                            ) NN_NOEXCEPT = 0;

    //!< スレッドをアクティブ化します。
    ::nn::Result ActivateThread() NN_NOEXCEPT;

    //!< スレッドを非アクティブ化します。
    ::nn::Result DeactivateThread() NN_NOEXCEPT;

private:
    //!< スレッドの関数エントリ
    static void ThreadFunc(void* value) NN_NOEXCEPT;

    //!< イベントの受付を開始します。
    void InvokeEventLoop() NN_NOEXCEPT;
};

}}} // namespace nn::hidbus::server
