﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/os/os_NativeHandle.h>
#include <nn/os/os_TransferMemory.h>
#include <nn/hidbus/hidbus.h>
#include <nn/hid/hid_ExternalBus.h>

#include "hidbus_Task.h"
#include "hidbus_ResourceHolder.h"
#include "../detail/hidbus_StatusManager.h"
#include "../detail/hidbus_InternalUtility.h"

namespace nn { namespace hidbus { namespace server {

enum HidbusState
{
    HidbusState_NotActivated = 0,
    HidbusState_Activated    = 1,
    HidbusState_Attached     = 2,
    HidbusState_PollingMode  = 3,
};

struct HidbusStatus
{
    HidbusState state;
    nn::hid::ExternalBusHandle handle;
};

//!< 定期実行されるタスクを扱うクラスです。
class SchedulerTask final : public Task
{
private:
    //!< TimerActivate 操作排他用のミューテックス
    ::nn::os::Mutex m_TimerActivateMutex;

    //!< HidbusState 操作用の mutex
    nn::os::SdkMutexType m_SteteMutex[nn::hidbus::detail::MaxHidbusNumber];

    //!< タイマイベント
    MultiWaitTimerEvent m_TimerEvent;

    //!< Hidbus の Status
    HidbusStatus m_Status[nn::hidbus::detail::MaxHidbusNumber];

    //!< Focus 状態の変更があったフラグ
    bool m_IsFocusChanged[nn::hidbus::detail::MaxHidbusNumber];

    //!< Unregister 要求フラグ
    bool m_UnregisterFlag;

    //!< PeriodicTimer の ActivationCount;
    ActivationCount m_PeriodicTimerActivationCount;

    //!< ResourceHolder へのポインタ
    nn::hidbus::server::ResourceHolder* m_pResourceHolder;

public:
    SchedulerTask() NN_NOEXCEPT;

    virtual ~SchedulerTask() NN_NOEXCEPT;

    ::nn::Result Initialize(nn::hidbus::server::ResourceHolder* pStatusManager) NN_NOEXCEPT;

    ::nn::Result Finalize() NN_NOEXCEPT;

    ::nn::Result Activate(BusHandle handle) NN_NOEXCEPT;

    ::nn::Result Deactivate(BusHandle handle) NN_NOEXCEPT;

    void StartPollingMode(nn::hidbus::JoyPollingMode mode, BusHandle handle) NN_NOEXCEPT;

    void StopPollingMode(BusHandle handle) NN_NOEXCEPT;

    void AddFocusChangeTask(int index) NN_NOEXCEPT;

    void UnregisterApplication(nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

private:
    void ActivatePeriodicTimer() NN_NOEXCEPT;

    void DeactivatePeriodicTimer() NN_NOEXCEPT;

    void HandleInvalidHandleResult(int index) NN_NOEXCEPT;

    void DeactivateTransferMemory(BusHandle handle) NN_NOEXCEPT;

    void HandleActivatedState(int index) NN_NOEXCEPT;
    void HandleAttachedState(int index, bool isForceDisable) NN_NOEXCEPT;
    void HandlePollingModeState(int index, bool isForceDisable) NN_NOEXCEPT;
    void ExcuteFocusChangedTask(int index) NN_NOEXCEPT;
    void ExcuteUnregister() NN_NOEXCEPT;

protected:
    //!< スレッド名を登録します。
    virtual void SetThreadName(::nn::os::ThreadType* pThreadType
                                ) NN_NOEXCEPT NN_OVERRIDE;

    //!< イベントをリンクします。
    virtual void LinkEvent(::nn::os::MultiWaitType* pMultiWait
                                ) NN_NOEXCEPT NN_OVERRIDE;

    //!< イベントをアンリンクします。
    virtual void UnlinkEvent() NN_NOEXCEPT NN_OVERRIDE;

    //!< クライアントからの要求の対応とタイマ要求によるサンプリング処理を実行します。
    virtual void HandleEvent(const ::nn::os::MultiWaitHolderType* waitId
                            ) NN_NOEXCEPT NN_OVERRIDE;
};

SchedulerTask& GetSchedulerTask() NN_NOEXCEPT;

}}} // namespace nn::hidbus::server
